package de.duehl.basics.autodetect;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die abstrakte Basis einer Klasse zur automatischen Bearbeitung dar
 *
 * @version 1.01     2022-07-26
 * @author Christian Dühl
 */

public abstract class AutoDetectBase {

    private final static boolean ADD_METHOD_NAMES_TO_AUTO_COMMENT = true;

    /** Beginn von Auto-Kommentaren für automatisch vergebene Bearbeitungen. */
    public static final String AUTO_DETECT_COMMENT = "Maschinell vergeben: ";

    /**
     * Beginn von Auto-Kommentare für vom Benutzer veränderten, zuvor automatisch vergebenen
     * Bearbeitungen.
     */
    public static final String AUTO_DETECT_CHANGED_BY_USER =
            "Automatische Vergabe vom Benutzer geändert.";

    /** Beginn von Auto-Kommentaren bei Fehlern, die eine automatische Bearbeitung verhindern. */
    public static final String AUTO_DETECT_FAILURE_START = "Fehler: ";

    /** Beginn von Auto-Kommentaren bei Fehlern, die eine automatische Bearbeitung verhindern. */
    public static final String AUTO_DETECT_FAILURE_CHANGED_BY_USER_START =
            "Vom Benutzer geändert, vorher Fehler der automatischen Bearbeitung: ";

    /** Beginn von Auto-Kommentaren, wenn nicht alle editierbaren Variablen leer waren. */
    public static final String NOT_AUTO_DETECTED_CONTAINED_DATA =
            "Nicht automatisch bearbeitet, es waren schon Daten eingetragen.";

    /** Beginn von Auto-Kommentaren, wenn nicht alle editierbaren Variablen leer waren. */
    public static final String NOT_AUTO_DETECTED_CONTAINED_DATA_CHANGED_BY_USER =
            "Nicht automatisch bearbeitet, es waren schon Daten eingetragen, die der Benutzer "
            + "geändert hat.";

    /** Hier entstehen die Kommentare zur automatischen Vergabe. */
    private final StringBuilder autoComment;

    /** Konstruktor. */
    protected AutoDetectBase() {
        autoComment = new StringBuilder();
    }

    /** Fügt etwas an den sich aufbauenden Kommentar zur automatischen Bearbeitung an. */
    protected final void appendAutoComment(String appendix) {
        autoComment.append(" ").append(appendix);
    }

    /** Fügt etwas an den sich aufbauenden Kommentar zur automatischen Bearbeitung an. */
    protected final void appendMethodName(String methodName) {
        if (ADD_METHOD_NAMES_TO_AUTO_COMMENT) {
            appendAutoComment(methodName);
        }
    }

    /** Gibt den Inhalt des Auto-Kommentars zurück. */
    public final String getAutoComment() {
        return autoComment.toString();
    }

    /** Legt den Inhalt des Auto-Kommentars fest. */
    public final void setAutoComment(String autoComment) {
        this.autoComment.setLength(0);
        this.autoComment.append(autoComment);
    }

    /** Bereinigt eine Variable. */
    public final static String finishVariable(String input) {
        String output = input;
        output = output.trim();
        output = Text.removeTextAtEndIfEndsWith(output, ",");
        output = output.trim();
        output = Text.removeTextAtEndIfEndsWith(output, "-");
        output = output.trim();
        output = Text.removeTextAtFrontIfStartsWith(output, "-");
        output = output.trim();
        output = Text.stripWhitespace(output);
        return output;
    }

    /** Gibt an, ob die Kommentare zur automatischen Vergabe die übergebene Eingabe enthalten. */
    public final boolean autoCommentContaines(String input) {
        return getAutoComment().contains(input);
    }

}
