package de.duehl.basics.debug;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.List;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.NumberString;

/**
 * Diese Klasse stellt verschiedene Methoden zur Verfügung, um Parameter auf bestimmte
 * Eigenschaften zu überprüfen.
 *
 * @version 1.01     2017-07-12
 * @author Christian Dühl
 */

public class Assure {

    /**
     * Prüft, ob das übergebene Objekt null ist. In dem Fall wird eine Ausnahme geworfen.
     *
     * @param descripton
     *            Beschreibung des zu prüfenden Objekts für die Fehlermeldung.
     * @param object
     *            Zu prüfendes Objekt.
     */
    public static void notNull(String description, Object object) {
        if (null == object) {
            throw new IllegalArgumentException(description + " darf nicht null sein!");
        }
    }

    /**
     * Prüft, ob das übergebene Objekt null ist. In dem Fall wird eine Ausnahme geworfen.
     *
     * @param object
     *            Zu prüfendes Objekt.
     */
    public static void notNull(Object object) {
        notNull("Das übergebene Objekt", object);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(String description, int value, int min, int max) {
        if (value < min || value > max) {
            String message = description + " liegt nicht im zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(int value, int min, int max) {
        isInRange("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(String description, int value, int min) {
        if (value < min) {
            String message = description + " ist zu klein!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(int value, int min) {
        isAtLeast("Der übergebene Wert", value, min);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(String description, int value, int max) {
        if (value > max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(int value, int max) {
        isAtMost("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(String description, long value, long min, long max) {
        if (value < min || value > max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(long value, long min, long max) {
        isInRange("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(String description, long value, long min) {
        if (value < min) {
            String message = description + " ist zu klein!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(long value, long min) {
        isAtLeast("Der übergebene Wert", value, min);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(String description, long value, long max) {
        if (value > max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(long value, long max) {
        isAtMost("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(String description, float value, float min, float max) {
        if (value < min || value > max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(float value, float min, float max) {
        isInRange("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(String description, float value, float min) {
        if (value < min) {
            String message = description + " ist zu klein!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(float value, float min) {
        isAtLeast("Der übergebene Wert", value, min);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(String description, float value, float max) {
        if (value > max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(float value, float max) {
        isAtMost("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(String description, double value, double min, double max) {
        if (value < min || value > max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen und maximalen Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isInRange(double value, double min, double max) {
        isInRange("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(String description, double value, double min) {
        if (value < min) {
            String message = description + " ist zu klein!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert mindestens den minimalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     */
    public static void isAtLeast(double value, double min) {
        isAtLeast("Der übergebene Wert", value, min);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(String description, double value, double max) {
        if (value > max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert inclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (einschließlich).
     */
    public static void isAtMost(double value, double max) {
        isAtMost("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(String description, int value, int min, int max) {
        if (value < min || value >= max) {
            String message = description  + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(int value, int min, int max) {
        isInRangeExclusiveMaximum("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(String description, long value, long min, long max) {
        if (value < min || value >= max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(long value, long min, long max) {
        isInRangeExclusiveMaximum("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(String description, float value, float min, float max) {
        if (value < min || value >= max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(float value, float min, float max) {
        isInRangeExclusiveMaximum("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(String description, double value, double min, double max) {
        if (value < min || value >= max) {
            String message = description + " liegt nicht im Zulässigen Bereich!"
                    + "\n\tWert                    : " + value
                    + "\n\tMinimaler Wert inclusive: " + min
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert innerhalb der minimalen (einschließlich) und maximalen
     * (ausschließlich) Grenzen liegt.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param min
     *            Minimaler erlaubter Wert (einschließlich).
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isInRangeExclusiveMaximum(double value, double min, double max) {
        isInRangeExclusiveMaximum("Der übergebene Wert", value, min, max);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(String description, int value, int max) {
        if (value >= max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(int value, int max) {
        isAtMostExclusive("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(String description, long value, long max) {
        if (value >= max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(long value, long max) {
        isAtMostExclusive("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(String description, float value, float max) {
        if (value >= max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(float value, float max) {
        isAtMostExclusive("Der übergebene Wert", value, max);
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(String description, double value, double max) {
        if (value >= max) {
            String message = description + " ist zu groß!"
                    + "\n\tWert                    : " + value
                    + "\n\tMaximaler Wert exclusive: " + max;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft, ob der angegebene Wert höchstens den maximalen erlaubten Wert hat.
     *
     * @param value
     *            Zu prüfender Wert.
     * @param max
     *            Maximaler erlaubter Wert (ausschließlich).
     */
    public static void isAtMostExclusive(double value, double max) {
        isAtMostExclusive("Der übergebene Wert", value, max);
    }

    /**
     * Überprüft, dass der angegebene Wert nicht leer ist. Null darf er auch nicht sein.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     */
    public static void notEmpty(String description, String value) {
        notNull(description, value);
        if (value.isEmpty()) {
            throw new IllegalArgumentException(description + " darf nicht leer sein!");
        }
    }

    /**
     * Überprüft, dass der angegebene Wert nicht leer ist. Null darf er auch nicht sein.
     *
     * @param value
     *            Zu prüfender Wert.
     */
    public static void notEmpty(String value) {
        notEmpty("Der übergebene String", value);
    }

    /**
     * Überprüft, dass die angegebene Liste nicht leer ist. Null darf sie auch nicht sein.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param list
     *            Zu prüfende Liste.
     */
    public static void notEmpty(String description, List<?> list) {
        notNull(description, list);
        if (list.isEmpty()) {
            throw new IllegalArgumentException(description + " darf nicht leer sein!");
        }
    }

    /**
     * Überprüft, dass die angegebene Liste nicht leer ist. Null darf sie auch nicht sein.
     *
     * @param list
     *            Zu prüfende Liste.
     */
    public static void notEmpty(List<?> list) {
        notEmpty("Die übergebene Liste", list);
    }

    /**
     * Überprüft, dass der angegebene Wert positiv ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isPositive(String description, int value) {
        isAtLeast(description, value, 1);
    }

    /**
     * Überprüft, dass der angegebene Wert positiv ist.
     *
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isPositive(int value) {
        isAtLeast(value, 1);
    }

    /**
     * Überprüft, dass der angegebene Wert negativ ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isNegative(String description, int value) {
        isAtMost(description, value, -1);
    }

    /**
     * Überprüft, dass der angegebene Wert negativ ist.
     *
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isNegative(int value) {
        isAtMost(value, -1);
    }

    /**
     * Überprüft, dass der angegebene Wert positiv ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isPositive(String description, long value) {
        isAtLeast(description, value, 1);
    }

    /**
     * Überprüft, dass der angegebene Wert positiv ist.
     *
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isPositive(long value) {
        isAtLeast(value, 1);
    }

    /**
     * Überprüft, dass der angegebene Wert negativ ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isNegative(String description, long value) {
        isAtMost(description, value, -1);
    }

    /**
     * Überprüft, dass der angegebene Wert negativ ist.
     *
     * @param value
     *            Zu prüfender Wert.
     */
    public static void isNegative(long value) {
        isAtMost(value, -1);
    }

    /**
     * Überprüft, ob eine Liste genau ein Element hat.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param list
     *            Zu überprüfende Liste.
     */
    public static void hasOneElement(String description, List<?> list) {
        if (list.size() != 1) {
            throw new IllegalArgumentException(description + " muss genau ein Element haben!");
        }
    }

    /**
     * Überprüft, ob eine Liste genau ein Element hat.
     *
     * @param list
     *            Zu überprüfende Liste.
     */
    public static void hasOneElement(List<?> list) {
        hasOneElement("Die übergebene Liste", list);
    }

    /**
     * überprüft, ob die angegebene boolsche Variable wahr ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param bool
     *            Zu überprüfende Variable.
     */
    public static void isTrue(String description, boolean bool) {
        if (!bool) {
            throw new IllegalArgumentException(description + " muss wahr sein!");
        }
    }

    /**
     * überprüft, ob die angegebene boolsche Variable wahr ist.
     *
     * @param bool
     *            Zu überprüfende Variable.
     */
    public static void isTrue(boolean bool) {
        isTrue("Die Variable", bool);
    }

    /**
     * überprüft, ob die angegebene boolsche Variable falsch ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param bool
     *            Zu überprüfende Variable.
     */
    public static void isFalse(String description, boolean bool) {
        if (bool) {
            throw new IllegalArgumentException(description + " muss falsch sein!");
        }
    }

    /**
     * überprüft, ob die angegebene boolsche Variable falsch ist.
     *
     * @param bool
     *            Zu überprüfende Variable.
     */
    public static void isFalse(boolean bool) {
        isFalse("Die Variable", bool);
    }

    /**
     * Überprüft, dass das angegebene Verzeichnis auch wirklich ein Verzeichnis ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param dir
     *            Zu überprüfendes Verzeichnis.
     */
    public static void isDirectory(String description, String dir) {
        if (!FileHelper.isDirectory(dir)) {
            throw new IllegalArgumentException(description + " ist kein Verzeichnis!"
                    + "\n\t" + "dir = " + dir);
        }
    }

    /**
     * Überprüft, dass das angegebene Verzeichnis auch wirklich ein Verzeichnis ist.
     *
     * @param dir
     *            Zu überprüfendes Verzeichnis.
     */
    public static void isDirectory(String dir) {
        isDirectory("Das Verzeichnis", dir);
    }

    /**
     * Überprüft, dass die angegebene Datei auch wirklich eine Datei ist.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void isFile(String description, String file) {
        if (!FileHelper.isFile(file)) {
            throw new IllegalArgumentException(description + " ist keine normale Datei!"
                    + "\n\t" + "file = " + file);
        }
    }

    /**
     * Überprüft, dass die angegebene Datei auch wirklich eine Datei ist.
     *
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void isFile(String file) {
        isFile("Die Datei", file);
    }

    /**
     * Überprüft, dass die angegebene Datei existiert.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void exists(String description, String file) {
        if (!FileHelper.exists(file)) {
            throw new IllegalArgumentException(description + " existert nicht!"
                    + "\n\t" + "file = " + file);
        }
    }

    /**
     * Überprüft, dass die angegebene Datei existiert.
     *
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void exists(String file) {
        exists("Die Datei", file);
    }

    /**
     * Überprüft, dass die angegebene Datei existiert.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void notExists(String description, String file) {
        if (FileHelper.exists(file)) {
            throw new IllegalArgumentException(description + " darf nicht existeren!"
                    + "\n\t" + "file = " + file);
        }
    }

    /**
     * Überprüft, dass die angegebene Datei existiert.
     *
     * @param file
     *            Zu überprüfende Datei.
     */
    public static void notExists(String file) {
        notExists("Die Datei", file);
    }

    /**
     * Prüft die Gleichheit der beiden übergebenen Objekte.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param o1
     *            Erstes Objekt.
     * @param o2
     *            Zweite Objekt.
     */
    public static void isEqual(String description, Object o1, Object o2) {
        if (!o1.equals(o2)) {
            String message = description + " dürfen nicht unterschiedlich sein!"
                    + "\n\t" + "o1 = " + o1
                    + "\n\t" + "o2 = " + o2;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft die Gleichheit der beiden übergebenen Objekte.
     *
     * @param o1
     *            Erstes Objekt.
     * @param o2
     *            Zweite Objekt.
     */
    public static void isEqual(Object o1, Object o2) {
        isEqual("Die übergebenen Objekte", o1, o2);
    }

    /**
     * Prüft die Verschiedenheit der beiden übergebenen Objekte.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param o1
     *            Erstes Objekt.
     * @param o2
     *            Zweite Objekt.
     */
    public static void isNotEqual(String description, Object o1, Object o2) {
        if (o1.equals(o2)) {
            String message = description + " dürfen nicht gleich sein!"
                    + "\n\t" + "o1 = " + o1
                    + "\n\t" + "o2 = " + o2;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Prüft die Verschiedenheit der beiden übergebenen Objekte.
     *
     * @param o1
     *            Erstes Objekt.
     * @param o2
     *            Zweite Objekt.
     */
    public static void isNotEqual(Object o1, Object o2) {
        isNotEqual("Die übergebenen Objekte", o1, o2);
    }

    /**
     * Überprüft, dass die Liste das Element nicht enthält.
     *
     * @param list
     *            Liste, die das Element nicht enthalten soll.
     * @param element
     *            Fragliches Element.
     */
    public static void notContains(String description, List<String> list, String element) {
        if (list.contains(element)) {
            String message = description + " darf das Element nicht enthalten!\n\t"
                    + "Doppeltes Element: " + element;
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Überprüft, dass die Liste das Element nicht enthält.
     *
     * @param list
     *            Liste, die das Element nicht enthalten soll.
     * @param element
     *            Fragliches Element.
     */
    public static void notContains(List<String> list, String element) {
        notContains("Die Liste", list, element);
    }

    /**
     * Überprüft, dass der übergebene String aus einer Folge von Ziffern besteht.
     *
     * @param description
     *            Beschreibung des zu prüfenden Wertes für die Fehlermeldung.
     * @param digits
     *            Zu überprüfender Wert.
     */
    public static void isDigitSequence(String description, String digits) {
        if (!NumberString.isDigitSequence(digits)) {
            String message = description + " besteht nicht ausschließlich aus Ziffern!\n\t"
                    + "digits : [" + digits + "]";
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Überprüft, dass der übergebene String aus einer Folge von Ziffern besteht.
     *
     * @param digits
     *            Zu überprüfender Wert.
     */
    public static void isDigitSequence(String digits) {
        isDigitSequence("Der angegebene Wert", digits);
    }

}
