package de.duehl.basics.io.done;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.FileHelper;

/**
 * Diese Klasse bietet Methoden rund um das Anlegen, Abfragen und Entfernen von Dateien als Markern
 * dafür, das etwas fertig gestellt wurde.
 *
 * @version 1.01     2022-07-05
 * @author Christian Dühl
 */

public class Done {

    /**
     * Die Erweiterung die an die Datei angehängt wird, zu der ein Marker zur Fertigstellung
     * erzeugt werden soll.
     */
    private static final String DONE_EXTENSION = ".done";

    /** Name der Datei, zu der ein Marker zur Fertigstellung erzeugt werden soll. */
    private final String fileToCreateDoneFileFor;

    /** Name der Datei der Fertig-Markierung. */
    private final String doneFilename;

    /**
     * Konstruktor zur Erzeugung des Fertig-Markers im gleichen Verzeichnis wie die übergebene
     * Datei.
     *
     * @param fileToCreateDoneFileFor
     *            Name der Datei, zu der ein Marker zur Fertigstellung erzeugt werden soll.
     */
    public Done(String fileToCreateDoneFileFor) {
        this.fileToCreateDoneFileFor = fileToCreateDoneFileFor;
        doneFilename = fileToCreateDoneFileFor + DONE_EXTENSION;
    }

    /**
     * Konstruktor zur Erzeugung des Fertig-Markers in einem anderen Verzeichnis als die übergebene
     * Datei.
     *
     * @param fileToCreateDoneFileFor
     *            Name der Datei, zu der ein Marker zur Fertigstellung erzeugt werden soll.
     * @param differentDirectory
     *            Verzeichnis, in der ein Marker zur Fertigstellung erzeugt werden soll.
     */
    public Done(String fileToCreateDoneFileFor, String differentDirectory) {
        this.fileToCreateDoneFileFor = fileToCreateDoneFileFor;
        String bareDoneFilename = FileHelper.getBareName(fileToCreateDoneFileFor) + DONE_EXTENSION;
        doneFilename = FileHelper.concatPathes(differentDirectory, bareDoneFilename);
    }

    /** Getter für den Namen der Datei, zu der ein Marker zur Fertigstellung erzeugt werden soll. */
    public String getFileToCreateDoneFileFor() {
        return fileToCreateDoneFileFor;
    }

    /** Getter für den Namen der Datei der Fertig-Markierung. */
    public String getDoneFilename() {
        return doneFilename;
    }

    /** Gibt an, ob die Fertig-Marker-Datei existiert. */
    public boolean isDoneMarkerSet() {
        return FileHelper.exists(doneFilename);
    }

    /** Gibt an, ob die Fertig-Marker-Datei nicht existiert. */
    public boolean isDoneMarkerMissing() {
        return !isDoneMarkerSet();
    }

    /** Legt die Fertig-Marker-Datei an. */
    public void createDoneFile() {
        FileHelper.createEmptyFile(doneFilename);
    }

    /** Löscht die Fertig-Marker-Datei im Dateisystem. */
    public void deleteDoneFile() {
        FileHelper.deleteFileIfExistent(doneFilename);
    }

}
