package de.duehl.basics.io.textfile;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;

/**
 * Diese Klasse liest eine Liste mit den Texten aus den Zeilen einer Textdatei ein.
 *
 * Die Zeilen werden getrimmt. Dabei werden leere Zeilen und solche, die mit dem Kommentarzeichen
 * (ggf., nach Leerraum) überlesen. Kommt in einer Datenzeile weiter hinten das Kommentarzeichen
 * vor, so wird ab dort abgeschnitten und getrimmt.
 *
 * Der Zeileninhalt wird dann als Komma-getrennte Liste verarbeitet.
 *
 * @version 1.01     2019-07-04
 * @author Christian Dühl
 */

public class ListsFromTextFileReader {

    private final StringsFromTextFileReader reader;

    public ListsFromTextFileReader(StringsFromTextFileReader reader) {
        this.reader = reader;
    }

    /** Legt fest, dass der Reader ruhig sein soll. */
    public void beQuiet() {
        reader.beQuiet();
    }

    /** Legt fest, dass Leerzeilen mit in das Ergebnis sollen. */
    public void includeBlankLinesIntoResult() {
        reader.includeBlankLinesIntoResult();
    }

    /**
     * Gibt an, dass die eingelesenen Zeilen in Kleinbuchstaben gewandelt werden.
     *
     * Hierbei wird Locale.GERMAN eingesetzt.
     */
    public void changeToLowerCase() {
        reader.changeToLowerCase();
    }

    /**
     * Gibt an, dass die eingelesenen Zeilen vorn und hinten nicht von Leerzeichen befreit werden.
     *
     * Kommentare werden hinter den Werten in jedem Fall entfernt.
     */
    public void doNotTrimLines() {
        reader.doNotTrimLines();
    }

    /** Liest die Listen aus der Datei ein. */
    public List<List<String>> read() {
        List<String> lines = reader.read();
        return splitLines(lines);
    }

    private List<List<String>> splitLines(List<String> lines) {
        List<List<String>> listOfLists = new ArrayList<>();

        for (String line : lines) {
            splitLine(listOfLists, line);
        }

        return listOfLists;
    }

    private void splitLine(List<List<String>> listOfLists, String line) {
        List<String> list = Text.splitByKomma(line);
        listOfLists.add(list);
    }

}
