package de.duehl.basics.text.html;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

/**
 * Diese Klasse stellt Basisfunktionalitäten für Klassen zur Verfügung, welche Texte für die
 * Anzeige in einem HTML-Editor einfärben.
 *
 * @version 1.01     2022-08-05
 * @author Christian Dühl
 */

public class Text2HtmlBasics {

    private static final String SPAN_START = "<span[^<>]*>";
    private static final String SPAN_END = "</span>";
    private static final String NO_SPAN = "(?:(?!</?span).)*";

    private static final String FONT_START1 = "<span style=\"font-size:";
    private static final String FONT_START2 = "pt\">";
    private static final String FONT_OFF = SPAN_END;

    private static final String HTML_START = "<html><head></head><body>";
    private static final String HTML_END = "</body></html>";
    private static final String HTML_STAR_TWITH_MONOSPACE_FONT_CSS = "<html>"
            + "<head>"
            + "<style>body {font-family: monospace;}</style>"
            + "</head><body>";


    private static final String BACKGROUND_COLOR_START = "<span style=\"background-color:";
    private static final String BACKGROUND_COLOR_END = "\">";
    private static final String BACKGROUND_COLOR = SPAN_END;

    /**
     * Entfernt ineinander verschachtelte <span ...> ... </span> Ausdrücke.
     *
     * @param input
     *            Zu bereinigender Text.
     * @return Bereinigter Text.
     */
    public static String removeNestedSpans(String input) {
        String output = input;

        String oldOutput = "";
        while (!output.equals(oldOutput)) {
            oldOutput = output;
            //output = output.replaceAll("(<span .*?>.*?)<span .*?>(.*?)</span>(.*?</span>)",
            //        "$1$2$3");
            output = output.replaceAll("(?s)(" + SPAN_START + NO_SPAN + ")"
                    + SPAN_START + "(" + NO_SPAN + ")" + SPAN_END
                    + "(" + NO_SPAN + SPAN_END + ")",
                    "$1$2$3");
        }

        return output;
    }

    /**
     * Hilfsmethode: Der Text wird in einer bestimmten Größe dargestellt. Dafür wird er mit einem
     * HTML-Befehl der Art
     *     <span style="font-size:12pt"> ... </span>
     * umgeben.
     *
     * @param text
     *            Eingabetext
     * @param fontSize
     *            Größe des Textes, die Angabe erfolgt in Punkt (pt).
     * @return Modifizierter Ausgabetext
     */
    public static String surroundTextWithFontSize(String text, int fontSize) {
        return FONT_START1 + fontSize + FONT_START2 + text + FONT_OFF;
    }

    /**
     * Hilfsmethode: Der Text wird in einer bestimmten Größe dargestellt. Dafür wird er mit einem
     * HTML-Befehl der Art
     *     <span style="font-size:12pt"> ... </span>
     * umgeben.
     *
     * @param text
     *            Eingabetext
     * @param backgroundColor
     *            Hintergrundfarbe des Textes, die Angabe erfolgt als HTML-Farbe.
     * @return Modifizierter Ausgabetext
     */
    public static String surroundTextWithBackgroundColor(String text, String backgroundHexColor) {
        return BACKGROUND_COLOR_START + backgroundHexColor + BACKGROUND_COLOR_END + text
                + BACKGROUND_COLOR;
    }

    /**
     * Hilfsmethode: Umgibt den Eingabetext mit Anfang und Ende einer HTML-Seite.
     *
     * @param input
     *            Eingabetext
     * @return Modifizierter Ausgabetext
     */
    public static String surroundTextWithHtml(String input) {
        String output = input;

        output = HTML_START + output + HTML_END;

        return output;
    }

    /**
     * Hilfsmethode: Umgibt den Eingabetext mit Anfang und Ende einer HTML-Seite.
     *
     * @param input
     *            Eingabetext
     * @return Modifizierter Ausgabetext
     */
    public static String surroundTextWithHtmlWithMonospacedFont(String input) {
        String output = input;

        output = HTML_STAR_TWITH_MONOSPACE_FONT_CSS + output + HTML_END;

        return output;
    }

}
