package de.duehl.basics.collections;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.Test;

import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.system.OutputStreamSaveAndRelease;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;

public class CollectionsHelperTest {

    @Test
    public void getSortedIndicesForEmptyKeySet() {
        Set<Integer> keySet = new HashSet<>();
        List<Integer> list = CollectionsHelper.getSortedIndices(keySet);
        assertTrue(list.isEmpty());
    }

    @Test
    public void getSortedIndicesForOneNumber() {
        Set<Integer> keySet = new HashSet<>();
        Integer zero = Integer.valueOf(0);
        keySet.add(zero);
        List<Integer> list = CollectionsHelper.getSortedIndices(keySet);
        assertEquals(1, list.size());
        assertEquals(zero, list.get(0));
    }

    @Test
    public void getSortedIndicesForTwoNumberInsertedCorrectlySorted() {
        Set<Integer> keySet = new HashSet<>();
        Integer zero = Integer.valueOf(0);
        Integer one = Integer.valueOf(1);
        keySet.add(zero);
        keySet.add(one);
        List<Integer> list = CollectionsHelper.getSortedIndices(keySet);
        assertEquals(2, list.size());
        assertEquals(zero, list.get(0));
        assertEquals(one, list.get(1));
    }

    @Test
    public void getSortedIndicesForTwoNumberInsertedIncorrectlySorted() {
        Set<Integer> keySet = new HashSet<>();
        Integer zero = Integer.valueOf(0);
        Integer one = Integer.valueOf(1);
        keySet.add(one);
        keySet.add(zero);
        List<Integer> list = CollectionsHelper.getSortedIndices(keySet);
        assertEquals(2, list.size());
        assertEquals(zero, list.get(0));
        assertEquals(one, list.get(1));
    }

    @Test
    public void getSortedIndicesForFiveNumber() {
        Integer zero = Integer.valueOf(0);
        Integer one = Integer.valueOf(1);
        Integer two = Integer.valueOf(2);
        Integer three = Integer.valueOf(3);
        Integer four = Integer.valueOf(4);

        Set<Integer> keySet = new HashSet<>();
        keySet.add(one);
        keySet.add(four);
        keySet.add(zero);
        keySet.add(two);
        keySet.add(three);

        List<Integer> list = CollectionsHelper.getSortedIndices(keySet);

        assertEquals(5, list.size());
        assertEquals(zero, list.get(0));
        assertEquals(one, list.get(1));
        assertEquals(two, list.get(2));
        assertEquals(three, list.get(3));
        assertEquals(four, list.get(4));
    }

    @Test
    public void getSortedMapIndicesForFiveNumber() {
        Integer zero = Integer.valueOf(0);
        Integer one = Integer.valueOf(1);
        Integer two = Integer.valueOf(2);
        Integer three = Integer.valueOf(3);
        Integer four = Integer.valueOf(4);

        Map<Integer, String> map = new HashMap<>();
        map.put(three, "three");
        map.put(four, "four");
        map.put(one, "one");
        map.put(zero, "zero");
        map.put(two, "two");

        List<Integer> list = CollectionsHelper.getSortedMapIndices(map);

        assertEquals(5, list.size());
        assertEquals(zero, list.get(0));
        assertEquals(one, list.get(1));
        assertEquals(two, list.get(2));
        assertEquals(three, list.get(3));
        assertEquals(four, list.get(4));
    }

    @Test
    public void getSortedStringIndicesForEmptyKeySet() {
        Set<String> keySet = new HashSet<>();
        List<String> list = CollectionsHelper.getSortedStringIndices(keySet);
        assertTrue(list.isEmpty());
    }

    @Test
    public void getSortedStringIndicesForOneNumber() {
        Set<String> keySet = new HashSet<>();
        String zero = "zero";
        keySet.add(zero);
        List<String> list = CollectionsHelper.getSortedStringIndices(keySet);
        assertEquals(1, list.size());
        assertEquals(zero, list.get(0));
    }

    @Test
    public void getSortedStringIndicesForTwoNumberInsertedCorrectlySorted() {
        Set<String> keySet = new HashSet<>();
        String zero = "zero";
        String one = "one";
        keySet.add(one);
        keySet.add(zero);
        List<String> list = CollectionsHelper.getSortedStringIndices(keySet);
        assertEquals(2, list.size());
        assertEquals(zero, list.get(1));
        assertEquals(one, list.get(0));
    }

    @Test
    public void getSortedStringIndicesForTwoNumberInsertedIncorrectlySorted() {
        Set<String> keySet = new HashSet<>();
        String zero = "zero";
        String one = "one";
        keySet.add(zero);
        keySet.add(one);
        List<String> list = CollectionsHelper.getSortedStringIndices(keySet);
        assertEquals(2, list.size());
        assertEquals(zero, list.get(1));
        assertEquals(one, list.get(0));
    }

    @Test
    public void getSortedStringIndicesForFiveStrings() {
        String zero = "zero";
        String one = "one";
        String two = "two";
        String three = "three";
        String four = "four";

        Set<String> keySet = new HashSet<>();
        keySet.add(one);
        keySet.add(four);
        keySet.add(zero);
        keySet.add(two);
        keySet.add(three);

        List<String> list = CollectionsHelper.getSortedStringIndices(keySet);

        assertEquals(5, list.size());
        assertEquals(zero, list.get(4));
        assertEquals(one, list.get(1));
        assertEquals(two, list.get(3));
        assertEquals(three, list.get(2));
        assertEquals(four, list.get(0));
    }

    @Test
    public void getSortedMapStringIndicesForFiveNumber() {
        String zero = "zero";
        String one = "one";
        String two = "two";
        String three = "three";
        String four = "four";

        Map<String, String> map = new HashMap<>();
        map.put(three, "drei");
        map.put(four, "vier");
        map.put(one, "eins");
        map.put(zero, "null");
        map.put(two, "zwei");

        List<String> list = CollectionsHelper.getSortedMapStringIndices(map);

        assertEquals(5, list.size());
        assertEquals(zero, list.get(4));
        assertEquals(one, list.get(1));
        assertEquals(two, list.get(3));
        assertEquals(three, list.get(2));
        assertEquals(four, list.get(0));
    }

    @Test
    public void reverseList() {
        List<String> list = new ArrayList<>();
        list.add("zero");
        list.add("one");
        list.add("two");
        list.add("three");
        list.add("four");

        assertEquals(5, list.size());
        assertEquals("zero", list.get(0));
        assertEquals("one", list.get(1));
        assertEquals("two", list.get(2));
        assertEquals("three", list.get(3));
        assertEquals("four", list.get(4));

        List<String> reverse = CollectionsHelper.reverseList(list);

        assertEquals(5, list.size());
        assertEquals("zero", list.get(0));
        assertEquals("one", list.get(1));
        assertEquals("two", list.get(2));
        assertEquals("three", list.get(3));
        assertEquals("four", list.get(4));

        assertEquals(5, reverse.size());
        assertEquals("four", reverse.get(0));
        assertEquals("three", reverse.get(1));
        assertEquals("two", reverse.get(2));
        assertEquals("one", reverse.get(3));
        assertEquals("zero", reverse.get(4));
    }

    @Test
    public void listListNice() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListNice(list);
        String expected = createExpectedOutputWithStartindex1();
        assertEquals(expected, actual);
    }

    @Test
    public void listListNice1() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListNice(list, 1);
        String expected = createExpectedOutputWithStartindex1();
        assertEquals(expected, actual);
    }

    @Test
    public void listListNice0() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListNice(list, 0);
        String expected = createExpectedOutputWithStartindex0();
        assertEquals(expected, actual);
    }

    @Test
    public void printListOfStrings() {
        List<String> list = createStringlist();
        OutputStreamSaveAndRelease ossar = new OutputStreamSaveAndRelease();
        ossar.saveSystemOut();
        CollectionsHelper.printListNice(list);
        String actual = ossar.getOutContents();
        ossar.releaseSystemOut();
        String expected = createExpectedOutputWithStartindex1();
        assertEquals(expected, actual);
    }

    @Test
    public void listListNiceWithoutBrackets() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListNiceWithoutBrackets(list);
        String expected = createExpectedOutput2();
        assertEquals(expected, actual);
    }

    @Test
    public void listListIndentedWithoutNumberAndBracket() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListIndentedWithoutNumberAndBracket(list);
        String expected = createExpectedOutput3();
        assertEquals(expected, actual);
    }

    @Test
    public void listListWithoutNumberAndBracket() {
        List<String> list = createStringlist();
        String actual = CollectionsHelper.listListWithoutNumberAndBracket(list);
        String expected = createExpectedOutput4();
        assertEquals(expected, actual);
    }

    private List<String> createStringlist() {
        List<String> list = new ArrayList<>();

        list.add("zero");
        list.add("one");
        list.add("two");
        list.add("three");
        list.add("four");

        return list;
    }

    private String createExpectedOutputWithStartindex1() {
        return ""
                + "     1. [zero]" + Text.LINE_BREAK
                + "     2. [one]" + Text.LINE_BREAK
                + "     3. [two]" + Text.LINE_BREAK
                + "     4. [three]" + Text.LINE_BREAK
                + "     5. [four]" + Text.LINE_BREAK
                ;
    }

    private String createExpectedOutputWithStartindex0() {
        return ""
                + "     0. [zero]" + Text.LINE_BREAK
                + "     1. [one]" + Text.LINE_BREAK
                + "     2. [two]" + Text.LINE_BREAK
                + "     3. [three]" + Text.LINE_BREAK
                + "     4. [four]" + Text.LINE_BREAK
                ;
    }

    private String createExpectedOutput2() {
        return ""
                + "     1. zero" + Text.LINE_BREAK
                + "     2. one" + Text.LINE_BREAK
                + "     3. two" + Text.LINE_BREAK
                + "     4. three" + Text.LINE_BREAK
                + "     5. four" + Text.LINE_BREAK
                ;
    }

    private String createExpectedOutput3() {
        return ""
                + "    zero" + Text.LINE_BREAK
                + "    one" + Text.LINE_BREAK
                + "    two" + Text.LINE_BREAK
                + "    three" + Text.LINE_BREAK
                + "    four" + Text.LINE_BREAK
                ;
    }

    private String createExpectedOutput4() {
        return ""
                + "zero" + Text.LINE_BREAK
                + "one" + Text.LINE_BREAK
                + "two" + Text.LINE_BREAK
                + "three" + Text.LINE_BREAK
                + "four" + Text.LINE_BREAK
                ;
    }

    @Test
    public void buildListFrom() {
        List<String> texts = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        assertEquals(3, texts.size());
        assertEquals("foo", texts.get(0));
        assertEquals("bar", texts.get(1));
        assertEquals("baz", texts.get(2));
    }

    @Test
    public void buildListFromEmptyList() {
        List<String> texts = CollectionsHelper.buildListFrom();
        assertEquals(0, texts.size());
    }

    @Test
    public void buildListFromOneElement() {
        List<String> texts = CollectionsHelper.buildListFrom("foo");
        assertEquals(1, texts.size());
        assertEquals("foo", texts.get(0));
    }

    @Test
    public void buildIntListFromWithoutInList() {
        List<Integer> values = CollectionsHelper.buildListFrom(1, 2, 3);
        assertEquals(3, values.size());
        assertEquals(Integer.valueOf(1), values.get(0));
        assertEquals(Integer.valueOf(2), values.get(1));
        assertEquals(Integer.valueOf(3), values.get(2));
    }

    @Test
    public void buildListFromWithList() {
        List<String> texts = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        assertEquals(3, texts.size());
        assertEquals("foo", texts.get(0));
        assertEquals("bar", texts.get(1));
        assertEquals("baz", texts.get(2));
        List<String> texts2 = CollectionsHelper.buildListFrom(texts, "Apfel", "Orange");
        assertEquals(5, texts2.size());
        assertEquals("foo", texts2.get(0));
        assertEquals("bar", texts2.get(1));
        assertEquals("baz", texts2.get(2));
        assertEquals("Apfel", texts2.get(3));
        assertEquals("Orange", texts2.get(4));
    }

//    @SuppressWarnings("deprecation")
//    @Test
//    public void buildIntListFrom() {
//        List<Integer> values = CollectionsHelper.buildIntListFrom(1, 2, 3);
//        assertEquals(3, values.size());
//        assertEquals(Integer.valueOf(1), values.get(0));
//        assertEquals(Integer.valueOf(2), values.get(1));
//        assertEquals(Integer.valueOf(3), values.get(2));
//    }
//
//    @SuppressWarnings("deprecation")
//    @Test
//    public void buildIntListFromEmptyList() {
//        List<Integer> values = CollectionsHelper.buildIntListFrom();
//        assertEquals(0, values.size());
//    }
//
//    @SuppressWarnings("deprecation")
//    @Test
//    public void buildIntListFromOneElement() {
//        List<Integer> values = CollectionsHelper.buildIntListFrom(1);
//        assertEquals(1, values.size());
//        assertEquals(Integer.valueOf(1), values.get(0));
//    }

    @Test
    public void buildIntListFrom() {
        List<Integer> values = CollectionsHelper.buildListFrom(1, 2, 3);
        assertEquals(3, values.size());
        assertEquals(Integer.valueOf(1), values.get(0));
        assertEquals(Integer.valueOf(2), values.get(1));
        assertEquals(Integer.valueOf(3), values.get(2));
    }

    @Test
    public void buildIntListFromEmptyList() {
        List<Integer> values = CollectionsHelper.buildListFrom();
        assertEquals(0, values.size());
    }

    @Test
    public void buildIntListFromOneElement() {
        List<Integer> values = CollectionsHelper.buildListFrom(1);
        assertEquals(1, values.size());
        assertEquals(Integer.valueOf(1), values.get(0));
    }

    @Test
    public void buildLoweredListFrom() {
        List<String> texts = CollectionsHelper.buildLoweredListFrom("FOO", "bar", "Baz");
        assertEquals(3, texts.size());
        assertEquals("foo", texts.get(0));
        assertEquals("bar", texts.get(1));
        assertEquals("baz", texts.get(2));
    }

    @Test
    public void buildLoweredListFromEmptyList() {
        List<String> texts = CollectionsHelper.buildLoweredListFrom();
        assertEquals(0, texts.size());
    }

    @Test
    public void buildLoweredListFromOneElement() {
        List<String> texts = CollectionsHelper.buildLoweredListFrom("fOo");
        assertEquals(1, texts.size());
        assertEquals("foo", texts.get(0));
    }

    @Test
    public void buildUpperedListFrom() {
        List<String> texts = CollectionsHelper.buildUpperedListFrom("FOO", "bar", "Baz");
        assertEquals(3, texts.size());
        assertEquals("FOO", texts.get(0));
        assertEquals("BAR", texts.get(1));
        assertEquals("BAZ", texts.get(2));
    }

    @Test
    public void buildSortByLengthDescandingListFrom() {
        List<String> texts = CollectionsHelper.buildSortByLengthDescandingListFrom("bb", "a", "ccc");
        assertEquals(3, texts.size());
        assertEquals("ccc", texts.get(0));
        assertEquals("bb", texts.get(1));
        assertEquals("a", texts.get(2));
    }

    @Test
    public void buildSortByLengthDescandingListAsListFrom() {
        List<String> list = CollectionsHelper.buildListFrom("bb", "a", "ccc");
        List<String> texts = CollectionsHelper.buildSortByLengthDescandingListFrom(list);
        assertEquals(3, texts.size());
        assertEquals("ccc", texts.get(0));
        assertEquals("bb", texts.get(1));
        assertEquals("a", texts.get(2));
    }

    @Test
    public void buildSortByLengthListFrom() {
        List<String> texts = CollectionsHelper.buildSortByLengthListFrom("bb", "a", "ccc");
        assertEquals(3, texts.size());
        assertEquals("a", texts.get(0));
        assertEquals("bb", texts.get(1));
        assertEquals("ccc", texts.get(2));
    }

    @Test
    public void buildSortByLengthListAsListFrom() {
        List<String> list = CollectionsHelper.buildListFrom("bb", "a", "ccc");
        List<String> texts = CollectionsHelper.buildSortByLengthListFrom(list);
        assertEquals(3, texts.size());
        assertEquals("a", texts.get(0));
        assertEquals("bb", texts.get(1));
        assertEquals("ccc", texts.get(2));
    }

    @Test
    public void buildSortedListFrom() {
        List<String> texts = CollectionsHelper.buildSortedListFrom("foo", "bar", "baz");
        assertEquals(3, texts.size());
        assertEquals("bar", texts.get(0));
        assertEquals("baz", texts.get(1));
        assertEquals("foo", texts.get(2));
    }

    @Test
    public void buildDescandingSortedListFrom() {
        List<String> texts = CollectionsHelper.buildDescandingSortedListFrom("foo", "bar", "baz");
        assertEquals(3, texts.size());
        assertEquals("foo", texts.get(0));
        assertEquals("baz", texts.get(1));
        assertEquals("bar", texts.get(2));
    }

    @Test
    public void removeEmptyStringsFromListWithoutEmptyElements() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(3, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
        assertEquals("c", list.get(2));
    }

    @Test
    public void removeEmptyStringsFromListFromEmptyList() {
        List<String> list = new ArrayList<>();
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsFromListWithEmptyElementAtFront() {
        List<String> list = CollectionsHelper.buildListFrom("", "b", "c");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithEmptyElementAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "", "c");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithEmptyElementAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithMultipleEmptyElementAtFront() {
        List<String> list = CollectionsHelper.buildListFrom("", "", "", "", "", "", "b", "c");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithMultipleEmptyElementAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "", "", "", "c");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithMultipleEmptyElementAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "", "", "");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeEmptyStringsFromListWithOnlyEmptyElements() {
        List<String> list = CollectionsHelper.buildListFrom("", "", "", "", "");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsFromListWithOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("");
        CollectionsHelper.removeEmptyStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListFromEmptyList() {
        List<String> list = new ArrayList<>();
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithEmptyElementAtFront() {
        List<String> list = CollectionsHelper.buildListFrom("   ", "b", "c");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithEmptyElementAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "    ", "c");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithEmptyElementAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "  \t   ");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithMultipleEmptyElementAtFront() {
        List<String> list = CollectionsHelper.buildListFrom(" ", "  ", "", "", "", "  ", "b", "c");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithMultipleEmptyElementAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "  ", "  ", "  ", "c");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithMultipleEmptyElementAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", " ", "  ", "\t     ");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithOnlyEmptyElements() {
        List<String> list = CollectionsHelper.buildListFrom("    ", "", "\t", "", "       ");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyAndOnlyWhitespaceStringsFromListWithOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("      ");
        CollectionsHelper.removeEmptyAndOnlyWhitespaceStringsFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeCommentLinesFromListFromEmptyList() {
        List<String> list = new ArrayList<>();
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeCommentLinesFromListWithCommentAtFront() {
        List<String> list = CollectionsHelper.buildListFrom("   # a", "b", "c");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithCommentAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "# b und so    ", "c");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithCommentAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "  \t   #");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithMultipleCommentsAtFront() {
        List<String> list = CollectionsHelper.buildListFrom("# ", "#  ", "#", "#", "#", "#    ",
                "b", "c");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithMultipleCommentsAtMiddle() {
        List<String> list = CollectionsHelper.buildListFrom("a", "#  ", "#  ", "  #", "c");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("c", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithMultipleCommentsAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", " #", "#  ", "\t     #");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(2, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
    }

    @Test
    public void removeCommentLinesFromListWithOnlyComments() {
        List<String> list = CollectionsHelper.buildListFrom("    #", "#", "\t#", "#", "#       ");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeCommentLinesFromListWithOneComment() {
        List<String> list = CollectionsHelper.buildListFrom("     # ");
        CollectionsHelper.removeCommentLinesFromList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeMultipleValues() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "a", "c", "a");
        List<String> singles = CollectionsHelper.removeMultipleValues(list);
        assertEquals(3,  singles.size());
        assertEquals("a", singles.get(0));
        assertEquals("b", singles.get(1));
        assertEquals("c", singles.get(2));
    }

    @Test
    public void removeMultipleValuesEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        List<String> singles = CollectionsHelper.removeMultipleValues(list);
        assertEquals(0,  singles.size());
    }

    @Test
    public void removeMultipleValuesOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        List<String> singles = CollectionsHelper.removeMultipleValues(list);
        assertEquals(1,  singles.size());
        assertEquals("a", singles.get(0));
    }

    @Test
    public void removeMultipleValuesOneMultipleElement() {
        List<String> list = CollectionsHelper.buildListFrom("a", "a", "a", "a", "a", "a", "a");
        List<String> singles = CollectionsHelper.removeMultipleValues(list);
        assertEquals(1,  singles.size());
        assertEquals("a", singles.get(0));
    }

    @Test
    public void removeLastListEntry() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e");
        CollectionsHelper.removeLastListEntry(list);
        assertEquals(4,  list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
        assertEquals("c", list.get(2));
        assertEquals("d", list.get(3));
    }

    @Test
    public void removeLastListEntryEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeLastListEntry(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void removeLastListEntryOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        CollectionsHelper.removeLastListEntry(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void removeLastListEntryFromNotEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e");
        CollectionsHelper.removeLastListEntryFromNotEmptyList(list);
        assertEquals(4,  list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
        assertEquals("c", list.get(2));
        assertEquals("d", list.get(3));
    }

    @Test
    public void removeLastListEntryFromNotEmptyListEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        assertThrows(IllegalArgumentException.class, () ->
            CollectionsHelper.removeLastListEntryFromNotEmptyList(list));
    }

    @Test
    public void removeLastListEntryFromNotEmptyListOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        CollectionsHelper.removeLastListEntryFromNotEmptyList(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void removeFirstListEntry() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e");
        CollectionsHelper.removeFirstListEntry(list);
        assertEquals(4,  list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
        assertEquals("d", list.get(2));
        assertEquals("e", list.get(3));
    }

    @Test
    public void removeFirstListEntryEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeFirstListEntry(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void removeFirstListEntryOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        CollectionsHelper.removeFirstListEntry(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void removeFirstListEntryFromNotEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e");
        CollectionsHelper.removeFirstListEntryFromNotEmptyList(list);
        assertEquals(4,  list.size());
        assertEquals("b", list.get(0));
        assertEquals("c", list.get(1));
        assertEquals("d", list.get(2));
        assertEquals("e", list.get(3));
    }

    @Test
    public void removeFirstListEntryFromNotEmptyListEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        assertThrows(IllegalArgumentException.class, () ->
            CollectionsHelper.removeFirstListEntryFromNotEmptyList(list));
    }

    @Test
    public void removeFirstListEntryFromNotEmptyListOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        CollectionsHelper.removeFirstListEntryFromNotEmptyList(list);
        assertEquals(0,  list.size());
    }

    @Test
    public void stringArrayToList() {
        String[] array = { "a", "b", "c", "d", "e", "f", "g" };
        List<String> list = CollectionsHelper.stringArrayToList(array);
        checkListAndArrayAreEqual(list, array);
    }

    @Test
    public void stringListToArray01() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e", "f", "g");
        String[] array = CollectionsHelper.stringListToArray(list);
        checkListAndArrayAreEqual(list, array);
    }

    private void checkListAndArrayAreEqual(List<String> list, String[] array) {
        assertEquals(list.size(), array.length);
        for (int index = 0; index< list.size(); ++index) {
            assertEquals(list.get(index), array[index]);
        }
    }

    @Test
    public void stringListToArrayWithEmptyList() {
        List<String> list = new ArrayList<>();
        String[] array = CollectionsHelper.stringListToArray(list);
        checkListAndArrayAreEqual(list, array);
    }

    @Test
    public void stringListToArrayWithEmptyList2() {
        List<String> list = CollectionsHelper.buildListFrom();
        String[] array = CollectionsHelper.stringListToArray(list);
        checkListAndArrayAreEqual(list, array);
    }

    @Test
    public void stringListToArrayWithEmptyOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("foo");
        String[] array = CollectionsHelper.stringListToArray(list);
        checkListAndArrayAreEqual(list, array);
    }

    @Test
    public void arrayToStringEmptyArray() {
        String[] array = {};
        String actual = CollectionsHelper.stringArrayToString(array);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void arrayToStringOneElement() {
        String[] array = { "foo" };
        String actual = CollectionsHelper.stringArrayToString(array);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void arrayToStringTwoElements() {
        String[] array = { "foo", "bar" };
        String actual = CollectionsHelper.stringArrayToString(array);
        String expected = "foo, bar";
        assertEquals(expected, actual);
    }

    @Test
    public void arrayToStringThreeElements() {
        String[] array = { "foo", "bar", "baz" };
        String actual = CollectionsHelper.stringArrayToString(array);
        String expected = "foo, bar, baz";
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfFirstNotEmptyElementWhenFirstElementIsNotEmpty() {
        List<String> list = CollectionsHelper.buildListFrom("foo");
        int actual = CollectionsHelper.determineIndexOfFirstNotEmptyElement(list);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfFirstNotEmptyElementWhenListIsEmpty() {
        List<String> list = CollectionsHelper.buildListFrom();
        int actual = CollectionsHelper.determineIndexOfFirstNotEmptyElement(list);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfFirstNotEmptyElementWhenListHasSizeOneAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("");
        int actual = CollectionsHelper.determineIndexOfFirstNotEmptyElement(list);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfFirstNotEmptyElementWhenListHasSizeTwoAndTwoEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "");
        int actual = CollectionsHelper.determineIndexOfFirstNotEmptyElement(list);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfFirstNotEmptyElementWhenListHasSizeTwoAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "foo");
        int actual = CollectionsHelper.determineIndexOfFirstNotEmptyElement(list);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyStringsAtFrontOfListWhenFirstElementIsNotEmpty() {
        List<String> list = CollectionsHelper.buildListFrom("foo");
        CollectionsHelper.removeEmptyStringsAtFrontOfList(list);
        assertEquals(1, list.size());
        assertEquals("foo", list.get(0));
    }

    @Test
    public void removeEmptyStringsAtFrontOfListWhenListIsEmpty() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeEmptyStringsAtFrontOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtFrontOfListWhenListHasSizeOneAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("");
        CollectionsHelper.removeEmptyStringsAtFrontOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtFrontOfListWhenListHasSizeTwoAndTwoEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "");
        CollectionsHelper.removeEmptyStringsAtFrontOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtFrontOfListWhenListHasSizeTwoAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "foo");
        CollectionsHelper.removeEmptyStringsAtFrontOfList(list);
        assertEquals(1, list.size());
        assertEquals("foo", list.get(0));
    }

    @Test
    public void determineIndexOfLastNotEmptyElementWhenLastElementIsNotEmpty() {
        List<String> list = CollectionsHelper.buildListFrom("foo");
        int actual = CollectionsHelper.determineIndexOfLastNotEmptyElement(list);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfLastNotEmptyElementWhenListIsEmpty() {
        List<String> list = CollectionsHelper.buildListFrom();
        int actual = CollectionsHelper.determineIndexOfLastNotEmptyElement(list);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfLastNotEmptyElementWhenListHasSizeOneAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("");
        int actual = CollectionsHelper.determineIndexOfLastNotEmptyElement(list);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfLastNotEmptyElementWhenListHasSizeTwoAndTwoEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "");
        int actual = CollectionsHelper.determineIndexOfLastNotEmptyElement(list);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void determineIndexOfLastNotEmptyElementWhenListHasSizeTwoAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "");
        int actual = CollectionsHelper.determineIndexOfLastNotEmptyElement(list);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyStringsAtEndOfListWhenLastElementIsNotEmpty() {
        List<String> list = CollectionsHelper.buildListFrom("foo");
        CollectionsHelper.removeEmptyStringsAtEndOfList(list);
        assertEquals(1, list.size());
        assertEquals("foo", list.get(0));
    }

    @Test
    public void removeEmptyStringsAtEndOfListWhenListIsEmpty() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeEmptyStringsAtEndOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtEndOfListWhenListHasSizeOneAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("");
        CollectionsHelper.removeEmptyStringsAtEndOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtEndOfListWhenListHasSizeTwoAndTwoEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("", "");
        CollectionsHelper.removeEmptyStringsAtEndOfList(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeEmptyStringsAtEndOfListWhenListHasSizeTwoAndOneEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "");
        CollectionsHelper.removeEmptyStringsAtEndOfList(list);
        assertEquals(1, list.size());
        assertEquals("foo", list.get(0));
    }

    @Test
    public void sortStringListByLengthDescandingAlphabeticalOnEqualLengthElements() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        CollectionsHelper.sortStringListByLengthDescanding(list);
        assertEquals(3, list.size());
        assertEquals("bar", list.get(0));
        assertEquals("baz", list.get(1));
        assertEquals("foo", list.get(2));
    }

    @Test
    public void sortStringListByLengthDescandingOnDifferentLength() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "ccc");
        CollectionsHelper.sortStringListByLengthDescanding(list);
        assertEquals(3, list.size());
        assertEquals("ccc", list.get(0));
        assertEquals("aa", list.get(1));
        assertEquals("b", list.get(2));
    }

    @Test
    public void sortStringListByLengthDescandingOnMixedData() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd");
        CollectionsHelper.sortStringListByLengthDescanding(list);
        assertEquals(7, list.size());
        assertEquals("dddd", list.get(0));
        assertEquals("bar", list.get(1));
        assertEquals("baz", list.get(2));
        assertEquals("ccc", list.get(3));
        assertEquals("foo", list.get(4));
        assertEquals("aa", list.get(5));
        assertEquals("b", list.get(6));
    }

    @Test
    public void isDisjunctString() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd");
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void isDisjunctInt() {
        List<Integer> list = CollectionsHelper.buildListFrom(1, 2, -5, 13);
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void isDisjunctObject() {
        List<Object> list = new ArrayList<>();
        list.add("1");
        list.add(1);
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void isDisjunctStringFailes() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd", "foo");
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isDisjunctIntFailes() {
        List<Integer> list = CollectionsHelper.buildListFrom(1, 2, -5, 13, -5);
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isDisjunctObjectFailes() {
        List<Object> list = new ArrayList<>();
        list.add("1");
        list.add(1);
        list.add("1");
        boolean actual = CollectionsHelper.isDisjunct(list);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void getNotDisjunktTexts1() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd");
        List<String> notDisjunctTexts = CollectionsHelper.getNotDisjunktTexts(list);
        assertTrue(notDisjunctTexts.isEmpty());
    }

    @Test
    public void getNotDisjunktTexts2() {
        List<String> list = new ArrayList<>();
        list.add("1");
        list.add("2");
        List<String> notDisjunctTexts = CollectionsHelper.getNotDisjunktTexts(list);
        assertTrue(notDisjunctTexts.isEmpty());
    }

    @Test
    public void getNotDisjunktTextsFailes() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd", "foo");
        List<String> notDisjunctTexts = CollectionsHelper.getNotDisjunktTexts(list);
        assertFalse(notDisjunctTexts.isEmpty());
        assertEquals(1, notDisjunctTexts.size());
        assertEquals("foo", notDisjunctTexts.get(0));
    }

    @Test
    public void getNotDisjunktTextsFailesMultiple() {
        List<String> list = CollectionsHelper.buildListFrom("aa", "b", "foo", "bar", "baz", "ccc",
                "dddd", "foo", "foo", "bar", "ccc", "baz");
        List<String> notDisjunctTexts = CollectionsHelper.getNotDisjunktTexts(list);
        assertFalse(notDisjunctTexts.isEmpty());
        assertEquals(4, notDisjunctTexts.size());
        assertEquals("foo", notDisjunctTexts.get(0));
        assertEquals("bar", notDisjunctTexts.get(1));
        assertEquals("baz", notDisjunctTexts.get(2));
        assertEquals("ccc", notDisjunctTexts.get(3));
    }

    @Test
    public void createDisjunctLowerCaseList() {
        List<String> list = CollectionsHelper.buildListFrom("B", "a", "C", "A", "b", "D");
        List<String> disjunctLowerCaseList = CollectionsHelper.createDisjunctLowerCaseList(list);
        assertEquals(4, disjunctLowerCaseList.size());
        assertEquals("b", disjunctLowerCaseList.get(0));
        assertEquals("a", disjunctLowerCaseList.get(1));
        assertEquals("c", disjunctLowerCaseList.get(2));
        assertEquals("d", disjunctLowerCaseList.get(3));
    }

    @Test
    public void createDisjunctListFromDisjunctList() {
        List<String> list = CollectionsHelper.buildListFrom("c", "a", "b");
        List<String> disjunctList = CollectionsHelper.createDisjunctList(list);
        assertEquals(3, disjunctList.size());
        assertEquals("c", disjunctList.get(0));
        assertEquals("a", disjunctList.get(1));
        assertEquals("b", disjunctList.get(2));
    }

    @Test
    public void createDisjunctListFromNotDisjunctList() {
        List<String> list = CollectionsHelper.buildListFrom("c", "a", "c", "a", "b", "b", "a");
        List<String> disjunctList = CollectionsHelper.createDisjunctList(list);
        assertEquals(3, disjunctList.size());
        assertEquals("c", disjunctList.get(0));
        assertEquals("a", disjunctList.get(1));
        assertEquals("b", disjunctList.get(2));
    }

    @Test
    public void createDisjunctListFromNotDisjunctIntList() {
        List<Integer> list = CollectionsHelper.buildListFrom(3, 1, 3, 1, 2, 2, 1);
        List<Integer> disjunctList = CollectionsHelper.createDisjunctList(list);
        assertEquals(3, disjunctList.size());
        assertEquals(Integer.valueOf(3), disjunctList.get(0));
        assertEquals(Integer.valueOf(1), disjunctList.get(1));
        assertEquals(Integer.valueOf(2), disjunctList.get(2));
    }

    @Test
    public void createDisjunctListFromNotDisjunctListsOfLists() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("c", "a", "c");
        List<String> list3 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list4 = CollectionsHelper.buildListFrom("c", "b", "a");
        List<List<String>> list = new ArrayList<>();
        list.add(list1);
        list.add(list2);
        list.add(list3);
        list.add(list4);
        List<List<String>> disjunctList = CollectionsHelper.createDisjunctList(list);
        assertEquals(3, disjunctList.size());
        assertEquals(list1, disjunctList.get(0));
        assertEquals(list2, disjunctList.get(1));
        assertEquals(list4, disjunctList.get(2));
    }

    @Test
    public void makeListDisjunctFromDisjunctList() {
        List<String> list = CollectionsHelper.buildListFrom("c", "a", "b");
        CollectionsHelper.makeListDisjunct(list);
        assertEquals(3, list.size());
        assertEquals("c", list.get(0));
        assertEquals("a", list.get(1));
        assertEquals("b", list.get(2));
    }

    @Test
    public void makeListDisjunctFromNotDisjunctList() {
        List<String> list = CollectionsHelper.buildListFrom("c", "a", "c", "a", "b", "b", "a");
        CollectionsHelper.makeListDisjunct(list);
        assertEquals(3, list.size());
        assertEquals("c", list.get(0));
        assertEquals("a", list.get(1));
        assertEquals("b", list.get(2));
    }

    @Test
    public void makeListDisjunctFromNotDisjunctIntList() {
        List<Integer> list = CollectionsHelper.buildListFrom(3, 1, 3, 1, 2, 2, 1);
        CollectionsHelper.makeListDisjunct(list);
        assertEquals(3, list.size());
        assertEquals(Integer.valueOf(3), list.get(0));
        assertEquals(Integer.valueOf(1), list.get(1));
        assertEquals(Integer.valueOf(2), list.get(2));
    }

    @Test
    public void makeListDisjunctFromNotDisjunctListsOfLists() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("c", "a", "c");
        List<String> list3 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list4 = CollectionsHelper.buildListFrom("c", "b", "a");
        List<List<String>> list = new ArrayList<>();
        list.add(list1);
        list.add(list2);
        list.add(list3);
        list.add(list4);
        CollectionsHelper.makeListDisjunct(list);
        assertEquals(3, list.size());
        assertEquals(list1, list.get(0));
        assertEquals(list2, list.get(1));
        assertEquals(list4, list.get(2));
    }

    @Test
    public void toLowerCase() {
        List<String> list = CollectionsHelper.buildListFrom("Aaa", "A", "BbbBbb", "B", "cCcc", "d",
                "D", "EeeeE");
        List<String> loweredList = CollectionsHelper.toLowerCase(list);
        assertEquals(8, loweredList.size());
        assertEquals("aaa", loweredList.get(0));
        assertEquals("a", loweredList.get(1));
        assertEquals("bbbbbb", loweredList.get(2));
        assertEquals("b", loweredList.get(3));
        assertEquals("cccc", loweredList.get(4));
        assertEquals("d", loweredList.get(5));
        assertEquals("d", loweredList.get(6));
        assertEquals("eeeee", loweredList.get(7));
    }

    @Test
    public void toUpperCase() {
        List<String> list = CollectionsHelper.buildListFrom("Aaa", "A", "BbbBbb", "B", "cCcc", "d",
                "D", "EeeeE");
        List<String> loweredList = CollectionsHelper.toUpperCase(list);
        assertEquals(8, loweredList.size());
        assertEquals("AAA", loweredList.get(0));
        assertEquals("A", loweredList.get(1));
        assertEquals("BBBBBB", loweredList.get(2));
        assertEquals("B", loweredList.get(3));
        assertEquals("CCCC", loweredList.get(4));
        assertEquals("D", loweredList.get(5));
        assertEquals("D", loweredList.get(6));
        assertEquals("EEEEE", loweredList.get(7));
    }

    @Test
    public void sortAndToLowerCase() {
        List<String> list = CollectionsHelper.buildSortByLengthDescandingListFrom("Aaa", "A",
                "BbbBbb", "B", "cCcc", "d", "D", "EeeeE");
        List<String> loweredList = CollectionsHelper.toLowerCase(list);
        assertEquals(8, loweredList.size());
        assertEquals("bbbbbb", loweredList.get(0));
        assertEquals("eeeee", loweredList.get(1));
        assertEquals("cccc", loweredList.get(2));
        assertEquals("aaa", loweredList.get(3));
        assertEquals("a", loweredList.get(4));
        assertEquals("b", loweredList.get(5));
        assertEquals("d", loweredList.get(6));
        assertEquals("d", loweredList.get(7));
    }

    @Test
    public void sortAndToUpperCase() {
        List<String> list = CollectionsHelper.buildSortByLengthDescandingListFrom("Aaa", "A",
                "BbbBbb", "B", "cCcc", "d", "D", "EeeeE");
        List<String> loweredList = CollectionsHelper.toUpperCase(list);
        assertEquals(8, loweredList.size());
        assertEquals("BBBBBB", loweredList.get(0));
        assertEquals("EEEEE", loweredList.get(1));
        assertEquals("CCCC", loweredList.get(2));
        assertEquals("AAA", loweredList.get(3));
        assertEquals("A", loweredList.get(4));
        assertEquals("B", loweredList.get(5));
        assertEquals("D", loweredList.get(6));
        assertEquals("D", loweredList.get(7));
    }

    @Test
    public void surroundElements() {
        List<String> list = CollectionsHelper.buildSortByLengthDescandingListFrom("foo", "bar", "baz");
        String before = "[";
        String after = "]";
        List<String> surroundedList = CollectionsHelper.surroundElements(list, before, after);
        assertEquals(3, surroundedList.size());
        assertEquals("[foo]", surroundedList.get(0));
        assertEquals("[bar]", surroundedList.get(1));
        assertEquals("[baz]", surroundedList.get(2));
    }

    @Test
    public void keepOnlyFirstLines() {
        List<String> list = CollectionsHelper.buildListFrom("1", "2", "3", "4", "5", "6", "7", "8",
                "9", "10");
        int numberOfLinesToKeep = 3;
        List<String> shorterList = CollectionsHelper.keepOnlyFirstLines(list, numberOfLinesToKeep);
        assertEquals(3, shorterList.size());
        assertEquals("1", shorterList.get(0));
        assertEquals("2", shorterList.get(1));
        assertEquals("3", shorterList.get(2));
    }

    @Test
    public void appendInfrontOfArray() {
        String firstValue = "eins";
        String[] otherValues = { "zwei", "drei", "vier", "fünf" };
        String[] array = CollectionsHelper.appendInfrontOfArray(firstValue, otherValues);
        assertEquals(5, array.length);
        assertEquals("eins", array[0]);
        assertEquals("zwei", array[1]);
        assertEquals("drei", array[2]);
        assertEquals("vier", array[3]);
        assertEquals("fünf", array[4]);
    }

    @Test
    public void appendAtEndOfArray() {
        String[] otherValues = { "eins", "zwei", "drei", "vier" };
        String lastValue = "fünf";
        String[] array = CollectionsHelper.appendAtEndOfArray(otherValues, lastValue);
        assertEquals(5, array.length);
        assertEquals("eins", array[0]);
        assertEquals("zwei", array[1]);
        assertEquals("drei", array[2]);
        assertEquals("vier", array[3]);
        assertEquals("fünf", array[4]);
    }

    @Test
    public void getSection() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.getSection(list, start, end);
        assertEquals(1, section.size());
        assertEquals("bar", section.get(0));
    }

    @Test
    public void getSectionInLine() {
        List<String> list = CollectionsHelper.buildListFrom("    foo   ", "bar", "    baz    ");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.getSection(list, start, end);
        assertEquals(1, section.size());
        assertEquals("bar", section.get(0));
    }

    @Test (expected = RuntimeException.class)
    public void getSectionNoStartFound() {
        List<String> list = CollectionsHelper.buildListFrom("    foo   ", "bar", "    baz    ");
        String start = "Hugo";
        String end = "baz";
        CollectionsHelper.getSection(list, start, end);
    }

    @Test (expected = RuntimeException.class)
    public void getSectionNoEndFound() {
        List<String> list = CollectionsHelper.buildListFrom("    foo   ", "bar", "    baz    ");
        String start = "foo";
        String end = "Hugo";
        CollectionsHelper.getSection(list, start, end);
    }

    @Test
    public void removeSectionExcludingStartAndEnd() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.removeSectionExcludingStartAndEnd(list, start, end);
        assertEquals(2, section.size());
        assertEquals("foo", section.get(0));
        assertEquals("baz", section.get(1));
    }

    @Test
    public void removeSectionExcludingStartAndEndInLine() {
        List<String> list = CollectionsHelper.buildListFrom("   foo    ", "bar", "     foobazbar");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.removeSectionExcludingStartAndEnd(list, start, end);
        assertEquals(2, section.size());
        assertEquals("   foo    ", section.get(0));
        assertEquals("     foobazbar", section.get(1));
    }

    @Test
    public void removeSectionIncludingStartAndEndRemoveEverything() {
        List<String> list = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.removeSectionIncludingStartAndEnd(list, start, end);
        assertEquals(0, section.size());
    }

    @Test
    public void removeSectionIncludingStartAndEnd() {
        List<String> list = CollectionsHelper.buildListFrom("erste", "foo", "bar", "baz", "letzte");
        String start = "foo";
        String end = "baz";
        List<String> section = CollectionsHelper.removeSectionIncludingStartAndEnd(list, start, end);
        assertEquals(2, section.size());
        assertEquals("erste", section.get(0));
        assertEquals("letzte", section.get(1));
    }

    @Test
    public void anyLineContains() {
        List<String> list = CollectionsHelper.buildListFrom("  foo   ", "  bar ", " baz  ");
        assertTrue(CollectionsHelper.anyLineContains(list, "foo"));
        assertTrue(CollectionsHelper.anyLineContains(list, "bar"));
        assertTrue(CollectionsHelper.anyLineContains(list, "baz"));
        assertFalse(CollectionsHelper.anyLineContains(list, "foobar"));
    }

    @Test
    public void findAllIndces() {
        List<String> list = CollectionsHelper.buildListFrom(
                "foo", "bar", "foo", "baz", "foo", "nichtfoo");
        List<Integer> indices = CollectionsHelper.findAllIndces("foo", list);
        assertEquals(3, indices.size());
        assertEquals(Integer.valueOf(0), indices.get(0));
        assertEquals(Integer.valueOf(2), indices.get(1));
        assertEquals(Integer.valueOf(4), indices.get(2));
    }

    @Test
    public void findAllIndcesInt() {
        List<Integer> list = CollectionsHelper.buildListFrom(1, 2, 1, 3, 1, 4);
        List<Integer> indices = CollectionsHelper.findAllIndces(1, list);
        assertEquals(3, indices.size());
        assertEquals(Integer.valueOf(0), indices.get(0));
        assertEquals(Integer.valueOf(2), indices.get(1));
        assertEquals(Integer.valueOf(4), indices.get(2));
    }

    @Test
    public void joinStringLists() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz");
        List<String> list2 = CollectionsHelper.buildListFrom(
                "Äpfel", "Birnen", "Zitronen");
        List<String> list = CollectionsHelper.joinStringLists(list1, list2);
        assertEquals(6, list.size());
        assertEquals(list1.get(0), list.get(0));
        assertEquals(list1.get(1), list.get(1));
        assertEquals(list1.get(2), list.get(2));
        assertEquals(list2.get(0), list.get(3));
        assertEquals(list2.get(1), list.get(4));
        assertEquals(list2.get(2), list.get(5));
    }

    @Test
    public void joinLists() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list2 = CollectionsHelper.buildListFrom(3, 4);
        List<Integer> list3 = CollectionsHelper.buildListFrom(5, 6);
        List<Integer> list = CollectionsHelper.joinLists(list1, list2, list3);
        assertEquals(6, list.size());
        assertEquals(list1.get(0), list.get(0));
        assertEquals(list1.get(1), list.get(1));
        assertEquals(list2.get(0), list.get(2));
        assertEquals(list2.get(1), list.get(3));
        assertEquals(list3.get(0), list.get(4));
        assertEquals(list3.get(1), list.get(5));
    }

    @Test
    public void joinListsWithEqualElements() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list2 = CollectionsHelper.buildListFrom(2, 3);
        List<Integer> list3 = CollectionsHelper.buildListFrom(3, 4);
        List<Integer> list = CollectionsHelper.joinLists(list1, list2, list3);
        assertEquals(6, list.size());
        assertEquals(list1.get(0), list.get(0));
        assertEquals(list1.get(1), list.get(1));
        assertEquals(list2.get(0), list.get(2));
        assertEquals(list2.get(1), list.get(3));
        assertEquals(list3.get(0), list.get(4));
        assertEquals(list3.get(1), list.get(5));
    }

    @Test
    public void joinListsDisjunct() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list2 = CollectionsHelper.buildListFrom(3, 4);
        List<Integer> list3 = CollectionsHelper.buildListFrom(5, 6);
        List<Integer> list = CollectionsHelper.joinListsDisjunct(list1, list2, list3);
        assertEquals(6, list.size());
        assertEquals(list1.get(0), list.get(0));
        assertEquals(list1.get(1), list.get(1));
        assertEquals(list2.get(0), list.get(2));
        assertEquals(list2.get(1), list.get(3));
        assertEquals(list3.get(0), list.get(4));
        assertEquals(list3.get(1), list.get(5));
    }

    @Test
    public void joinListsDisjunctWithEqualElements() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list2 = CollectionsHelper.buildListFrom(2, 3);
        List<Integer> list3 = CollectionsHelper.buildListFrom(3, 4);
        List<Integer> list = CollectionsHelper.joinListsDisjunct(list1, list2, list3);
        assertEquals(4, list.size());
        assertEquals(list1.get(0), list.get(0));
        assertEquals(list1.get(1), list.get(1));
        assertEquals(list2.get(1), list.get(2));
        assertEquals(list3.get(1), list.get(3));
    }

    @Test
    public void copyList() {
        List<String> list = CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen");
        List<String> copied = CollectionsHelper.copyList(list);
        list = CollectionsHelper.buildListFrom("Automaten", "Bastionen", "Zentralen");
        assertEquals(CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen"), copied);
    }

    @Test
    public void trimList() {
        List<String> list = CollectionsHelper.buildListFrom("  Äpfel ", "Birnen    ", "Zitronen");
        List<String> trimmed = CollectionsHelper.trimList(list);
        assertEquals(CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen"), trimmed);
    }

    @Test
    public void trimAndCompactSpaces() {
        List<String> list = CollectionsHelper.buildListFrom("  Äpfel   mit   Zimt ", "Birnen    ",
                "Zitronen");
        List<String> trimmed = CollectionsHelper.trimAndCompactSpaces(list);
        assertEquals(CollectionsHelper.buildListFrom("Äpfel mit Zimt", "Birnen", "Zitronen"),
                trimmed);
    }

    @Test
    public void listStartsWith() {
        List<String> list = CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen");
        List<String> start = CollectionsHelper.buildListFrom("Äpfel", "Birnen");
        assertTrue(CollectionsHelper.listStartsWith(list, start));
    }

    @Test
    public void listStartsNotWith() {
        List<String> list = CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen");
        List<String> start = CollectionsHelper.buildListFrom("Äpfel", "Bananen");
        assertFalse(CollectionsHelper.listStartsWith(list, start));
    }

    @Test
    public void listStartsNotWith2() {
        List<String> list = CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen");
        List<String> start = CollectionsHelper.buildListFrom("Ananas", "Birnen");
        assertFalse(CollectionsHelper.listStartsWith(list, start));
    }

    @Test
    public void listStartsWithEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom("Äpfel", "Birnen", "Zitronen");
        List<String> start = CollectionsHelper.buildListFrom();
        assertTrue(CollectionsHelper.listStartsWith(list, start));
    }

    @Test
    public void emptyListStartsWithEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        List<String> start = CollectionsHelper.buildListFrom();
        assertTrue(CollectionsHelper.listStartsWith(list, start));
    }

    @Test
    public void inputEqualsAnyListElementEmptyInputEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputEqualsAnyListElement(input, list));
    }

    @Test
    public void inputEqualsAnyListElementNotEmptyInputEmptyList() {
        String input = "foo";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputEqualsAnyListElement(input, list));
    }

    @Test
    public void inputEqualsAnyListElementEmptyInputNotEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertFalse(CollectionsHelper.inputEqualsAnyListElement(input, list));
    }

    @Test
    public void inputEqualsAnyListElementNormalCase() {
        String input = "AT";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertTrue(CollectionsHelper.inputEqualsAnyListElement(input, list));
    }

    @Test
    public void inputEqualsAnyListElementNormalCaseNotEquals() {
        String input = "IT";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertFalse(CollectionsHelper.inputEqualsAnyListElement(input, list));
    }

    @Test
    public void inputStartsWithAnyListElementEmptyInputEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputStartsWithAnyListElement(input, list));
    }

    @Test
    public void inputStartsWithAnyListElementNotEmptyInputEmptyList() {
        String input = "foo";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputStartsWithAnyListElement(input, list));
    }

    @Test
    public void inputStartsWithAnyListElementEmptyInputNotEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertFalse(CollectionsHelper.inputStartsWithAnyListElement(input, list));
    }

    @Test
    public void inputStartsWithAnyListElementNormalCase() {
        String input = "AT-Wien";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertTrue(CollectionsHelper.inputStartsWithAnyListElement(input, list));
    }

    @Test
    public void inputStartsWithAnyListElementNormalCaseNotStarting() {
        String input = "IT-Rom";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertFalse(CollectionsHelper.inputStartsWithAnyListElement(input, list));
    }

    @Test
    public void inputEndsWithAnyListElementEmptyInputEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputEndsWithAnyListElement(input, list));
    }

    @Test
    public void inputEndsWithAnyListElementNotEmptyInputEmptyList() {
        String input = "foo";
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.inputEndsWithAnyListElement(input, list));
    }

    @Test
    public void inputEndsWithAnyListElementEmptyInputNotEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertFalse(CollectionsHelper.inputEndsWithAnyListElement(input, list));
    }

    @Test
    public void inputEndsWithAnyListElementNormalCase() {
        String input = "AT-Wien";
        List<String> list = CollectionsHelper.buildListFrom("Rom", "München", "Wien");
        assertTrue(CollectionsHelper.inputEndsWithAnyListElement(input, list));
    }

    @Test
    public void inputEndsWithAnyListElementNormalCaseNotStarting() {
        String input = "IT-Rom";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertFalse(CollectionsHelper.inputEndsWithAnyListElement(input, list));
    }

    @Test
    public void getElementOnWichTheInputEndsEmptyInputEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom();
        assertEquals("", CollectionsHelper.getElementOnWichTheInputEnds(input, list));
    }

    @Test
    public void getElementOnWichTheInputEndsNotEmptyInputEmptyList() {
        String input = "foo";
        List<String> list = CollectionsHelper.buildListFrom();
        assertEquals("", CollectionsHelper.getElementOnWichTheInputEnds(input, list));
    }

    @Test
    public void getElementOnWichTheInputEndsEmptyInputNotEmptyList() {
        String input = "";
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertEquals("", CollectionsHelper.getElementOnWichTheInputEnds(input, list));
    }

    @Test
    public void getElementOnWichTheInputEndsNormalCase() {
        String input = "AT-Wien";
        List<String> list = CollectionsHelper.buildListFrom("Rom", "München", "Wien");
        assertEquals("Wien", CollectionsHelper.getElementOnWichTheInputEnds(input, list));
    }

    @Test
    public void getElementOnWichTheInputEndsNormalCaseNotStarting() {
        String input = "IT-Rom";
        List<String> list = CollectionsHelper.buildListFrom("GB", "AT", "DE");
        assertEquals("", CollectionsHelper.getElementOnWichTheInputEnds(input, list));
    }

    @Test
    public void splitStringIntoLettersEmptyText() {
        String text = "";
        List<String> actual = CollectionsHelper.splitStringIntoLetters(text);
        List<String> expected = CollectionsHelper.buildListFrom();
        assertEquals(expected, actual);
    }

    @Test
    public void splitStringIntoLettersAbc() {
        String text = "abc";
        List<String> actual = CollectionsHelper.splitStringIntoLetters(text);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "c");
        assertEquals(expected, actual);
    }

    @Test
    public void replace() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "Äpfel, Birnen und Zitronen.",
                "Bananen enthalten keine Birnen.",
                "Eine Zeile ohne gesuchtes Wort.",
                "Der Herbst ist Birnenzeit!");
        String toReplace = "Birnen";
        String replacement = "Pampelmusen";

        List<String> actual = CollectionsHelper.replace(lines, toReplace, replacement);
        List<String> expected = CollectionsHelper.buildListFrom(
                "Äpfel, Pampelmusen und Zitronen.",
                "Bananen enthalten keine Pampelmusen.",
                "Eine Zeile ohne gesuchtes Wort.",
                "Der Herbst ist Pampelmusenzeit!");
        assertEquals(expected, actual);
    }

    @Test
    public void replaceAll() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "Äpfel, Birnen und Zitronen.",
                "Bananen enthalten keine Birnen.",
                "Eine Zeile ohne gesuchtes Wort.",
                "Der Herbst ist Birnenzeit!");
        String toReplaceRegex = "(?:Zitronen|Birnen)";
        String replacement = "Pampelmusen";

        List<String> actual = CollectionsHelper.replaceAll(lines, toReplaceRegex, replacement);
        List<String> expected = CollectionsHelper.buildListFrom(
                "Äpfel, Pampelmusen und Pampelmusen.",
                "Bananen enthalten keine Pampelmusen.",
                "Eine Zeile ohne gesuchtes Wort.",
                "Der Herbst ist Pampelmusenzeit!");
        assertEquals(expected, actual);
    }

    @Test
    public void applyToListDouble() {
        List<Double> values = CollectionsHelper.buildListFrom(
                98.12134D,
                -5121.22134545D,
                0.0000341D
                );
        List<String> results = CollectionsHelper.applyToList(values,
                value -> NumberString.twoDecimalPlaces(value));
        assertEquals(3, results.size());
        assertEquals("98,12", results.get(0));
        assertEquals("-5121,22", results.get(1));
        assertEquals("0,00", results.get(2));
    }

    @Test
    public void applyToListInt() {
        List<Integer> values = CollectionsHelper.buildListFrom(
                15,
                33,
                7
                );
        List<String> results = CollectionsHelper.applyToList(values,
                value -> NumberString.addLeadingZeroes(value, 3));
        assertEquals(3, results.size());
        assertEquals("015", results.get(0));
        assertEquals("033", results.get(1));
        assertEquals("007", results.get(2));
    }

    @Test
    public void arrayToListString() {
        String[] array = { "eins", "zwei", "drei", "vier", "fünf" };
        List<String> list = CollectionsHelper.arrayToList(array);
        assertEquals(5, list.size());
        assertEquals(array[0], list.get(0));
        assertEquals(array[1], list.get(1));
        assertEquals(array[2], list.get(2));
        assertEquals(array[3], list.get(3));
        assertEquals(array[4], list.get(4));
    }

    @Test
    public void arrayToListInteger() {
        Integer[] array = { 1, 2, 3, 4, 5 };
        List<Integer> list = CollectionsHelper.arrayToList(array);
        assertEquals(5, list.size());
        assertEquals(array[0], list.get(0));
        assertEquals(array[1], list.get(1));
        assertEquals(array[2], list.get(2));
        assertEquals(array[3], list.get(3));
        assertEquals(array[4], list.get(4));
    }

    @Test
    public void arrayToListIntToInteger() {
        int[] array = { 1, 2, 3, 4, 5 };
        List<Integer> list = CollectionsHelper.arrayToList(array);
        assertEquals(array.length, list.size());
        for (int index = 0; index < list.size(); ++index) {
            int fromArray = array[index];
            Integer fromList = list.get(index);
            int fromListInt = fromList;
            assertEquals(fromArray, fromListInt);
        }
    }

    @Test
    public void arrayToListDoubleToDouble() {
        double[] array = { 1.1d, 2.2d, 3.3d, 4.4d, 55.5d };
        List<Double> list = CollectionsHelper.arrayToList(array);
        assertEquals(array.length, list.size());
        for (int index = 0; index < list.size(); ++index) {
            double fromArray = array[index];
            Double fromList = list.get(index);
            double fromListDouble = fromList;
            assertEquals(fromArray, fromListDouble, 0.0001);
        }
    }

    @Test
    public void sortAscanding() {
        List<Integer> list = CollectionsHelper.buildListFrom(4, 2, 3, 1, 5);
        CollectionsHelper.sortAscanding(list);
        assertEquals(5, list.size());
        assertEquals(Integer.valueOf(1), list.get(0));
        assertEquals(Integer.valueOf(2), list.get(1));
        assertEquals(Integer.valueOf(3), list.get(2));
        assertEquals(Integer.valueOf(4), list.get(3));
        assertEquals(Integer.valueOf(5), list.get(4));
    }

    @Test
    public void sortDescanding() {
        List<Integer> list = CollectionsHelper.buildListFrom(4, 2, 3, 1, 5);
        CollectionsHelper.sortDescanding(list);
        assertEquals(5, list.size());
        assertEquals(Integer.valueOf(5), list.get(0));
        assertEquals(Integer.valueOf(4), list.get(1));
        assertEquals(Integer.valueOf(3), list.get(2));
        assertEquals(Integer.valueOf(2), list.get(3));
        assertEquals(Integer.valueOf(1), list.get(4));
    }

    @Test
    public void removeElementsFromListByIndices() {
        List<String> list = CollectionsHelper.buildListFrom("A", "B", "C", "D", "E", "F", "G");
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom(1, 2, 3, 1, 3);
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
        assertEquals(4, list.size());
        assertEquals("A", list.get(0));
        assertEquals("E", list.get(1));
        assertEquals("F", list.get(2));
        assertEquals("G", list.get(3));
    }

    @Test
    public void removeElementsFromListByIndices2() {
        List<String> list = CollectionsHelper.buildListFrom("A", "B", "C", "D", "E", "F", "G");
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom(2, 3, 2, 3);
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
        assertEquals(5, list.size());
        assertEquals("A", list.get(0));
        assertEquals("B", list.get(1));
        assertEquals("E", list.get(2));
        assertEquals("F", list.get(3));
        assertEquals("G", list.get(4));
    }

    @Test (expected = IndexOutOfBoundsException.class)
    public void removeElementsFromListByIndicesWrongIndex() {
        List<String> list = CollectionsHelper.buildListFrom("A");
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom(23);
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
    }

    @Test (expected = IndexOutOfBoundsException.class)
    public void removeElementsFromListByIndicesWrongIndexLow() {
        List<String> list = CollectionsHelper.buildListFrom("A");
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom(-4);
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
    }

    @Test
    public void removeElementsFromListByIndicesEmptyIndeicesListRemovesNothing() {
        List<String> list = CollectionsHelper.buildListFrom("A", "B", "C", "D", "E", "F", "G");
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
        assertEquals(7, list.size());
        assertEquals("A", list.get(0));
        assertEquals("B", list.get(1));
        assertEquals("C", list.get(2));
        assertEquals("D", list.get(3));
        assertEquals("E", list.get(4));
        assertEquals("F", list.get(5));
        assertEquals("G", list.get(6));
    }

    @Test
    public void removeElementsFromListByIndicesEmptyIndeicesListRemovesNothingFromEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        List<Integer> indicesToDelete = CollectionsHelper.buildListFrom();
        CollectionsHelper.removeElementsFromListByIndices(list, indicesToDelete);
        assertEquals(0, list.size());
    }

    @Test
    public void createListDifferenceOnFilledEqualLists() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("a", "b", "c");

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Beide Listen haben 3 Elemente." + n
            + "Die Elemente an gemeinsamen Indices sind in beiden Listen identisch." + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void createListDifferenceOnEmptylLists() {
        List<String> list1 = CollectionsHelper.buildListFrom();
        List<String> list2 = CollectionsHelper.buildListFrom();

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Beide Listen haben 0 Elemente." + n
            + "Die Elemente an gemeinsamen Indices sind in beiden Listen identisch." + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void createListDifferenceOnFilledDifferentListsWithSameCommonValuesSecondLonger() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("a", "b", "c", "d");

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Die Listen haben eine unterschiedliche Anzahl an Elementen!" + n
            + "    Liste 1 hat 3 Elemente." + n
            + "    Liste 2 hat 4 Elemente." + n
            + "Weitere Elemente in Liste 2:" + n
            + "    4 : d" + n
            + "Die Elemente an gemeinsamen Indices sind in beiden Listen identisch." + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void createListDifferenceOnFilledDifferentListsWithSameCommonValuesFirstLonger() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c", "d");
        List<String> list2 = CollectionsHelper.buildListFrom("a", "b", "c");

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Die Listen haben eine unterschiedliche Anzahl an Elementen!" + n
            + "    Liste 1 hat 4 Elemente." + n
            + "    Liste 2 hat 3 Elemente." + n
            + "Weitere Elemente in Liste 1:" + n
            + "    4 : d" + n
            + "Die Elemente an gemeinsamen Indices sind in beiden Listen identisch." + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void createListDifferenceOnFilledDifferentListsWithSameLengthButDifferentCommonValues() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "B", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("a", "b", "c");

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Beide Listen haben 3 Elemente." + n
            + "Element Nummer 2 (also am Index 1) ist verschieden:" + n
            + "    Liste 1 hat dort das Element 'B'." + n
            + "    Liste 2 hat dort das Element 'b'." + n
            + "Die beiden Listen haben 1 unterschiedliche(s) Element(e) an an gemeinsamen Indices."
            + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void createListDifferenceOnFilledDifferentLists() {
        List<String> list1 = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> list2 = CollectionsHelper.buildListFrom("d", "e");

        String description1 = "Liste 1";
        String description2 = "Liste 2";
        String actual = CollectionsHelper.createListDifference(list1, description1,
                list2, description2);
        String n = Text.LINE_BREAK;
        String expected = ""
            + "Vergleich der Listen 'Liste 1' (1) und 'Liste 2' (2):" + n
            + "Die Listen haben eine unterschiedliche Anzahl an Elementen!" + n
            + "    Liste 1 hat 3 Elemente." + n
            + "    Liste 2 hat 2 Elemente." + n
            + "Element Nummer 1 (also am Index 0) ist verschieden:" + n
            + "    Liste 1 hat dort das Element 'a'." + n
            + "    Liste 2 hat dort das Element 'd'." + n
            + "Element Nummer 2 (also am Index 1) ist verschieden:" + n
            + "    Liste 1 hat dort das Element 'b'." + n
            + "    Liste 2 hat dort das Element 'e'." + n
            + "Weitere Elemente in Liste 1:" + n
            + "    3 : c" + n
            + "Die beiden Listen haben 2 unterschiedliche(s) Element(e) an an gemeinsamen Indices."
            + n
            ;

        assertEquals(expected, actual);
    }

    @Test
    public void containsIgnoreCase() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        String element = "a";
        assertTrue(CollectionsHelper.containsIgnoreCase(list, element));
    }

    @Test
    public void containsIgnoreCaseOtherCase() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        String element = "A";
        assertTrue(CollectionsHelper.containsIgnoreCase(list, element));
    }

    @Test
    public void containsIgnoreCaseNotContained() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        String element = "d";
        assertFalse(CollectionsHelper.containsIgnoreCase(list, element));
    }

    @Test
    public void containsIgnoreCaseEmptyListDoesNotContaineEmptyElement() {
        List<String> list = CollectionsHelper.buildListFrom();
        String element = "";
        assertFalse(CollectionsHelper.containsIgnoreCase(list, element));
    }

    @Test
    public void mixIntoListAddBehind() {
        List<String> originalList = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> newElements = CollectionsHelper.buildListFrom("d", "e", "f");
        int insertColumn = 3;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixIntoListAddBefore() {
        List<String> originalList = CollectionsHelper.buildListFrom("d", "e", "f");
        List<String> newElements = CollectionsHelper.buildListFrom("a", "b", "c");
        int insertColumn = 0;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixIntoListAddInMiddle() {
        List<String> originalList = CollectionsHelper.buildListFrom("a", "b", "e", "f");
        List<String> newElements = CollectionsHelper.buildListFrom("c", "d");
        int insertColumn = 2;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "c", "d", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixEmptyListIntoListAddBehind() {
        List<String> originalList = CollectionsHelper.buildListFrom("a", "b", "c");
        List<String> newElements = CollectionsHelper.buildListFrom();
        int insertColumn = 3;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "c");
        assertEquals(expected, actual);
    }

    @Test
    public void mixEmptyListIntoListAddBefore() {
        List<String> originalList = CollectionsHelper.buildListFrom("d", "e", "f");
        List<String> newElements = CollectionsHelper.buildListFrom();
        int insertColumn = 0;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("d", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixEmptyListIntoListAddInMiddle() {
        List<String> originalList = CollectionsHelper.buildListFrom("a", "b", "e", "f");
        List<String> newElements = CollectionsHelper.buildListFrom();
        int insertColumn = 2;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("a", "b", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixIntoListAddToEmptyList() {
        List<String> originalList = CollectionsHelper.buildListFrom();
        List<String> newElements = CollectionsHelper.buildListFrom("d", "e", "f");
        int insertColumn = 0;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom("d", "e", "f");
        assertEquals(expected, actual);
    }

    @Test
    public void mixEmptyListIntoListAddToEmptyList() {
        List<String> originalList = CollectionsHelper.buildListFrom();
        List<String> newElements = CollectionsHelper.buildListFrom();
        int insertColumn = 0;
        List<String> actual = CollectionsHelper.mixIntoList(originalList, newElements, insertColumn);
        List<String> expected = CollectionsHelper.buildListFrom();
        assertEquals(expected, actual);
    }

    @Test
    public void createEmptyStringListZeroElements() {
        int size = 0;
        List<String> list = CollectionsHelper.createEmptyStringList(size);
        assertEquals(size, list.size());
        for (String element : list) {
            assertEquals("", element);
        }
    }

    @Test
    public void createEmptyStringListOneElement() {
        int size = 1;
        List<String> list = CollectionsHelper.createEmptyStringList(size);
        assertEquals(size, list.size());
        for (String element : list) {
            assertEquals("", element);
        }
    }

    @Test
    public void createEmptyStringListFiveElements() {
        int size = 5;
        List<String> list = CollectionsHelper.createEmptyStringList(size);
        assertEquals(size, list.size());
        for (String element : list) {
            assertEquals("", element);
        }
    }

    @Test
    public void createEmptyStringListNineteenNineElements() {
        int size = 99;
        List<String> list = CollectionsHelper.createEmptyStringList(size);
        assertEquals(size, list.size());
        for (String element : list) {
            assertEquals("", element);
        }
    }

    @Test
    public void createEmptyStringListNegativeNumberOfElements() {
        int size = -1;
        List<String> list = CollectionsHelper.createEmptyStringList(size);
        assertEquals(0, list.size());
    }

    @Test
    public void getFirstMapKeyByValueNormalCase() {
        Map<Integer, String> map = new HashMap<>();
        map.put(1, "Apfel");
        map.put(2, "Birne");
        map.put(3, "Apfel");
        map.put(4, "Orange");

        int notFoundIndex = -1;
        int actual = CollectionsHelper.getFirstMapKeyByValue(map, "Apfel", notFoundIndex);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void getFirstMapKeyByValueNormalCaseManyIndices() {
        Map<Integer, String> map = new HashMap<>();
        map.put(11, "Apfel");
        map.put(2, "Birne");
        map.put(3, "Apfel");
        map.put(17, "Apfel");
        map.put(1155, "Apfel");
        map.put(4, "Orange");
        map.put(1163, "Apfel");
        map.put(1, "Apfel");

        int notFoundIndex = -1;
        int actual = CollectionsHelper.getFirstMapKeyByValue(map, "Apfel", notFoundIndex);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void getFirstMapKeyByValueNotFound() {
        Map<Integer, String> map = new HashMap<>();
        map.put(1, "Apfel");
        map.put(2, "Birne");
        map.put(3, "Apfel");
        map.put(4, "Orange");

        int notFoundIndex = -1;
        int actual = CollectionsHelper.getFirstMapKeyByValue(map, "Zitrone", notFoundIndex);
        int expected = notFoundIndex;
        assertEquals(expected, actual);
    }

    @Test
    public void getFirstMapKeyByValueNotFoundInEmptyMap() {
        Map<Integer, String> map = new HashMap<>();

        int notFoundIndex = -1;
        int actual = CollectionsHelper.getFirstMapKeyByValue(map, "Apfel", notFoundIndex);
        int expected = notFoundIndex;
        assertEquals(expected, actual);
    }

    @Test
    public void getFirstMapKeyByValueNormalCaseWithDates() {
        Map<Integer, ImmutualDate> map = new HashMap<>();
        map.put(1, new ImmutualDate("11.02.1967"));
        map.put(2, new ImmutualDate("04.11.2021"));
        map.put(3, new ImmutualDate("02.03.2020"));
        map.put(4, new ImmutualDate("24.12.2021"));

        int notFoundIndex = -1;
        ImmutualDate searchValue = new ImmutualDate("04.11.2021");
        int actual = CollectionsHelper.getFirstMapKeyByValue(map, searchValue, notFoundIndex);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void getLastListIndex() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        assertEquals(2, CollectionsHelper.getLastListIndex(list));
    }

    @Test
    public void getLastListIndexOfListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertEquals(0, CollectionsHelper.getLastListIndex(list));
    }

    @Test
    public void getLastListIndexOfEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        assertEquals(-1, CollectionsHelper.getLastListIndex(list));
    }

    @Test
    public void getLastListMember() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        assertEquals("c", CollectionsHelper.getLastListMember(list));
    }

    @Test
    public void getLastListMemberOfListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertEquals("a", CollectionsHelper.getLastListMember(list));
    }

    @Test (expected = IllegalArgumentException.class)
    public void getLastListMemberOfEmptyListAndFail() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.getLastListMember(list);
    }

    @Test
    public void getLastListMemberWithReturnValueInCaseOfEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom("a", "b", "c");
        assertEquals("c", CollectionsHelper.getLastListMember(list, "list is empty"));
    }

    @Test
    public void getLastListMemberWithReturnValueInCaseOfEmptyListOfListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("a");
        assertEquals("a", CollectionsHelper.getLastListMember(list, "list is empty"));
    }

    @Test
    public void getLastListMemberWithReturnValueInCaseOfEmptyListOfEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        assertEquals("list is empty", CollectionsHelper.getLastListMember(list, "list is empty"));
    }

    @Test
    public void containedMoreThanOnce() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");

        assertTrue(CollectionsHelper.containedMoreThanOnce(list, "Apfel"));
    }

    @Test
    public void containedMoreThanOnceFalseIfOnlyOneTimeInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");

        assertFalse(CollectionsHelper.containedMoreThanOnce(list, "Birne"));
    }

    @Test
    public void containedMoreThanOnceFalseIfNotInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");

        assertFalse(CollectionsHelper.containedMoreThanOnce(list, "Banane"));
    }

    @Test
    public void containedThreeTines() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");
        list.add("Apfel");
        list.add("Orange");

        assertTrue(CollectionsHelper.containedMinNTines(list, "Apfel", 3));
    }

    @Test
    public void containedThreeTinesFalseIfOnlyTwoTimesInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");
        list.add("Apfel");
        list.add("Orange");

        assertFalse(CollectionsHelper.containedMinNTines(list, "Orange", 3));
    }

    @Test
    public void containedThreeTinesFalseIfOnlyOneTimeInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");
        list.add("Apfel");
        list.add("Orange");

        assertFalse(CollectionsHelper.containedMinNTines(list, "Birne", 3));
    }

    @Test
    public void containedThreeTinesFalseIfNotInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");
        list.add("Apfel");
        list.add("Orange");

        assertFalse(CollectionsHelper.containedMinNTines(list, "Banane", 3));
    }

    @Test
    public void countElementInList() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Apfel");
        list.add("Orange");
        list.add("Apfel");
        list.add("Orange");

        assertEquals(0, CollectionsHelper.countElementInList(list, "Banane"));
        assertEquals(1, CollectionsHelper.countElementInList(list, "Birne"));
        assertEquals(2, CollectionsHelper.countElementInList(list, "Orange"));
        assertEquals(3, CollectionsHelper.countElementInList(list, "Apfel"));
    }

    @Test
    public void countElementInEmptyList() {
        List<String> list = new ArrayList<>();
        assertEquals(0, CollectionsHelper.countElementInList(list, "Banane"));
        assertEquals(0, CollectionsHelper.countElementInList(list, "Birne"));
        assertEquals(0, CollectionsHelper.countElementInList(list, "Orange"));
        assertEquals(0, CollectionsHelper.countElementInList(list, "Apfel"));
    }

    @Test
    public void toEnumerationEmpty() {
        List<String> list = new ArrayList<>();

        assertEquals("", CollectionsHelper.toEnumeration(list, "und"));
    }

    @Test
    public void toEnumerationOneElement() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");

        assertEquals("'Apfel'", CollectionsHelper.toEnumeration(list, "und"));
    }

    @Test
    public void toEnumerationTwoElements() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");

        assertEquals("'Apfel' und 'Birne'", CollectionsHelper.toEnumeration(list, "und"));
    }

    @Test
    public void toEnumerationThreeElements() {
        List<String> list = new ArrayList<>();
        list.add("Apfel");
        list.add("Birne");
        list.add("Orange");

        assertEquals("'Apfel', 'Birne' und 'Orange'", CollectionsHelper.toEnumeration(list, "und"));
    }

    @Test
    public void toEnumeration123() {
        List<String> list = new ArrayList<>();
        list.add("1");
        list.add("2");
        list.add("3");

        assertEquals("'1', '2' oder '3'", CollectionsHelper.toEnumeration(list, "oder"));
    }

    @Test
    public void toEnumeration12345() {
        List<String> list = new ArrayList<>();
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        assertEquals("'1', '2', '3', '4' oder '5'", CollectionsHelper.toEnumeration(list, "oder"));
    }

    @Test
    public void sublist() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 2;
        int endIndex = 4;

        List<String> sublist = CollectionsHelper.sublist(list, startIndex, endIndex);
        assertEquals(2, sublist.size());
        assertEquals("2", sublist.get(0));
        assertEquals("3", sublist.get(1));
    }

    @Test
    public void sublistAtStart() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 0;
        int endIndex = 3;

        List<String> sublist = CollectionsHelper.sublist(list, startIndex, endIndex);
        assertEquals(3, sublist.size());
        assertEquals("0", sublist.get(0));
        assertEquals("1", sublist.get(1));
        assertEquals("2", sublist.get(2));
    }

    @Test
    public void sublistAtEnd() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 3;
        int endIndex = 6;

        List<String> sublist = CollectionsHelper.sublist(list, startIndex, endIndex);
        assertEquals(3, sublist.size());
        assertEquals("3", sublist.get(0));
        assertEquals("4", sublist.get(1));
        assertEquals("5", sublist.get(2));
    }

    @Test
    public void sublistEndIndexTooBig() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 3;
        int endIndex = 7;

        assertThrows(IndexOutOfBoundsException.class, () ->
            CollectionsHelper.sublist(list, startIndex, endIndex));
    }

    @Test
    public void sublistStartIndexTooSmall() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = -1;
        int endIndex = 2;

        assertThrows(IndexOutOfBoundsException.class, () ->
            CollectionsHelper.sublist(list, startIndex, endIndex));
    }

    @Test
    public void sublistWithoutFirstAndLastLine() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        List<String> sublist = CollectionsHelper.sublistWithoutFirstAndLastLine(list);
        assertEquals(4, sublist.size());
        assertEquals("1", sublist.get(0));
        assertEquals("2", sublist.get(1));
        assertEquals("3", sublist.get(2));
        assertEquals("4", sublist.get(3));
    }

    @Test
    public void sublistWithoutFirstAndLastLineListWithOneLine() {
        List<String> list = new ArrayList<>();
        list.add("0");

        List<String> sublist = CollectionsHelper.sublistWithoutFirstAndLastLine(list);
        assertEquals(0, sublist.size());
    }

    @Test
    public void sublistWithoutFirstAndLastLineEmptyList() {
        List<String> list = new ArrayList<>();

        assertThrows(RuntimeException.class, () ->
                CollectionsHelper.sublistWithoutFirstAndLastLine(list));
    }

    @Test
    public void sublistAtEndWithoutEndIndex() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 3;

        List<String> sublist = CollectionsHelper.sublist(list, startIndex);
        assertEquals(3, sublist.size());
        assertEquals("3", sublist.get(0));
        assertEquals("4", sublist.get(1));
        assertEquals("5", sublist.get(2));
    }

    @Test
    public void sublistStartAndEndEqual() {
        List<String> list = new ArrayList<>();
        list.add("0");
        list.add("1");
        list.add("2");
        list.add("3");
        list.add("4");
        list.add("5");

        int startIndex = 3;
        int endIndex = 3;

        List<String> sublist = CollectionsHelper.sublist(list, startIndex, endIndex);
        assertEquals(0, sublist.size());
    }

    @Test
    public void determineIndexOfFirstElementContaining() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String search = "Vier";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(4, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingNotContained() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String search = "Zweiundwzanzig";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(-1, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingEmptySearch() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String search = "";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(0, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingEmptyList() {
        List<String> list = new ArrayList<>();

        String search = "Vier";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(-1, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingEmptyListAndEmptySearch() {
        List<String> list = new ArrayList<>();

        String search = "";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(-1, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingEmptySearchMultipleValue() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String search = "Vier";

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search);
        assertEquals(4, index);
    }

    @Test
    public void determineIndexOfFirstElementContainingWithStartIndex() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Null");

        String search = "Null";
        int startIndex = 2;

        int index = CollectionsHelper.determineIndexOfFirstElementContaining(list, search,
                startIndex);
        assertEquals(5, index);
    }

    @Test
    public void addSpaceToAllListMembers() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        List<String> results = CollectionsHelper.addSpaceToAllListMembers(list);
        assertEquals(list.size(), results.size());
        for (int index = 0; index < list.size(); ++index) {
            String element = list.get(index);
            String result = results.get(index);
            assertEquals(element + " ", result);
        }
    }

    @Test
    public void addToAllListMembers() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String additional = "_foo";

        List<String> results = CollectionsHelper.addToAllListMembers(list, additional);
        assertEquals(list.size(), results.size());
        assertEquals("Null_foo", results.get(0));
        assertEquals("Eins_foo", results.get(1));
        assertEquals("Zwei_foo", results.get(2));
        assertEquals("Drei_foo", results.get(3));
        assertEquals("Vier_foo", results.get(4));
        assertEquals("Fünf_foo", results.get(5));
    }

    @Test
    public void addBeforeAllListMembers() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        String intro = "foo_";

        List<String> results = CollectionsHelper.addBeforeAllListMembers(list, intro);
        assertEquals(list.size(), results.size());
        assertEquals("foo_Null", results.get(0));
        assertEquals("foo_Eins", results.get(1));
        assertEquals("foo_Zwei", results.get(2));
        assertEquals("foo_Drei", results.get(3));
        assertEquals("foo_Vier", results.get(4));
        assertEquals("foo_Fünf", results.get(5));
    }

    @Test
    public void addAllBeforeAllListMembers() {
        List<String> list = new ArrayList<>();
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");

        List<String> intros = new ArrayList<>();
        intros.add("foo_");
        intros.add("bar_");
        intros.add("baz_");

        List<String> results = CollectionsHelper.addAllBeforeAllListMembers(list, intros);
        assertEquals(list.size() * intros.size(), results.size());
        assertEquals("foo_Eins", results.get(0));
        assertEquals("foo_Zwei", results.get(1));
        assertEquals("foo_Drei", results.get(2));
        assertEquals("bar_Eins", results.get(3));
        assertEquals("bar_Zwei", results.get(4));
        assertEquals("bar_Drei", results.get(5));
        assertEquals("baz_Eins", results.get(6));
        assertEquals("baz_Zwei", results.get(7));
        assertEquals("baz_Drei", results.get(8));
    }

    @Test
    public void containedInAnyListElement() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins"); // E-in-s
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        assertTrue(CollectionsHelper.containedInAnyListElement("in", list));
    }

    @Test
    public void containedInAnyListElementNotContained() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        assertFalse(CollectionsHelper.containedInAnyListElement("out", list));
    }

    @Test
    public void containedInAnyListElementNotContainedEmptyList() {
        List<String> list = new ArrayList<>();

        assertFalse(CollectionsHelper.containedInAnyListElement("in", list));
    }

    @Test
    public void containedInAnyListElementEmptyStringNotContainedInEmptyList() {
        List<String> list = new ArrayList<>();

        assertFalse(CollectionsHelper.containedInAnyListElement("", list));
    }

    @Test
    public void containedInAnyListElementEmptyStringContainedInNotEmptyList() {
        List<String> list = new ArrayList<>();
        list.add("Null");
        list.add("Eins");
        list.add("Zwei");
        list.add("Drei");
        list.add("Vier");
        list.add("Fünf");

        assertTrue(CollectionsHelper.containedInAnyListElement("", list));
    }

    @Test
    public void sortNumberStringsAsInteger() {
        List<String> list = new ArrayList<>();
        list.add("1");
        list.add("10");
        list.add("1000");
        list.add("101");
        list.add("202");
        list.add("2");
        list.add("20");

        CollectionsHelper.sortNumberStringsAsInteger(list);

        List<String> expected = new ArrayList<>();
        expected.add("1");
        expected.add("2");
        expected.add("10");
        expected.add("20");
        expected.add("101");
        expected.add("202");
        expected.add("1000");

        assertEquals(expected, list);
    }

    @Test
    public void addEmptyLines() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel");
        list.add("Birnen");
        list.add("Datteln");
        list.add("Feigen");
        list.add("Guaven");
        list.add("Kakis");
        list.add("Limonen");

        CollectionsHelper.addEmptyLines(list);

        List<String> expected = new ArrayList<>();
        expected.add("Äpfel");
        expected.add("");
        expected.add("Birnen");
        expected.add("");
        expected.add("Datteln");
        expected.add("");
        expected.add("Feigen");
        expected.add("");
        expected.add("Guaven");
        expected.add("");
        expected.add("Kakis");
        expected.add("");
        expected.add("Limonen");

        assertEquals(expected, list);
    }

    @Test
    public void addEmptyLinesOneElement() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel");

        CollectionsHelper.addEmptyLines(list);

        List<String> expected = new ArrayList<>();
        expected.add("Äpfel");

        assertEquals(expected, list);
    }

    @Test
    public void addEmptyLinesEmptyList() {
        List<String> list = new ArrayList<>();

        CollectionsHelper.addEmptyLines(list);

        List<String> expected = new ArrayList<>();

        assertEquals(expected, list);
    }

    @Test
    public void createFlatListFromDictionary() {
        List<List<String>> dictionary = new ArrayList<>();
        dictionary.add(CollectionsHelper.buildListFrom("A", "a", "ah"));
        dictionary.add(CollectionsHelper.buildListFrom("B", "b"));
        dictionary.add(CollectionsHelper.buildListFrom("C", "c"));

        @SuppressWarnings("deprecation")
        List<String> list = CollectionsHelper.createFlatListFromDictionary(dictionary);
        assertEquals(7, list.size());
        assertEquals("A", list.get(0));
        assertEquals("a", list.get(1));
        assertEquals("ah", list.get(2));
        assertEquals("B", list.get(3));
        assertEquals("b", list.get(4));
        assertEquals("C", list.get(5));
        assertEquals("c", list.get(6));
    }

    @Test
    public void removeColonAtEnd() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel:");
        list.add("Birnen:");
        list.add("Obst: Äpfel und Birnen:");
        list.add("Datteln");
        list.add("Feigen");
        list.add("Guaven");
        list.add("Kakis");
        list.add("Limonen");

        CollectionsHelper.removeColonAtEnd(list);

        List<String> expected = new ArrayList<>();
        expected.add("Äpfel");
        expected.add("Birnen");
        expected.add("Obst: Äpfel und Birnen");
        expected.add("Datteln");
        expected.add("Feigen");
        expected.add("Guaven");
        expected.add("Kakis");
        expected.add("Limonen");

        assertEquals(expected, list);
    }

    @Test
    public void removeColonAtEndWithOutColon() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel");
        list.add("Birnen");
        list.add("Datteln");
        list.add("Feigen");
        list.add("Guaven");
        list.add("Kakis");
        list.add("Limonen");

        CollectionsHelper.removeColonAtEnd(list);

        List<String> expected = new ArrayList<>();
        expected.add("Äpfel");
        expected.add("Birnen");
        expected.add("Datteln");
        expected.add("Feigen");
        expected.add("Guaven");
        expected.add("Kakis");
        expected.add("Limonen");

        assertEquals(expected, list);
    }

    @Test
    public void withAndWithoutColon() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel");
        list.add("Birnen");
        list.add("Datteln");
        list.add("Feigen");
        list.add("Guaven");
        list.add("Kakis");
        list.add("Limonen");

        list = CollectionsHelper.withAndWithoutColon(list);

        List<String> expected = new ArrayList<>();
        expected.add("Äpfel:");
        expected.add("Äpfel");
        expected.add("Birnen:");
        expected.add("Birnen");
        expected.add("Datteln:");
        expected.add("Datteln");
        expected.add("Feigen:");
        expected.add("Feigen");
        expected.add("Guaven:");
        expected.add("Guaven");
        expected.add("Kakis:");
        expected.add("Kakis");
        expected.add("Limonen:");
        expected.add("Limonen");

        assertEquals(expected, list);
    }

    @Test
    public void withAndWithoutComma() {
        List<String> list = new ArrayList<>();
        list.add("Äpfel");
        list.add("Birnen");
        list.add("Datteln");
        list.add("Feigen");
        list.add("Guaven");
        list.add("Kakis");
        list.add("Limonen");

        list = CollectionsHelper.withAndWithoutComma(list);

        List<String> expected = new ArrayList<>();
        expected.add("; Äpfel");
        expected.add(", Äpfel");
        expected.add("Äpfel");
        expected.add("; Birnen");
        expected.add(", Birnen");
        expected.add("Birnen");
        expected.add("; Datteln");
        expected.add(", Datteln");
        expected.add("Datteln");
        expected.add("; Feigen");
        expected.add(", Feigen");
        expected.add("Feigen");
        expected.add("; Guaven");
        expected.add(", Guaven");
        expected.add("Guaven");
        expected.add("; Kakis");
        expected.add(", Kakis");
        expected.add("Kakis");
        expected.add("; Limonen");
        expected.add(", Limonen");
        expected.add("Limonen");

        assertEquals(expected, list);
    }

    @Test
    public void listOfLinesToListOfFieldLines() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "id\tobst\tgelagert",
                "1\tApfel\tgroße Porzellanschale",
                "2\tOrange\tgroße Glasschale",
                "3\tMandarine\tgroße Glasschale");

        List<List<String>> actualFieldLines = CollectionsHelper.listOfLinesToListOfFieldLines(lines);

        List<List<String>> expectedFieldLines = new ArrayList<>();
        List<String> titles = CollectionsHelper.buildListFrom(
                "id",
                "obst",
                "gelagert"
                );
        expectedFieldLines.add(titles);

        List<String> fields1 = CollectionsHelper.buildListFrom(
                "1",
                "Apfel",
                "große Porzellanschale"
                );
        expectedFieldLines.add(fields1);

        List<String> fields2 = CollectionsHelper.buildListFrom(
                "2",
                "Orange",
                "große Glasschale"
                );
        expectedFieldLines.add(fields2);

        List<String> fields3 = CollectionsHelper.buildListFrom(
                "3",
                "Mandarine",
                "große Glasschale"
                );
        expectedFieldLines.add(fields3);

        assertEquals(expectedFieldLines, actualFieldLines);
    }

    @Test
    public void listOfFieldLinesToListOfLines() {
        List<List<String>> fieldsList = new ArrayList<>();
        List<String> titles = CollectionsHelper.buildListFrom(
                "id",
                "obst",
                "gelagert"
                );
        fieldsList.add(titles);

        List<String> fields1 = CollectionsHelper.buildListFrom(
                "1",
                "Apfel",
                "große Porzellanschale"
                );
        fieldsList.add(fields1);

        List<String> fields2 = CollectionsHelper.buildListFrom(
                "2",
                "Orange",
                "große Glasschale"
                );
        fieldsList.add(fields2);

        List<String> fields3 = CollectionsHelper.buildListFrom(
                "3",
                "Mandarine",
                "große Glasschale"
                );
        fieldsList.add(fields3);

        List<String> list = CollectionsHelper.listOfFieldLinesToListOfLines(fieldsList);
        assertEquals(4, list.size());
        assertEquals("id\tobst\tgelagert", list.get(0));
        assertEquals("1\tApfel\tgroße Porzellanschale", list.get(1));
        assertEquals("2\tOrange\tgroße Glasschale", list.get(2));
        assertEquals("3\tMandarine\tgroße Glasschale", list.get(3));
    }

    @Test
    public void listOfFieldLinesToListOfLines2() {
        List<List<String>> fieldsList = new ArrayList<>();
        List<String> titles = CollectionsHelper.buildListFrom(
                "id",
                "obst",
                "gelagert"
                );
        fieldsList.add(titles);

        List<String> fields1 = CollectionsHelper.buildListFrom(
                "1",
                "Apfel",
                "große Porzellanschale"
                );
        fieldsList.add(fields1);

        List<String> fields2 = CollectionsHelper.buildListFrom(
                "2",
                "Orange",
                "große Glasschale"
                );
        fieldsList.add(fields2);

        List<String> fields3 = CollectionsHelper.buildListFrom(
                "3",
                "Mandarine",
                "große Glasschale"
                );
        fieldsList.add(fields3);

        List<String> actualLines = CollectionsHelper.listOfFieldLinesToListOfLines(fieldsList);

        List<String> expectedLines = CollectionsHelper.buildListFrom(
                "id\tobst\tgelagert",
                "1\tApfel\tgroße Porzellanschale",
                "2\tOrange\tgroße Glasschale",
                "3\tMandarine\tgroße Glasschale");
        assertEquals(expectedLines, actualLines);
    }

    @Test
    public void changeOnlySpacesToEmptyString() {
        List<String> list = CollectionsHelper.buildListFrom(
                "a",
                "  ",
                " b ",
                "",
                "    ",
                "c"
                );
        List<String> cleanedList =  CollectionsHelper.changeOnlySpacesToEmptyString(list);

        assertEquals(6, cleanedList.size());
        assertEquals("a", cleanedList.get(0));
        assertEquals("", cleanedList.get(1));
        assertEquals(" b ", cleanedList.get(2));
        assertEquals("", cleanedList.get(3));
        assertEquals("", cleanedList.get(4));
        assertEquals("c", cleanedList.get(5));
    }

    @Test
    public void switchPipesToSlashes() {
        List<String> list = CollectionsHelper.buildListFrom(
                "29122200001819" + "\t" + "Sellova Digital UG (haftungsbeschränkt)" + "\t"
                        + "28.12.2022" + "\t" + "Die Entwicklung und der Betrieb von "
                        + "Online-Plattformen sowie der damit verbundene Online-Handel mit "
                        + "digitalen und nicht-digitalen Produkten (bspw. Templates, "
                        + "Software-Tools, Video-Kurse und Computer-Hardware).",
                "29122200001822" + "\t" + "KLINIKEN ESSEN SÜD Evangelisches Krankenhaus "
                        + "Essen-Werden gemeinnützige GmbH" + "\t" + "23.12.2022" + "\t" + "Die "
                        + "Förderung des öffentlichen Gesundheitswesens und der öffentlichen "
                        + "Gesundheitspflege sowie die Förderung des Wohlfahrtswesens und die "
                        + "Unterstützung der evangelischen Kirche. Gegenstand der Gesellschaft "
                        + "ist die Verwirklichung der Aufgaben der Diakonie als Lebens- und "
                        + "Wesensäußerung der Evangelischen Kirche. Der Satzungszweck wird "
                        + "verwirklicht durch die Erbringung von Kooperationsleistungen in dem "
                        + "Bereich der Seelsorge durch Bereitstellung von seelsorgerlichem "
                        + "Personal und der Betreuung, Organisation und der Finanzierung der "
                        + "Dienste der Grünen Damen und Herren an andere steuerbegünstigte "
                        + "Körperschaften im Sinne von § 57 Abs. 3 Abgabenordnung, an mit "
                        + "Gesellschaft verbundene steuerbegünstigte Unternehmen, an die "
                        + "KEM | Evang. Kliniken Essen-Mitte gGmbH. Die Gesellschaft verwirklicht "
                        + "ihre steuerbegünstigten Zwecke ferner unter Einbeziehung von "
                        + "Kooperationsleistungen von anderen steuerbegünstigten Körperschaften "
                        + "im Sinne von § 57 Abs. 3 AO, nämlich durch Geschäftsführungs-, "
                        + "Management und Verwaltungsleistungen (insbesondere in den Bereichen "
                        + "Rechnungswesen, Personal, Controlling, technische Dienstleistung, "
                        + "Einkauf, Informationstechnologie, Hygienemanagement von mit der "
                        + "Gesellschaft verbundenen steuerbegünstigten Unternehmen, der "
                        + "KEM | Evang. Kliniken Essen-Mitte gGmbH.",
                "29122200001834" + "\t" + "Royal Dream Event GmbH" + "\t" + "23.12.2022" + "\t"
                        + "Die Vermietung einer Veranstaltungsstätte zur Durchführung von "
                        + "Hochzeiten, Kongressen, Seminaren, Partys, Konzerten, "
                        + "Folkloreveranstaltungen, Catering, Vermietung von Möbeln und "
                        + "Gastronomiegegenständen.",
                "29122200001851" + "\t" + "GE-Back GmbH" + "\t" + "27.12.2022" + "\t"
                        + "Betrieb einer Großbäckerei."
                );
        CollectionsHelper.switchPipesToSlashes(list);

        assertEquals(4, list.size());
        assertEquals("29122200001819" + "\t" + "Sellova Digital UG (haftungsbeschränkt)" + "\t"
                + "28.12.2022" + "\t" + "Die Entwicklung und der Betrieb von "
                + "Online-Plattformen sowie der damit verbundene Online-Handel mit "
                + "digitalen und nicht-digitalen Produkten (bspw. Templates, "
                + "Software-Tools, Video-Kurse und Computer-Hardware).",
                list.get(0));
        assertEquals("29122200001822" + "\t" + "KLINIKEN ESSEN SÜD Evangelisches Krankenhaus "
                + "Essen-Werden gemeinnützige GmbH" + "\t" + "23.12.2022" + "\t" + "Die "
                + "Förderung des öffentlichen Gesundheitswesens und der öffentlichen "
                + "Gesundheitspflege sowie die Förderung des Wohlfahrtswesens und die "
                + "Unterstützung der evangelischen Kirche. Gegenstand der Gesellschaft "
                + "ist die Verwirklichung der Aufgaben der Diakonie als Lebens- und "
                + "Wesensäußerung der Evangelischen Kirche. Der Satzungszweck wird "
                + "verwirklicht durch die Erbringung von Kooperationsleistungen in dem "
                + "Bereich der Seelsorge durch Bereitstellung von seelsorgerlichem "
                + "Personal und der Betreuung, Organisation und der Finanzierung der "
                + "Dienste der Grünen Damen und Herren an andere steuerbegünstigte "
                + "Körperschaften im Sinne von § 57 Abs. 3 Abgabenordnung, an mit "
                + "Gesellschaft verbundene steuerbegünstigte Unternehmen, an die "
                + "KEM / Evang. Kliniken Essen-Mitte gGmbH. Die Gesellschaft verwirklicht "    // <-
                + "ihre steuerbegünstigten Zwecke ferner unter Einbeziehung von "
                + "Kooperationsleistungen von anderen steuerbegünstigten Körperschaften "
                + "im Sinne von § 57 Abs. 3 AO, nämlich durch Geschäftsführungs-, "
                + "Management und Verwaltungsleistungen (insbesondere in den Bereichen "
                + "Rechnungswesen, Personal, Controlling, technische Dienstleistung, "
                + "Einkauf, Informationstechnologie, Hygienemanagement von mit der "
                + "Gesellschaft verbundenen steuerbegünstigten Unternehmen, der "
                + "KEM / Evang. Kliniken Essen-Mitte gGmbH.",                                  // <-
                list.get(1));
        assertEquals("29122200001834" + "\t" + "Royal Dream Event GmbH" + "\t" + "23.12.2022" + "\t"
                + "Die Vermietung einer Veranstaltungsstätte zur Durchführung von "
                + "Hochzeiten, Kongressen, Seminaren, Partys, Konzerten, "
                + "Folkloreveranstaltungen, Catering, Vermietung von Möbeln und "
                + "Gastronomiegegenständen.",
                list.get(2));
        assertEquals("29122200001851" + "\t" + "GE-Back GmbH" + "\t" + "27.12.2022" + "\t"
                + "Betrieb einer Großbäckerei.",
                list.get(3));
    }

    @Test
    public void replaceMultipleLines() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "a",
                "b1",
                "b2",
                "b3",
                "b4",
                "c"
                );
        List<String> linesToExchange = CollectionsHelper.buildListFrom(
                "b1",
                "b2",
                "b3",
                "b4"
                );
        List<String> newLines = CollectionsHelper.buildListFrom(
                "B1",
                "B2"
                );
        CollectionsHelper.replaceMultipleLines(lines, linesToExchange, newLines);

        assertEquals(4, lines.size());
        assertEquals("a", lines.get(0));
        assertEquals("B1", lines.get(1));
        assertEquals("B2", lines.get(2));
        assertEquals("c", lines.get(3));
    }

    @Test
    public void replaceMultipleLinesMultiple() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "a",
                "b1",
                "b2",
                "b3",
                "b4",
                "c",
                "b1",
                "b2",
                "b3",
                "b4",
                "d"
                );
        List<String> linesToExchange = CollectionsHelper.buildListFrom(
                "b1",
                "b2",
                "b3",
                "b4"
                );
        List<String> newLines = CollectionsHelper.buildListFrom(
                "B1",
                "B2"
                );
        CollectionsHelper.replaceMultipleLines(lines, linesToExchange, newLines);

        assertEquals(7, lines.size());
        assertEquals("a", lines.get(0));
        assertEquals("B1", lines.get(1));
        assertEquals("B2", lines.get(2));
        assertEquals("c", lines.get(3));
        assertEquals("B1", lines.get(4));
        assertEquals("B2", lines.get(5));
        assertEquals("d", lines.get(6));
    }

    @Test
    public void replaceMultipleLinesOverlapping() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "a",
                "b",
                "b",
                "b",
                "b",
                "b",
                "c"
                );
        List<String> linesToExchange = CollectionsHelper.buildListFrom(
                "b",
                "b",
                "b",
                "b"
                );
        List<String> newLines = CollectionsHelper.buildListFrom(
                "B1",
                "B2"
                );
        CollectionsHelper.replaceMultipleLines(lines, linesToExchange, newLines);

        assertEquals(5, lines.size());
        assertEquals("a", lines.get(0));
        assertEquals("B1", lines.get(1));
        assertEquals("B2", lines.get(2));
        assertEquals("b", lines.get(3));
        assertEquals("c", lines.get(4));
    }

    @Test
    public void removeCommaAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom(
                "a",
                "b,",
                "c,,",
                "d"
                );
        CollectionsHelper.removeCommaAtEnd(list);

        assertEquals(4, list.size());
        assertEquals("a", list.get(0));
        assertEquals("b", list.get(1));
        assertEquals("c,", list.get(2));
        assertEquals("d", list.get(3));
    }

    @Test
    public void determineFirstIndexStartingWith() {
        List<String> list = CollectionsHelper.buildListFrom(
                "a) foo",
                "b) bar",
                "b) bar 2",
                "c) baz",
                "d) foo bar",
                "c) baz 2"
                );
        assertEquals(0, CollectionsHelper.determineFirstIndexStartingWith(list, "a)"));
        assertEquals(1, CollectionsHelper.determineFirstIndexStartingWith(list, "b)"));
        assertEquals(3, CollectionsHelper.determineFirstIndexStartingWith(list, "c)"));
        assertEquals(4, CollectionsHelper.determineFirstIndexStartingWith(list, "d)"));
        assertEquals(-1, CollectionsHelper.determineFirstIndexStartingWith(list, "e)"));
    }

    @Test
    public void determineFirstIndexStartingWithAndGivenStart() {
        List<String> list = CollectionsHelper.buildListFrom(
                "a) foo",
                "b) bar",
                "b) bar 2",
                "c) baz",
                "d) foo bar",
                "c) baz 2"
                );
        assertEquals(0, CollectionsHelper.determineFirstIndexStartingWith(list, "a)", 0));
        assertEquals(1, CollectionsHelper.determineFirstIndexStartingWith(list, "b)", 0));
        assertEquals(2, CollectionsHelper.determineFirstIndexStartingWith(list, "b)", 2));
        assertEquals(3, CollectionsHelper.determineFirstIndexStartingWith(list, "c)", 0));
        assertEquals(5, CollectionsHelper.determineFirstIndexStartingWith(list, "c)", 4));
        assertEquals(-1, CollectionsHelper.determineFirstIndexStartingWith(list, "c)", 6));
        assertEquals(4, CollectionsHelper.determineFirstIndexStartingWith(list, "d)", 0));
        assertEquals(-1, CollectionsHelper.determineFirstIndexStartingWith(list, "e)", 0));
    }

    @Test
    public void determineAllIndicesStartingWith() {
        List<String> list = CollectionsHelper.buildListFrom(
                "a) foo",
                "b) bar",
                "b) bar 2",
                "c) baz",
                "d) foo bar",
                "c) baz 2"
                );
        List<Integer> indicesA = CollectionsHelper.determineAllIndicesStartingWith(list, "a)");
        List<Integer> indicesB = CollectionsHelper.determineAllIndicesStartingWith(list, "b)");
        List<Integer> indicesC = CollectionsHelper.determineAllIndicesStartingWith(list, "c)");
        List<Integer> indicesD = CollectionsHelper.determineAllIndicesStartingWith(list, "d)");
        List<Integer> indicesE = CollectionsHelper.determineAllIndicesStartingWith(list, "e)");

        assertEquals(1, indicesA.size());
        assertEquals(Integer.valueOf(0), indicesA.get(0));

        assertEquals(2, indicesB.size());
        assertEquals(Integer.valueOf(1), indicesB.get(0));
        assertEquals(Integer.valueOf(2), indicesB.get(1));

        assertEquals(2, indicesC.size());
        assertEquals(Integer.valueOf(3), indicesC.get(0));
        assertEquals(Integer.valueOf(5), indicesC.get(1));

        assertEquals(1, indicesD.size());
        assertEquals(Integer.valueOf(4), indicesD.get(0));

        assertEquals(0, indicesE.size());
    }

    @Test
    public void createSortedKeysForStringIntegerMapByNumberDescanding() {
        Map<String, Integer> map = new HashMap<>();
        map.put("fünfmal", 5);
        map.put("dreimal", 3);
        map.put("achtmal zwei", 8);
        map.put("einmal", 1);
        map.put("achtmal eins", 8);
        map.put("neunmal", 9);
        map.put("zweimal", 2);

        List<String> sortedKeys = CollectionsHelper
                .createSortedKeysForStringIntegerMapByNumberDescanding(map);

        assertEquals(map.size(), sortedKeys.size());
        assertEquals("neunmal", sortedKeys.get(0));
        assertEquals("achtmal eins", sortedKeys.get(1));
        assertEquals("achtmal zwei", sortedKeys.get(2));
        assertEquals("fünfmal", sortedKeys.get(3));
        assertEquals("dreimal", sortedKeys.get(4));
        assertEquals("zweimal", sortedKeys.get(5));
        assertEquals("einmal", sortedKeys.get(6));
    }

    @Test
    public void appendCommaPointEtc() {
        List<String> list = CollectionsHelper.buildListFrom(
                "foo",
                "bar",
                "baz"
                );
        List<String> actual = CollectionsHelper.appendCommaPointEtc(list);
        List<String> expected = CollectionsHelper.buildListFrom(
                "foo,",
                "foo.",
                "foo:",
                "foo;",
                "foo",
                "bar,",
                "bar.",
                "bar:",
                "bar;",
                "bar",
                "baz,",
                "baz.",
                "baz:",
                "baz;",
                "baz"
                );
        assertEquals(expected, actual);
    }

    @Test
    public void switchListElements() {
        List<String> list = CollectionsHelper.buildListFrom(
                "foo",
                "bar",
                "baz"
                );
        CollectionsHelper.switchListElements(list, 0, 1);
        List<String> expected = CollectionsHelper.buildListFrom(
                "bar",
                "foo",
                "baz"
                );
        assertEquals(expected, list);
    }

    @Test
    public void switchListElements2() {
        List<String> list = CollectionsHelper.buildListFrom(
                "foo",
                "bar",
                "baz"
                );
        CollectionsHelper.switchListElements(list, 1, 0);
        List<String> expected = CollectionsHelper.buildListFrom(
                "bar",
                "foo",
                "baz"
                );
        assertEquals(expected, list);
    }

    @Test
    public void toBareFilenames() {
        List<String> list = CollectionsHelper.buildListFrom(
                "c:\\foo\\bar\\baz1.txt",
                "d:/foo/bar/baz2.txt",
                "baz3.txt"
                );
        List<String> actual = CollectionsHelper.toBareFilenames(list);
        List<String> expected = CollectionsHelper.buildListFrom(
                "baz1.txt",
                "baz2.txt",
                "baz3.txt"
                );
        assertEquals(expected, actual);
    }

    @Test
    public void createCartesianProduct() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "Tisch in",
                "Stuhl in",
                "Schrank in"
                );
        String between = " der Farbe ";
        List<String> list2 = CollectionsHelper.buildListFrom(
                "rot",
                "grün"
                );
        List<String> actual = CollectionsHelper.createCartesianProduct(list1, between, list2);
        List<String> expected = CollectionsHelper.buildListFrom(
                "Tisch in der Farbe rot",
                "Tisch in der Farbe grün",
                "Stuhl in der Farbe rot",
                "Stuhl in der Farbe grün",
                "Schrank in der Farbe rot",
                "Schrank in der Farbe grün"
                );
        assertEquals(expected, actual);
    }

    @Test
    public void checkSizes() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "ROT",
                "GRÜN"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        CollectionsHelper.checkSizes(list1, list2);
    }

    @Test
    public void checkSizesFail() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "ROT",
                "GRÜN",
                "BLAU"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        assertThrows(RuntimeException.class, () ->
                CollectionsHelper.checkSizes(list1, list2));
    }

    @Test
    public void checkAllElementsOfFirstAreInSecond() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        CollectionsHelper.checkAllElementsOfFirstAreInSecond(list1, list2);
    }

    @Test
    public void checkAllElementsOfFirstAreInSecondBigger() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "blau",
                "rot"
                );
        CollectionsHelper.checkAllElementsOfFirstAreInSecond(list1, list2);
    }

    @Test
    public void checkAllElementsOfFirstAreInSecondFail() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "blau",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot",
                "gelb"
                );
        assertThrows(RuntimeException.class, () ->
                CollectionsHelper.checkAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void checkAllElementsOfFirstAreInSecondSmallerFail() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "blau",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        assertThrows(RuntimeException.class, () ->
                CollectionsHelper.checkAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void areAllElementsOfFirstAreInSecond() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        assertTrue(CollectionsHelper.areAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void areAllElementsOfFirstAreInSecondBigger() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "blau",
                "rot"
                );
        assertTrue(CollectionsHelper.areAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void areAllElementsOfFirstAreInSecondFail() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "blau",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot",
                "gelb"
                );
        assertFalse(CollectionsHelper.areAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void areAllElementsOfFirstAreInSecondSmallerFail() {
        List<String> list1 = CollectionsHelper.buildListFrom(
                "rot",
                "blau",
                "grün"
                );
        List<String> list2 = CollectionsHelper.buildListFrom(
                "grün",
                "rot"
                );
        assertFalse(CollectionsHelper.areAllElementsOfFirstAreInSecond(list1, list2));
    }

    @Test
    public void addToMapWithQuantities() {
        Map<String, Integer> mapWithQuantity = new HashMap<>();
        mapWithQuantity.put("foo", 17);
        mapWithQuantity.put("bar",  3);
        mapWithQuantity.put("baz", 32);
        assertEquals(3, mapWithQuantity.size());

        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "foo");

        assertEquals(3, mapWithQuantity.size());
        assertEquals(Integer.valueOf(18), mapWithQuantity.get("foo"));
        assertEquals(Integer.valueOf( 3), mapWithQuantity.get("bar"));
        assertEquals(Integer.valueOf(32), mapWithQuantity.get("baz"));
    }

    @Test
    public void addToMapWithQuantities2() {
        Map<String, Integer> mapWithQuantity = new HashMap<>();
        mapWithQuantity.put("foo", 17);
        mapWithQuantity.put("bar",  3);
        mapWithQuantity.put("baz", 32);
        assertEquals(3, mapWithQuantity.size());

        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "foo");
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "foo");
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "bar");
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "baz");
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "baz");
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "baz");

        assertEquals(3, mapWithQuantity.size());
        assertEquals(Integer.valueOf(19), mapWithQuantity.get("foo"));
        assertEquals(Integer.valueOf( 4), mapWithQuantity.get("bar"));
        assertEquals(Integer.valueOf(35), mapWithQuantity.get("baz"));
    }

    @Test
    public void addToMapWithQuantities3() {
        Map<String, Integer> mapWithQuantity = new HashMap<>();
        mapWithQuantity.put("foo", 17);
        mapWithQuantity.put("bar",  3);
        mapWithQuantity.put("baz", 32);
        assertEquals(3, mapWithQuantity.size());

        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "foo", 2);
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "bar", 1);
        CollectionsHelper.addToMapWithQuantities(mapWithQuantity, "baz", 3);

        assertEquals(3, mapWithQuantity.size());
        assertEquals(Integer.valueOf(19), mapWithQuantity.get("foo"));
        assertEquals(Integer.valueOf( 4), mapWithQuantity.get("bar"));
        assertEquals(Integer.valueOf(35), mapWithQuantity.get("baz"));
    }

    @Test
    public void removeEntriesStartingWithRemovingContainedStart() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "_blau",
                "violett"
                );
        assertEquals(4, list.size());

        String entryStart = "_";

        CollectionsHelper.removeEntriesStartingWith(list, entryStart);
        assertEquals(3, list.size());
        assertEquals("rot", list.get(0));
        assertEquals("grün", list.get(1));
        assertEquals("violett", list.get(2));
    }

    @Test
    public void removeEntriesStartingWithRemovingNotContainedStart() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "_blau",
                "violett"
                );
        assertEquals(4, list.size());

        String entryStart = "#";

        CollectionsHelper.removeEntriesStartingWith(list, entryStart);
        assertEquals(4, list.size());
        assertEquals("rot", list.get(0));
        assertEquals("grün", list.get(1));
        assertEquals("_blau", list.get(2));
        assertEquals("violett", list.get(3));
    }

    @Test
    public void removeEntriesStartingWithEmptyStart() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "blau",
                "violett"
                );
        assertEquals(4, list.size());

        String entryStart = "";

        CollectionsHelper.removeEntriesStartingWith(list, entryStart);
        assertEquals(0, list.size());
    }

    @Test
    public void createStringListFromIntegerList() {
        List<Integer> numberList = new ArrayList<>();
        numberList.add(1000);
        numberList.add(2000);
        numberList.add(3000);
        numberList.add(4000);

        List<String> list = CollectionsHelper.createStringListFromIntegerList(numberList);
        assertEquals(4, list.size());
        assertEquals("1.000", list.get(0));
        assertEquals("2.000", list.get(1));
        assertEquals("3.000", list.get(2));
        assertEquals("4.000", list.get(3));
    }

    @Test
    public void removeFirstAndLastElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "blau",
                "violett"
                );
        CollectionsHelper.removeFirstAndLastElement(list);
        assertEquals(2, list.size());
        assertEquals("grün", list.get(0));
        assertEquals("blau", list.get(1));
    }

    @Test
    public void removeFirstAndLastElementEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom(
                );
        CollectionsHelper.removeFirstAndLastElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeFirstAndLastElementListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "violett"
                );
        CollectionsHelper.removeFirstAndLastElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeFirstAndLastElementListWithTwoElements() {
        List<String> list = CollectionsHelper.buildListFrom(
                "blau",
                "violett"
                );
        CollectionsHelper.removeFirstAndLastElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeFirstElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "blau",
                "violett"
                );
        CollectionsHelper.removeFirstElement(list);
        assertEquals(3, list.size());
        assertEquals("grün", list.get(0));
        assertEquals("blau", list.get(1));
        assertEquals("violett", list.get(2));
    }

    @Test
    public void removeFirstElementEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom(
                );
        CollectionsHelper.removeFirstElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeFirstElementListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "violett"
                );
        CollectionsHelper.removeFirstElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeLastElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "rot",
                "grün",
                "blau",
                "violett"
                );
        CollectionsHelper.removeLastElement(list);
        assertEquals(3, list.size());
        assertEquals("rot", list.get(0));
        assertEquals("grün", list.get(1));
        assertEquals("blau", list.get(2));
    }

    @Test
    public void removeElementEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom(
                );
        CollectionsHelper.removeLastElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void removeLastElementListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom(
                "violett"
                );
        CollectionsHelper.removeLastElement(list);
        assertEquals(0, list.size());
    }

    @Test
    public void countLinesStartingWith() {
        List<String> list = CollectionsHelper.buildListFrom(
                "blau",
                "violett",
                "blaugrau",
                "orange",
                "blaugrün",
                "gelb oder blau"
                );
        int blauCount = CollectionsHelper.countLinesStartingWith(list, "blau");
        assertEquals(3, blauCount);
        int gruenCount = CollectionsHelper.countLinesStartingWith(list, "grün");
        assertEquals(0, gruenCount);
    }

    @Test
    public void areAllElementsBlankNotBlank() {
        List<String> list = CollectionsHelper.buildListFrom(
                "",
                "blau",
                "",
                "violett",
                "blaugrau",
                "    ",
                "orange",
                "blaugrün",
                "\t",
                "gelb oder blau"
                );
        assertFalse(CollectionsHelper.areAllElementsBlank(list));
    }

    @Test
    public void areAllElementsBlank() {
        List<String> list = CollectionsHelper.buildListFrom(
                "",
                "    ",
                "\t",
                "",
                "        ",
                ""
                );
        assertTrue(CollectionsHelper.areAllElementsBlank(list));
    }

    @Test
    public void areAllElementsBlankEmpty() {
        List<String> list = CollectionsHelper.buildListFrom(
                );
        assertTrue(CollectionsHelper.areAllElementsBlank(list));
    }

    @Test
    public void sortBetterAlphabethicalEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.sortBetterAlphabethical(list);
        assertEquals(0, list.size());
    }

    @Test
    public void sortBetterAlphabethicalListWithOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("Auto");
        CollectionsHelper.sortBetterAlphabethical(list);
        assertEquals(1, list.size());
        assertEquals("Auto", list.get(0));
    }

    @Test
    public void sortBetterAlphabethicalNormalCase() {
        List<String> list = CollectionsHelper.buildListFrom(
                "Auto", "autonom", "Automat", "Banane", "Azur");
        CollectionsHelper.sortBetterAlphabethical(list);
        assertEquals(5, list.size());
        assertEquals("Auto", list.get(0));
        assertEquals("Automat", list.get(1));
        assertEquals("autonom", list.get(2));
        assertEquals("Azur", list.get(3));
        assertEquals("Banane", list.get(4));
    }

    @Test
    public void sortBetterAlphabethicalUmlaute() {
        List<String> list = CollectionsHelper.buildListFrom(
                "Obst", "Ökonom", "Ordnung", "öffentlich");
        CollectionsHelper.sortBetterAlphabethical(list);
        assertEquals(4, list.size());
        assertEquals("Obst", list.get(0));
        assertEquals("öffentlich", list.get(1));
        assertEquals("Ökonom", list.get(2));
        assertEquals("Ordnung", list.get(3));
    }

    @Test
    public void isSortedIntegerListEmptyList() {
        List<Integer> numbers = CollectionsHelper.buildListFrom();
        assertTrue(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListOneLement() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(17);
        assertTrue(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListTwoElementsCorrect() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(1, 2);
        assertTrue(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListTwoElementsWrong() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(2, 1);
        assertFalse(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListTwoElementsEqual() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(1, 1);
        assertFalse(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListThreeElementsCorrect() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(1, 2, 5);
        assertTrue(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListThreeElementsWrong() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(2, 1, 5);
        assertFalse(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void isSortedIntegerListThreeElementsEqual() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(1, 1, 5);
        assertFalse(CollectionsHelper.isSortedIntegerList(numbers));
    }

    @Test
    public void removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers1() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2, 3);
        List<Integer> list2 = CollectionsHelper.buildListFrom(1);
        List<Integer> list3 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list4 = CollectionsHelper.buildListFrom(4);
        List<Integer> list5 = CollectionsHelper.buildListFrom(4, 5, 6);
        List<Integer> list6 = CollectionsHelper.buildListFrom(4, 5);
        List<List<Integer>> listOfLists = CollectionsHelper.buildListFrom(
                list1,
                list2,
                list3,
                list4,
                list5,
                list6
                );
        CollectionsHelper.removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers(
                listOfLists);
        assertEquals(2, listOfLists.size());
        assertEquals(list1, listOfLists.get(0));
        assertEquals(list5, listOfLists.get(1));
    }

    @Test
    public void removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers2() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2, 3);
        List<Integer> list2 = CollectionsHelper.buildListFrom(1);
        List<Integer> list3 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list4 = CollectionsHelper.buildListFrom(3, 1);
        List<Integer> list5 = CollectionsHelper.buildListFrom(4);
        List<Integer> list6 = CollectionsHelper.buildListFrom(4, 5, 6);
        List<Integer> list7 = CollectionsHelper.buildListFrom(4, 5);
        List<Integer> list8 = CollectionsHelper.buildListFrom(6, 4);
        List<List<Integer>> listOfLists = CollectionsHelper.buildListFrom(
                list1,
                list2,
                list3,
                list4,
                list5,
                list6,
                list7,
                list8
                );
        CollectionsHelper.removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers(
                listOfLists);
        assertEquals(2, listOfLists.size());
        assertEquals(list1, listOfLists.get(0));
        assertEquals(list6, listOfLists.get(1));
    }

    @Test
    public void removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers3() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2, 3);
        List<Integer> list2 = CollectionsHelper.buildListFrom(1);
        List<Integer> list3 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list4 = CollectionsHelper.buildListFrom(3, 1);
        List<Integer> list5 = CollectionsHelper.buildListFrom(4);
        List<Integer> list6 = CollectionsHelper.buildListFrom(4, 5, 6);
        List<Integer> list7 = CollectionsHelper.buildListFrom(4, 5);
        List<Integer> list8 = CollectionsHelper.buildListFrom(6, 4);
        List<List<Integer>> listOfLists = CollectionsHelper.buildListFrom(
                list1,
                list5,
                list2,
                list5,
                list3,
                list4,
                list2,
                list5,
                list6,
                list2,
                list7,
                list5,
                list8
                );
        CollectionsHelper.removeIntegerListsThatAreContainedInOtherListsFromListWithListsOfIntegers(
                listOfLists);
        assertEquals(2, listOfLists.size());
        assertEquals(list1, listOfLists.get(0));
        assertEquals(list6, listOfLists.get(1));
    }

    @Test
    public void sortListOfIntegerListByFirstElementOfEachListAndLength() {
        List<Integer> list1 = CollectionsHelper.buildListFrom(1, 2, 3);
        List<Integer> list2 = CollectionsHelper.buildListFrom(1);
        List<Integer> list3 = CollectionsHelper.buildListFrom(1, 2);
        List<Integer> list4 = CollectionsHelper.buildListFrom(4);
        List<Integer> list5 = CollectionsHelper.buildListFrom(4, 5, 6);
        List<Integer> list6 = CollectionsHelper.buildListFrom(4, 5);
        List<List<Integer>> listOfLists = CollectionsHelper.buildListFrom(
                list1,
                list2,
                list3,
                list4,
                list5,
                list6
                );
        CollectionsHelper.sortListOfIntegerListByFirstElementOfEachListAndLength(listOfLists);
        assertEquals(6, listOfLists.size());
        assertEquals(list2, listOfLists.get(0));
        assertEquals(list3, listOfLists.get(1));
        assertEquals(list1, listOfLists.get(2));
        assertEquals(list4, listOfLists.get(3));
        assertEquals(list6, listOfLists.get(4));
        assertEquals(list5, listOfLists.get(5));
    }

    @Test
    public void createShortListThreeFromFiveIntegers() {
        List<Integer> list = CollectionsHelper.buildListFrom(1, 2, 3, 4, 5);
        List<Integer> shorterList = CollectionsHelper.createShortList(list, 3);
        assertEquals(3, shorterList.size());
        assertEquals(Integer.valueOf(1), shorterList.get(0));
        assertEquals(Integer.valueOf(2), shorterList.get(1));
        assertEquals(Integer.valueOf(3), shorterList.get(2));
    }

    @Test
    public void createShortListThreeFromThreeStrings() {
        List<String> list = CollectionsHelper.buildListFrom("1", "2", "3");
        List<String> shorterList = CollectionsHelper.createShortList(list, 3);
        assertEquals(3, shorterList.size());
        assertEquals("1", shorterList.get(0));
        assertEquals("2", shorterList.get(1));
        assertEquals("3", shorterList.get(2));
    }

    @Test
    public void createShortListZeroElementsFromThreeStrings() {
        List<String> list = CollectionsHelper.buildListFrom("1", "2", "3");
        List<String> shorterList = CollectionsHelper.createShortList(list, 0);
        assertEquals(0, shorterList.size());
    }

    @Test
    public void createShortListZeroElementsFromEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        List<String> shorterList = CollectionsHelper.createShortList(list, 0);
        assertEquals(0, shorterList.size());
    }

    @Test
    public void createShortListNoLongerList() {
        List<String> list = CollectionsHelper.buildListFrom("1", "2", "3");
        List<String> shorterList = CollectionsHelper.createShortList(list, 3000);
        assertEquals(3, shorterList.size());
        assertEquals("1", shorterList.get(0));
        assertEquals("2", shorterList.get(1));
        assertEquals("3", shorterList.get(2));
    }

    @Test
    public void startsAnyListElementWithSuccess() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "Birnbaum", "Orangenbaum");
        String start = "Orange";
        assertTrue(CollectionsHelper.startsAnyListElementWith(list, start));
    }

    @Test
    public void startsAnyListElementWithEmptyStartHasSuccess() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "Birnbaum", "Orangenbaum");
        String start = "";
        assertTrue(CollectionsHelper.startsAnyListElementWith(list, start));
    }

    @Test
    public void startsAnyListElementWithFails() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "Birnbaum", "Orangenbaum");
        String start = "Birne"; // das e ist nicht mit enthalten!
        assertFalse(CollectionsHelper.startsAnyListElementWith(list, start));
    }

    @Test
    public void pruneEmptyListElementsAtEndEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        CollectionsHelper.pruneEmptyListElementsAtEnd(list);
        assertEquals(0, list.size());
    }

    @Test
    public void pruneEmptyListElementsAtEndNoEmptyElementsAtEnd1() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "Birnbaum", "Orangenbaum");
        CollectionsHelper.pruneEmptyListElementsAtEnd(list);
        assertEquals(3, list.size());
        assertEquals("Apfelbaum", list.get(0));
        assertEquals("Birnbaum", list.get(1));
        assertEquals("Orangenbaum", list.get(2));
    }

    @Test
    public void pruneEmptyListElementsAtEndNoEmptyElementsAtEnd2() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "", "Orangenbaum");
        CollectionsHelper.pruneEmptyListElementsAtEnd(list);
        assertEquals(3, list.size());
        assertEquals("Apfelbaum", list.get(0));
        assertEquals("", list.get(1));
        assertEquals("Orangenbaum", list.get(2));
    }

    @Test
    public void pruneEmptyListElementsAtEndNoEmptyElementsAtEndOneEmptyElementAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "", "Orangenbaum", "");
        CollectionsHelper.pruneEmptyListElementsAtEnd(list);
        assertEquals(3, list.size());
        assertEquals("Apfelbaum", list.get(0));
        assertEquals("", list.get(1));
        assertEquals("Orangenbaum", list.get(2));
    }

    @Test
    public void pruneEmptyListElementsAtEndNoEmptyElementsAtEndTwoEmptyElementsAtEnd() {
        List<String> list = CollectionsHelper.buildListFrom("Apfelbaum", "", "Orangenbaum", "", "");
        CollectionsHelper.pruneEmptyListElementsAtEnd(list);
        assertEquals(3, list.size());
        assertEquals("Apfelbaum", list.get(0));
        assertEquals("", list.get(1));
        assertEquals("Orangenbaum", list.get(2));
    }

    @Test
    public void allListElementsAreEqualEmptyList() {
        List<String> list = CollectionsHelper.buildListFrom();
        assertFalse(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void allListElementsAreEqualOneElement() {
        List<String> list = CollectionsHelper.buildListFrom("Baum");
        assertTrue(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void allListElementsAreEqualSameElements2() {
        List<String> list = CollectionsHelper.buildListFrom("Baum", "Baum");
        assertTrue(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void allListElementsAreEqualSameElements3() {
        List<String> list = CollectionsHelper.buildListFrom("Baum", "Baum", "Baum");
        assertTrue(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void allListElementsAreEqualDifferentElements2() {
        List<String> list = CollectionsHelper.buildListFrom("Baum", "Strauch");
        assertFalse(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void allListElementsAreEqualDifferentElements3() {
        List<String> list = CollectionsHelper.buildListFrom("Baum", "Baum", "Strauch");
        assertFalse(CollectionsHelper.allListElementsAreEqual(list));
    }

    @Test
    public void getSmallestValue() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(1, 2, 3);
        int actual = CollectionsHelper.getSmallestValue(numbers, -1);
        int expected = 1;
        assertEquals(expected , actual);
    }

    @Test
    public void getSmallestValueNotFirstValue() {
        List<Integer> numbers = CollectionsHelper.buildListFrom(7, 9, 23, 1, 2, 3);
        int actual = CollectionsHelper.getSmallestValue(numbers, -1);
        int expected = 1;
        assertEquals(expected , actual);
    }

    @Test
    public void getSmallestValueEmptyList() {
        List<Integer> numbers = CollectionsHelper.buildListFrom();
        int actual = CollectionsHelper.getSmallestValue(numbers, -1);
        int expected = -1;
        assertEquals(expected , actual);
    }

}
