package de.duehl.basics.datetime;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

public class TimestampCorrectorTest {

    @Test
    public void create() {
        int secondsToAdd = 0;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        assertNotNull(corrector);
    }

    @Test
    public void correctWithZeroSeconds() {
        int secondsToAdd = 0;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = timestamp;
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddOneSecond() {
        int secondsToAdd = 1;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_152010_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractOneSecond() {
        int secondsToAdd = -1;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_152008_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddOneMinute() {
        int secondsToAdd = 60;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_152109_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractOneMinute() {
        int secondsToAdd = -60;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_151909_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddOneHour() {
        int secondsToAdd = 60 * 60;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_162009_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractOneHour() {
        int secondsToAdd = -60 * 60;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_152009_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_142009_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddTenSecondsOverMidnight() {
        int secondsToAdd = 10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161011_235955_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161012_000005_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractTenSecondsOverMidnight() {
        int secondsToAdd = -10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161012_000005_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161011_235955_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddTenSecondsOverMidnightAndMonthBorder() {
        int secondsToAdd = 10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20160930_235955_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20161001_000005_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractTenSecondsOverMidnightAndMonthBorder() {
        int secondsToAdd = -10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20161001_000005_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20160930_235955_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctAddTenSecondsOverMidnightMonthAndYearBorder() {
        int secondsToAdd = 10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20151231_235955_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20160101_000005_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

    @Test
    public void correctSubtractTenSecondsOverMidnightMonthAndYearBorder() {
        int secondsToAdd = -10;
        TimestampCorrector corrector = new TimestampCorrector(secondsToAdd);
        String timestamp = "20160101_000005_D60_foo_bar_baz.jpg";
        String actual = corrector.correct(timestamp);
        String expected = "20151231_235955_D60_foo_bar_baz.jpg";
        assertEquals(expected, actual);
    }

}
