package de.duehl.basics.io;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.io.exceptions.IORuntimeException;
import de.duehl.basics.test.TestHelper;

public class FineFileReaderTest {

    private final String path;

    public FineFileReaderTest() {
        TestHelper testHelper = TestHelper.createForBasics(getClass(), 6);
        path =  testHelper.getTestDataPath() + File.separator;
    }


    /* -------------------------- TESTS -------------------------- */


    @Test(expected = IORuntimeException.class)
    public void createWrongObject() {
        new FineFileReader("nichtda.txt");
    }

    @Test
    public void createWrongObject2() {
        Reader reader = null;
        try {
            reader = new FineFileReader("nichtda.txt");
        }
        catch (IORuntimeException e) {
        }
        assertEquals("Reader muss null sein!", null, reader);
    }

    @Test
    public void createObject() {
        FineFileReader reader = new FineFileReader(path + "testfile1.txt");
        reader.close();
    }

    @Test
    public void createObject2() {
        FineFileReader reader = new FineFileReader(path + "testfile1.txt");
        assertTrue(null != reader);
        reader.close();
    }

    @Test
    public void printReaderToString() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        String expected = "FineFileReader[fileName:C:\\Users\\usr207\\Daten\\"
                + "java\\workspace\\de.duehl.basics\\data\\testfiles\\"
                + "testfile1.txt, charset:ISO-8859-1]";
        String actual = reader.toString();
        assertEquals(expected, actual);
        reader.close();
    }

    @Test
    public void readFile1() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        boolean reading = true;
        List<String> lines = new ArrayList<String>();
        while (reading) {
            String line = reader.readNextLine();
            if (null == line) // EOF reached
                reading = false;
            else
                lines.add(line);
        }
        assertEquals(lines.size(), 2);
        reader.close();
    }

    @Test
    public void readFile2() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        boolean reading = true;
        List<String> lines = new ArrayList<String>();
        while (reading) {
            String line = reader.readNextLine();
            if (null == line) // EOF reached
                reading = false;
            else
                lines.add(line);
        }
        assertEquals(reader.getLineNumber(), 2);
        reader.close();
    }

    @Test
    public void readFile3() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        boolean reading = true;
        List<String> lines = new ArrayList<String>();
        while (reading) {
            String line = reader.readNextLine();
            if (null == line) // EOF reached
                reading = false;
            else
                lines.add(line);
        }
        String result = lines.get(0);
        String expected = "name1\t\\w+";
        assertEquals(expected, result);
        reader.close();
    }

    @Test
    public void readFile4() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        boolean reading = true;
        List<String> lines = new ArrayList<String>();
        while (reading) {
            String line = reader.readNextLine();
            if (null == line) // EOF reached
                reading = false;
            else
                lines.add(line);
        }
        String result = lines.get(1);
        String expected = "name2\tc[^b]*b";
        assertEquals(expected, result);
        reader.close();
    }

    @Test(expected = IORuntimeException.class)
    public void readAfterClose() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        reader.close();
        reader.readNextLine();
    }

    @Test
    public void readFieldsLine01() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        List<String> fields = reader.readFieldsLine();
        assertEquals(2, fields.size());
        assertEquals("name1", fields.get(0));
        assertEquals("\\w+", fields.get(1));
        reader.close();
    }

    @Test
    public void getFileName() {
        Reader reader = new FineFileReader(path + "testfile1.txt");
        String actual = reader.getFileName();
        String expected = path + "testfile1.txt";
        assertEquals(expected, actual);
    }

}