package de.duehl.basics.text;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class NumberStringTest {

    @Test
    public void taupu01() {
        int zahl = 1;
        String expected = "1";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu02() {
        int zahl = 999;
        String expected = "999";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu03() {
        int zahl = 1000;
        String expected = "1.000";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu04() {
        int zahl = 999999;
        String expected = "999.999";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu05() {
        int zahl = 1234567;
        String expected = "1.234.567";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu80k() {
        int zahl = 80_000;
        String expected = "80.000";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void taupu05long() {
        long zahl = 1234567;
        String expected = "1.234.567";
        String actual = NumberString.taupu(zahl);
        assertEquals(expected, actual);
    }

    @Test
    public void removeTaupuEmptyInput() {
        String taupu = "";
        String actual = NumberString.removeTaupu(taupu);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTaupuNoPoint() {
        String taupu = "123";
        String actual = NumberString.removeTaupu(taupu);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTaupuOnePoint() {
        String taupu = "999.999";
        String actual = NumberString.removeTaupu(taupu);
        String expected = "999999";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTaupuTwoPoints() {
        String taupu = "1.234.567";
        String actual = NumberString.removeTaupu(taupu);
        String expected = "1234567";
        assertEquals(expected, actual);
    }

    @Test
    public void isDigitSequenceEmptyString() {
        assertFalse(NumberString.isDigitSequence(""));
    }

    @Test
    public void isDigitSequenceSpaceBefore() {
        assertFalse(NumberString.isDigitSequence(" 123"));
    }

    @Test
    public void isDigitSequenceSpaceBehind() {
        assertFalse(NumberString.isDigitSequence("123 "));
    }

    @Test
    public void isDigitSequenceSpaceBeforeAndBehind() {
        assertFalse(NumberString.isDigitSequence(" 123 "));
    }

    @Test
    public void isDigitPerhapsWithPointSequence() {
        assertTrue(NumberString.isDigitPerhapsWithPointSequence("123.0"));
    }

    @Test
    public void isDigitPerhapsWithPointSequenceWithoutPoint() {
        assertTrue(NumberString.isDigitPerhapsWithPointSequence("123"));
    }

    @Test
    public void isDigitPerhapsWithPointSequenceOnlyPointIsWrong() {
        assertFalse(NumberString.isDigitPerhapsWithPointSequence("."));
    }

    @Test
    public void isDigitPerhapsWithPointSequenceStartingWithPointIsWrong() {
        assertFalse(NumberString.isDigitPerhapsWithPointSequence(".0123"));
    }

    @Test
    public void isDigitPerhapsWithCommaSequence() {
        assertTrue(NumberString.isDigitPerhapsWithCommaSequence("123,0"));
    }

    @Test
    public void isDigitPerhapsWithCommaSequenceWithoutPoint() {
        assertTrue(NumberString.isDigitPerhapsWithCommaSequence("123"));
    }

    @Test
    public void isDigitPerhapsWithCommaSequenceOnlyCommaIsWrong() {
        assertFalse(NumberString.isDigitPerhapsWithCommaSequence(","));
    }

    @Test
    public void isDigitPerhapsWithCommaSequenceCommaIsWrong() {
        assertFalse(NumberString.isDigitPerhapsWithCommaSequence(",0123"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceWithOnlyZero() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("123,"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceWithOnlyOne() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("123,0"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceWithoutPoint() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("123"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceCorrect() {
        assertTrue(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("123,45"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceOnlyCommaIsWrong() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence(","));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceStartsWithComma() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence(",0123"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceTooMany() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("12,0123"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceAInFront() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("1A2,01"));
    }

    @Test
    public void isDigitWithCommaAndTwoNachkommastellenSequenceAInRear() {
        assertFalse(NumberString.isDigitWithCommaAndTwoNachkommastellenSequence("12,A1"));
    }

    @Test
    public void isAnyDigitSequenceWithComma() {
        assertTrue(NumberString.isAnyDigitSequence("123,0"));
    }

    @Test
    public void isAnyDigitSequenceWithPoint() {
        assertTrue(NumberString.isAnyDigitSequence("123.0"));
    }

    @Test
    public void isAnyDigitSequenceWithoutPointAndComma() {
        assertTrue(NumberString.isAnyDigitSequence("123"));
    }

    @Test
    public void isAnyDigitSequenceWithPointAndComma() {
        assertTrue(NumberString.isAnyDigitSequence("123.456,789"));
    }

    @Test
    public void isAnyDigitSequenceOnlyCommaIsWrong() {
        assertFalse(NumberString.isAnyDigitSequence(","));
    }

    @Test
    public void isAnyDigitSequenceOnlyPointIsWrong() {
        assertFalse(NumberString.isAnyDigitSequence("."));
    }

    @Test
    public void isAnyDigitSequenceCommaAtStartIsWrong() {
        assertFalse(NumberString.isAnyDigitSequence(",0123"));
    }

    @Test
    public void isAnyDigitSequencePointAtStartIsWrong() {
        assertFalse(NumberString.isAnyDigitSequence(".0123"));
    }

    @Test
    public void isDigitSequence01() {
        assertTrue(NumberString.isDigitSequence("012345"));
    }

    @Test
    public void isDigitSequenceWithDot() {
        assertFalse(NumberString.isDigitSequence("123.0"));
    }

    @Test
    public void isDigitSequenceWithCommy() {
        assertFalse(NumberString.isDigitSequence("123,0"));
    }

    @Test
    public void isNegativeDigitSequenceEmpty() {
        assertFalse(NumberString.isNegativeDigitSequence(""));
    }

    @Test
    public void isNegativeDigitSequenceZero() {
        assertFalse(NumberString.isNegativeDigitSequence("0"));
    }

    @Test
    public void isNegativeDigitSequencePositiveNumber() {
        assertFalse(NumberString.isNegativeDigitSequence("123"));
    }

    @Test
    public void isNegativeDigitSequenceDoubleMinusInFront() {
        assertFalse(NumberString.isNegativeDigitSequence("--123"));
    }

    @Test
    public void isNegativeDigitSequenceDoubleMinus() {
        assertFalse(NumberString.isNegativeDigitSequence("--"));
    }

    @Test
    public void isNegativeDigitSequenceMinus() {
        assertFalse(NumberString.isNegativeDigitSequence("-"));
    }

    @Test
    public void isNegativeDigitSequenceMinusOne() {
        assertTrue(NumberString.isNegativeDigitSequence("-1"));
    }

    @Test
    public void isNegativeDigitSequenceMinusZero() {
        assertTrue(NumberString.isNegativeDigitSequence("-0"));
    }

    @Test
    public void isNegativeDigitSequence() {
        assertTrue(NumberString.isNegativeDigitSequence("-234"));
    }

    @Test
    public void containsDigitEmptyInput() {
        assertFalse(NumberString.containsDigit(""));
    }

    @Test
    public void containsDigitNonDigitInput() {
        assertFalse(NumberString.containsDigit("Am Apfelbaum hängt ein Apfel."));
    }

    @Test
    public void containsDigitDigitInput() {
        assertTrue(NumberString.containsDigit("Am Apfelbaum hängt 1 Apfel."));
    }

    @Test
    public void containsDigitOnlyDigits() {
        assertTrue(NumberString.containsDigit("12324235"));
    }

    @Test
    public void containsDigitNumber() {
        assertTrue(NumberString.containsDigit("12.324.235,65"));
    }

    @Test
    public void containsDigitWithCurency() {
        assertTrue(NumberString.containsDigit("12.324.235,65 Euro"));
    }

    @Test
    public void removeLeadingZeroesEmptyString() {
        String input = "";
        String actual = NumberString.removeLeadingZeroes(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLeadingZeroesOnlyZeroes() {
        String input = "0000000";
        String actual = NumberString.removeLeadingZeroes(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLeadingZeroesSpaceBeforeZeroes() {
        String input = " 0000000";
        String actual = NumberString.removeLeadingZeroes(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLeadingZeroesNormalCase() {
        String input = "000123";
        String actual = NumberString.removeLeadingZeroes(input);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingEmptyString() {
        String input = "";
        String actual = NumberString.removeZeroEnding(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingWithZeroWithoutDotAtEnd() {
        String input = "20";
        String actual = NumberString.removeZeroEnding(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingWithNoNumberAtEnd() {
        String input = "12345.000 xxx";
        String actual = NumberString.removeZeroEnding(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingLeadingZeroesAndDot() {
        String input = "12345.000";
        String actual = NumberString.removeZeroEnding(input);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingLeadingDot() {
        String input = "12345.";
        String actual = NumberString.removeZeroEnding(input);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void removeZeroEndingLeadingZeros() {
        String input = "12345.12000";
        String actual = NumberString.removeZeroEnding(input);
        String expected = "12345.12";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberEmptyNumber() {
        String text = "";
        String expected = "";
        String actual = NumberString.cleanNumber(text);
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumber01() {
        String text = "10.234.642,12";
        String expected = "10234642.12";
        String actual = NumberString.cleanNumber(text);
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumber02() {
        String text = "10234642.12";
        String expected = "10234642.12";
        String actual = NumberString.cleanNumber(text);
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberTwoCOmmas() {
        String text = "102,34642,12";
        String expected = "1023464212";
        String actual = NumberString.cleanNumber(text);
        assertEquals(expected, actual);
    }

    @Test
    public void findNumberStringsWithoutNumber() {
        String in = "Müll";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertTrue(numbers.isEmpty());
    }

    @Test
    public void findNumberStringsWithPureNumber() {
        String in = "12345";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(1, numbers.size());
        assertEquals("12345", numbers.get(0));
    }

    @Test
    public void findNumberStringsWithPureNumberWithLeadingZero() {
        String in = "0012345";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(1, numbers.size());
        assertEquals("0012345", numbers.get(0));
    }

    @Test
    public void findNumberStringsWithOneNumber() {
        String in = "foo12345bar";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(1, numbers.size());
        assertEquals("12345", numbers.get(0));
    }

    @Test
    public void findNumberStringsWithOneNumberWithSpace() {
        String in = "foo 12345 bar";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(1, numbers.size());
        assertEquals("12345", numbers.get(0));
    }

    @Test
    public void findNumberStringsWithTwoNumbers() {
        String in = "100,32";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(2, numbers.size());
        assertEquals("100", numbers.get(0));
        assertEquals("32", numbers.get(1));
    }

    @Test
    public void findNumberStringsWithThreeNumbers() {
        String in = "100.32E002";
        List<String> numbers = NumberString.findNumberStrings(in);
        assertEquals(3, numbers.size());
        assertEquals("100", numbers.get(0));
        assertEquals("32", numbers.get(1));
        assertEquals("002", numbers.get(2));
    }

    @Test
    public void findNumbersWithoutNumber() {
        String in = "Müll";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertTrue(numbers.isEmpty());
    }

    @Test
    public void findNumbersWithPureNumber() {
        String in = "12345";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(1, numbers.size());
        assertEquals(Integer.valueOf(12345), numbers.get(0));
    }

    @Test
    public void findNumbersWithPureNumberWithLeadingZero() {
        String in = "0012345";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(1, numbers.size());
        assertEquals(Integer.valueOf(12345), numbers.get(0));
    }

    @Test
    public void findNumbersWithOneNumber() {
        String in = "foo12345bar";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(1, numbers.size());
        assertEquals(Integer.valueOf(12345), numbers.get(0));
    }

    @Test
    public void findNumbersWithOneNumberWithSpace() {
        String in = "foo 12345 bar";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(1, numbers.size());
        assertEquals(Integer.valueOf(12345), numbers.get(0));
    }

    @Test
    public void findNumbersWithTwoNumbers() {
        String in = "100,32";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(2, numbers.size());
        assertEquals(Integer.valueOf(100), numbers.get(0));
        assertEquals(Integer.valueOf( 32), numbers.get(1));
    }

    @Test
    public void findNumbersWithThreeNumbers() {
        String in = "100.32E002";
        List<Integer> numbers = NumberString.findNumbers(in);
        assertEquals(3, numbers.size());
        assertEquals(Integer.valueOf(100), numbers.get(0));
        assertEquals(Integer.valueOf( 32), numbers.get(1));
        assertEquals(Integer.valueOf(  2), numbers.get(2));
    }

    @Test
    public void findNaturalNumberAtStartString() {
        String text = "0012345";
        String number = NumberString.findNaturalNumberAtStartString(text);
        assertEquals("0012345", number);
    }

    @Test
    public void findNaturalNumberAtStart() {
        String text = "0012345";
        int number = NumberString.findNaturalNumberAtStart(text);
        assertEquals(12345, number);
    }

    @Test
    public void findNaturalNumberAtStartStringWihOtherBehind() {
        String text = "2>> foo";
        String number = NumberString.findNaturalNumberAtStartString(text);
        assertEquals("2", number);
    }

    @Test
    public void findNaturalNumberAtStartWihOtherBehind() {
        String text = "2>> foo";
        int number = NumberString.findNaturalNumberAtStart(text);
        assertEquals(2, number);
    }

    @Test
    public void findNaturalNumberAtStartStringNoNumberAtStart() {
        String text = "<<vn:2>> foo";
        String number = NumberString.findNaturalNumberAtStartString(text);
        assertEquals("", number);
    }

    @Test
    public void findNaturalNumberAtStartNoNumberAtStart() {
        String text = "<<vn:2>> foo";
        int number = NumberString.findNaturalNumberAtStart(text);
        assertEquals(-1, number);
    }

    @Test
    public void findNaturalNumberAtStartStringWithNegativeNumber() {
        String text = "-2>> foo";
        String number = NumberString.findNaturalNumberAtStartString(text);
        assertEquals("", number);
    }

    @Test
    public void findNaturalNumberAtStartWithNegativeNumber() {
        String text = "-2>> foo";
        int number = NumberString.findNaturalNumberAtStart(text);
        assertEquals(-1, number);
    }

    @Test
    public void addLeadingZeroesWithNegativeNumber() {
        int input = -1;
        int numberOfDigits = 13;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "-1";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesWithZero() {
        int input = 0;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "000";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesWithLongerNumber() {
        int input = 123;
        int numberOfDigits = 2;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesWithExaktLongNumber() {
        int input = 123;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesWithShorterNumber() {
        int input = 12;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "012";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesLongWithNegativeNumber() {
        long input = -1;
        int numberOfDigits = 13;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "-1";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesLongWithZero() {
        long input = 0;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "000";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesLongWithLongerNumber() {
        long input = 123;
        int numberOfDigits = 2;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesLongWithExaktLongNumber() {
        long input = 123;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesLongWithShorterNumber() {
        long input = 12;
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "012";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesStringWithNegativeNumber() {
        String input = "-1";
        int numberOfDigits = 13;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "-1";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesStringWithZero() {
        String input = "0";
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "000";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesStringWithStringerNumber() {
        String input = "123";
        int numberOfDigits = 2;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesStringWithExaktStringNumber() {
        String input = "123";
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "123";
        assertEquals(expected, actual);
    }

    @Test
    public void addLeadingZeroesStringWithShorterNumber() {
        String input = "12";
        int numberOfDigits = 3;
        String actual = NumberString.addLeadingZeroes(input, numberOfDigits);
        String expected = "012";
        assertEquals(expected, actual);
    }

    @Test
    public void parseIntGood() {
        String in = "15";
        int actual = NumberString.parseInt(in, "Hier sollte kein Fehler auftreten!");
        int expected = 15;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBadNumber() {
        String in = "15,7";
        NumberString.parseInt(in, "Zahl mit Komma kann nicht geparst werden!");
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBadDot() {
        String in = "15.7";
        NumberString.parseInt(in, "Zahl mit Punkt kann nicht geparst werden!");
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBad() {
        String in = "Müll";
        NumberString.parseInt(in, "Buchstaben können nicht geparst werden!");
    }

    @Test
    public void parseIntGoodWithoutErrorMessage() {
        String in = "15";
        int actual = NumberString.parseInt(in);
        int expected = 15;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBadNumberWithoutErrorMessage() {
        String in = "15,7";
        NumberString.parseInt(in);
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBadDotWithoutErrorMessage() {
        String in = "15.7";
        NumberString.parseInt(in);
    }

    @Test (expected = RuntimeException.class)
    public void parseIntBadWithoutErrorMessage() {
        String in = "Müll";
        NumberString.parseInt(in);
    }

    @Test
    public void parseIntsGood() {
        List<String> list = CollectionsHelper.buildListFrom("15", "5", "700");
        List<Integer> integers = NumberString.parseInts(list);
        assertEquals(3, integers.size());
        assertEquals(Integer.valueOf( 15), integers.get(0));
        assertEquals(Integer.valueOf(  5), integers.get(1));
        assertEquals(Integer.valueOf(700), integers.get(2));
    }


    @Test
    public void parseIntIgnoreGood() {
        String in = "15";
        int actual = NumberString.parseIntIgnore(in, -1);
        int expected = 15;
        assertEquals(expected, actual);
    }

    @Test
    public void parseIntIgnoreBadNumber() {
        String in = "15,7";
        int actual = NumberString.parseIntIgnore(in, -1);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void parseIntIgnoreBadDot() {
        String in = "15.7";
        int actual = NumberString.parseIntIgnore(in, -1);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void parseIntIgnoreBad() {
        String in = "Müll";
        int actual = NumberString.parseIntIgnore(in, -1);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void parseIntIgnoreEmpty() {
        String in = "";
        int actual = NumberString.parseIntIgnore(in, -1);
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void isInteger0() {
        String number = "0";
        assertTrue(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerMinusOne() {
        String number = "-1";
        assertTrue(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerEmptyString() {
        String number = "";
        assertFalse(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerWhitespace() {
        String number = " ";
        assertFalse(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerNineDigitNumber() {
        String number = "800112471";
        assertTrue(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerHighestNineDigitNumber() {
        String number = "999999999";
        assertTrue(NumberString.isInteger(number));
    }

    @Test
    public void isIntegerBigNumber() {
        String number = "8001124714237260";
        assertFalse(NumberString.isInteger(number));
    }

    @Test
    public void parseLongGood() {
        String in = "8001124714237260";
        long actual = NumberString.parseLong(in, "Hier sollte kein Fehler auftreten!");
        long expected = 8001124714237260L;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void parseLongBadNumber() {
        String in = "15,7";
        NumberString.parseLong(in, "Zahl mit Komma kann nicht geparst werden!");
    }

    @Test
    public void parseDoubleGood() {
        String in = "15.7";
        double actual = NumberString.parseDouble(in, "Hier sollte kein Fehler auftreten!");
        double expected = 15.7d;
        assertEquals(expected, actual, 0.0001);
    }

    @Test (expected = RuntimeException.class)
    public void parseDoubleBadNumber() {
        String in = "fünfzehn komma sieben!";
        NumberString.parseDouble(in, "Text kann nicht geparst werden!");
    }

    @Test
    public void twoDigitStringToInteger15() {
        String in = "15";
        int actual = NumberString.twoDigitStringToInteger(in);
        int expected = 15;
        assertEquals(expected, actual);
    }

    @Test
    public void twoDigitStringToInteger01() {
        String in = "01";
        int actual = NumberString.twoDigitStringToInteger(in);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void twoDigitStringToIntegerTooShort() {
        String in = "1";
        NumberString.twoDigitStringToInteger(in);
    }

    @Test (expected = RuntimeException.class)
    public void twoDigitStringToIntegerTooLong() {
        String in = "123";
        NumberString.twoDigitStringToInteger(in);
    }

    @Test (expected = RuntimeException.class)
    public void twoDigitStringToIntegerNoNumber() {
        String in = "ab";
        NumberString.twoDigitStringToInteger(in);
    }

    @Test (expected = RuntimeException.class)
    public void twoDigitStringToIntegerNoNumberAtSecondPlace() {
        String in = "0a";
        NumberString.twoDigitStringToInteger(in);
    }

    @Test
    public void twoDecimalPlaces() {
        double input = 12345.67890123d;
        String actual = NumberString.twoDecimalPlaces(input);
        String expected = "12345,68";
        assertEquals(expected, actual);
    }

    @Test
    public void twoDecimalPlacesTaupu() {
        double input = 12345.67890123d;
        String actual = NumberString.twoDecimalPlacesTaupu(input);
        String expected = "12.345,68";
        assertEquals(expected, actual);
    }

    @Test
    public void twoDecimalPlacesTaupuTwoDecimals() {
        double input = 12345.67d;
        String actual = NumberString.twoDecimalPlacesTaupu(input);
        String expected = "12.345,67";
        assertEquals(expected, actual);
    }

    @Test
    public void twoDecimalPlacesTaupuOneDecimal() {
        double input = 12345.6d;
        String actual = NumberString.twoDecimalPlacesTaupu(input);
        String expected = "12.345,60";
        assertEquals(expected, actual);
    }

    @Test
    public void twoDecimalPlacesTaupuWithoutDecimals() {
        double input = 12345d;
        String actual = NumberString.twoDecimalPlacesTaupu(input);
        String expected = "12.345,00";
        assertEquals(expected, actual);
    }

    @Test
    public void twoDecimalPlacesTaupuBig() {
        double input = 12345e12;
        String actual = NumberString.twoDecimalPlacesTaupu(input);
        String expected = "12.345.000.000.000.000,00";
        assertEquals(expected, actual);
    }

    @Test
    public void doubleListToTwoDecimalPlacesTaupuList() {
        List<Double> numbers = CollectionsHelper.buildListFrom(
                12345.67890123d,
                12345.67d,
                12345.6d,
                12345d,
                12345e12
                );
        List<String> actualValues = NumberString.doubleListToTwoDecimalPlacesTaupuList(numbers);
        assertEquals(5, actualValues.size());
        assertEquals("12.345,68", actualValues.get(0));
        assertEquals("12.345,67", actualValues.get(1));
        assertEquals("12.345,60", actualValues.get(2));
        assertEquals("12.345,00", actualValues.get(3));
        assertEquals("12.345.000.000.000.000,00", actualValues.get(4));
    }

    @Test
    public void calculateDigits0() {
        int number = 0;
        int actual = NumberString.calculateDigits(number);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void calculateDigitsMinusOne() {
        int number = -1;
        NumberString.calculateDigits(number);
    }

    @Test
    public void calculateDigits1() {
        int number = 1;
        int actual = NumberString.calculateDigits(number);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits9() {
        int number = 9;
        int actual = NumberString.calculateDigits(number);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits10() {
        int number = 10;
        int actual = NumberString.calculateDigits(number);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits99() {
        int number = 99;
        int actual = NumberString.calculateDigits(number);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits100() {
        int number = 100;
        int actual = NumberString.calculateDigits(number);
        int expected = 3;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits999() {
        int number = 999;
        int actual = NumberString.calculateDigits(number);
        int expected = 3;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits1000() {
        int number = 1000;
        int actual = NumberString.calculateDigits(number);
        int expected = 4;
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDigits9999() {
        int number = 9999;
        int actual = NumberString.calculateDigits(number);
        int expected = 4;
        assertEquals(expected, actual);
    }

    @Test
    public void percent2From10() {
        int count = 2;
        int total = 10;
        String actual = NumberString.percent(count, total);
        String expected = "20,00";
        assertEquals(expected, actual);
    }

    @Test
    public void percent20From100() {
        int count = 20;
        int total = 100;
        String actual = NumberString.percent(count, total);
        String expected = "20,00";
        assertEquals(expected, actual);
    }

    @Test
    public void percent0From80() {
        int count = 0;
        int total = 80;
        String actual = NumberString.percent(count, total);
        String expected = "0,00";
        assertEquals(expected, actual);
    }

    @Test
    public void percent12From12() {
        int count = 12;
        int total = 12;
        String actual = NumberString.percent(count, total);
        String expected = "100,00";
        assertEquals(expected, actual);
    }

    @Test
    public void percent12LFrom12L() {
        long count = 12L;
        long total = 12L;
        String actual = NumberString.percent(count, total);
        String expected = "100,00";
        assertEquals(expected, actual);
    }

    @Test
    public void percent0From0() {
        int count = 0;
        int total = 0;
        String actual = NumberString.percent(count, total);
        String expected = "NaN";
        assertEquals(expected, actual);
    }

    @Test
    public void toGermanNumberZero() {
        String actual = NumberString.toGermanNumber("0");
        String expected = "Null";
        assertEquals(expected, actual);
    }

    @Test
    public void toGermanNumberOne() {
        String actual = NumberString.toGermanNumber("1");
        String expected = "Ein";
        assertEquals(expected, actual);
    }

    @Test
    public void toGermanNumberOther() {
        String actual = NumberString.toGermanNumber("999");
        String expected = "999";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberStringZero() {
        String actual = NumberString.toSmallGermanNumber("0");
        String expected = "kein";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberStringOne() {
        String actual = NumberString.toSmallGermanNumber("1");
        String expected = "ein";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberStringOther() {
        String actual = NumberString.toSmallGermanNumber("999");
        String expected = "999";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberZero() {
        String actual = NumberString.toSmallGermanNumber(0);
        String expected = "kein";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberOne() {
        String actual = NumberString.toSmallGermanNumber(1);
        String expected = "ein";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberOther() {
        String actual = NumberString.toSmallGermanNumber(999);
        String expected = "999";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEStringZero() {
        String actual = NumberString.toSmallGermanNumberWithE("0");
        String expected = "keine";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEStringOne() {
        String actual = NumberString.toSmallGermanNumberWithE("1");
        String expected = "eine";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEStringOther() {
        String actual = NumberString.toSmallGermanNumberWithE("999");
        String expected = "999";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEZero() {
        String actual = NumberString.toSmallGermanNumberWithE(0);
        String expected = "keine";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEOne() {
        String actual = NumberString.toSmallGermanNumberWithE(1);
        String expected = "eine";
        assertEquals(expected, actual);
    }

    @Test
    public void toSmallGermanNumberWithEOther() {
        String actual = NumberString.toSmallGermanNumberWithE(999);
        String expected = "999";
        assertEquals(expected, actual);
    }

    @Test
    public void germanPluralZero() {
        String actual = NumberString.germanPlural(0, "Worte", "Wort");
        String expected = "Worte";
        assertEquals(expected, actual);
    }

    @Test
    public void germanPluralOne() {
        String actual = NumberString.germanPlural(1, "Worte", "Wort");
        String expected = "Wort";
        assertEquals(expected, actual);
    }

    @Test
    public void germanPluralOther() {
        String actual = NumberString.germanPlural(999, "Worte", "Wort");
        String expected = "Worte";
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow0() {
        int exponent = 0;
        long actual = NumberString.hexPow(exponent);
        long expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow1() {
        int exponent = 1;
        long actual = NumberString.hexPow(exponent);
        long expected = 16;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow2() {
        int exponent = 2;
        long actual = NumberString.hexPow(exponent);
        long expected = 256;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow3() {
        int exponent = 3;
        long actual = NumberString.hexPow(exponent);
        long expected = 4_096;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow4() {
        int exponent = 4;
        long actual = NumberString.hexPow(exponent);
        long expected = 65_536;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow5() {
        int exponent = 5;
        long actual = NumberString.hexPow(exponent);
        long expected = 1_048_576;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow6() {
        int exponent = 6;
        long actual = NumberString.hexPow(exponent);
        long expected = 16_777_216;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow7() {
        int exponent = 7;
        long actual = NumberString.hexPow(exponent);
        long expected = 268_435_456;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow8() {
        int exponent = 8;
        long actual = NumberString.hexPow(exponent);
        long expected = 4_294_967_296L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow9() {
        int exponent = 9;
        long actual = NumberString.hexPow(exponent);
        long expected = 68_719_476_736L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexPow10() {
        int exponent = 10;
        long actual = NumberString.hexPow(exponent);
        long expected = 1_099_511_627_776L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long0() {
        String hex = "0";
        long actual = NumberString.hexChar2long(hex);
        long expected = 0L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long1() {
        String hex = "1";
        long actual = NumberString.hexChar2long(hex);
        long expected = 1L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long2() {
        String hex = "2";
        long actual = NumberString.hexChar2long(hex);
        long expected = 2L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long3() {
        String hex = "3";
        long actual = NumberString.hexChar2long(hex);
        long expected = 3L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long4() {
        String hex = "4";
        long actual = NumberString.hexChar2long(hex);
        long expected = 4L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long5() {
        String hex = "5";
        long actual = NumberString.hexChar2long(hex);
        long expected = 5L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long6() {
        String hex = "6";
        long actual = NumberString.hexChar2long(hex);
        long expected = 6L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long7() {
        String hex = "7";
        long actual = NumberString.hexChar2long(hex);
        long expected = 7L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long8() {
        String hex = "8";
        long actual = NumberString.hexChar2long(hex);
        long expected = 8L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2long9() {
        String hex = "9";
        long actual = NumberString.hexChar2long(hex);
        long expected = 9L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longA() {
        String hex = "A";
        long actual = NumberString.hexChar2long(hex);
        long expected = 10L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleA() {
        String hex = "a";
        long actual = NumberString.hexChar2long(hex);
        long expected = 10L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longB() {
        String hex = "B";
        long actual = NumberString.hexChar2long(hex);
        long expected = 11L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleB() {
        String hex = "b";
        long actual = NumberString.hexChar2long(hex);
        long expected = 11L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longC() {
        String hex = "C";
        long actual = NumberString.hexChar2long(hex);
        long expected = 12L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleC() {
        String hex = "c";
        long actual = NumberString.hexChar2long(hex);
        long expected = 12L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longD() {
        String hex = "D";
        long actual = NumberString.hexChar2long(hex);
        long expected = 13L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleD() {
        String hex = "d";
        long actual = NumberString.hexChar2long(hex);
        long expected = 13L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longE() {
        String hex = "E";
        long actual = NumberString.hexChar2long(hex);
        long expected = 14L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleE() {
        String hex = "e";
        long actual = NumberString.hexChar2long(hex);
        long expected = 14L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longF() {
        String hex = "F";
        long actual = NumberString.hexChar2long(hex);
        long expected = 15L;
        assertEquals(expected, actual);
    }

    @Test
    public void hexChar2longLittleF() {
        String hex = "f";
        long actual = NumberString.hexChar2long(hex);
        long expected = 15L;
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hexChar2longLittleG() {
        String hex = "g";
        NumberString.hexChar2long(hex);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hexChar2longEmpty() {
        String hex = "";
        NumberString.hexChar2long(hex);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hexChar2longMoreThanOneChar() {
        String hex = "00";
        NumberString.hexChar2long(hex);
    }

    @Test
    public void hex2long0() {
        String hex = "0";
        long actual = NumberString.hex2long(hex);
        long expected = 0L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long1() {
        String hex = "1";
        long actual = NumberString.hex2long(hex);
        long expected = 1L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long2() {
        String hex = "2";
        long actual = NumberString.hex2long(hex);
        long expected = 2L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long3() {
        String hex = "3";
        long actual = NumberString.hex2long(hex);
        long expected = 3L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long4() {
        String hex = "4";
        long actual = NumberString.hex2long(hex);
        long expected = 4L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long5() {
        String hex = "5";
        long actual = NumberString.hex2long(hex);
        long expected = 5L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long6() {
        String hex = "6";
        long actual = NumberString.hex2long(hex);
        long expected = 6L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long7() {
        String hex = "7";
        long actual = NumberString.hex2long(hex);
        long expected = 7L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long8() {
        String hex = "8";
        long actual = NumberString.hex2long(hex);
        long expected = 8L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2long9() {
        String hex = "9";
        long actual = NumberString.hex2long(hex);
        long expected = 9L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longA() {
        String hex = "A";
        long actual = NumberString.hex2long(hex);
        long expected = 10L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleA() {
        String hex = "a";
        long actual = NumberString.hex2long(hex);
        long expected = 10L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longB() {
        String hex = "B";
        long actual = NumberString.hex2long(hex);
        long expected = 11L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleB() {
        String hex = "b";
        long actual = NumberString.hex2long(hex);
        long expected = 11L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longC() {
        String hex = "C";
        long actual = NumberString.hex2long(hex);
        long expected = 12L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleC() {
        String hex = "c";
        long actual = NumberString.hex2long(hex);
        long expected = 12L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longD() {
        String hex = "D";
        long actual = NumberString.hex2long(hex);
        long expected = 13L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleD() {
        String hex = "d";
        long actual = NumberString.hex2long(hex);
        long expected = 13L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longE() {
        String hex = "E";
        long actual = NumberString.hex2long(hex);
        long expected = 14L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleE() {
        String hex = "e";
        long actual = NumberString.hex2long(hex);
        long expected = 14L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longF() {
        String hex = "F";
        long actual = NumberString.hex2long(hex);
        long expected = 15L;
        assertEquals(expected, actual);
    }

    @Test
    public void hex2longLittleF() {
        String hex = "f";
        long actual = NumberString.hex2long(hex);
        long expected = 15L;
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hex2longEmpty() {
        String hex = "";
        NumberString.hex2long(hex);
    }

    @Test
    public void cleanNumberFromSpacesAndLeadingZerosEmptyInput() {
        String number = "";
        String actual = NumberString.cleanNumberFromSpacesAndLeadingZeros(number);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberFromSpacesAndLeadingZerosNumber() {
        String number = "12345";
        String actual = NumberString.cleanNumberFromSpacesAndLeadingZeros(number);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberFromSpacesAndLeadingZerosNumberWithWhitespace() {
        String number = " 12  \t 345     ";
        String actual = NumberString.cleanNumberFromSpacesAndLeadingZeros(number);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberFromSpacesAndLeadingZerosLeadingZeros() {
        String number = "0000012345";
        String actual = NumberString.cleanNumberFromSpacesAndLeadingZeros(number);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanNumberFromSpacesAndLeadingZerosNumberWithWhitespaceAndLeadingZeros() {
        String number = " 00  000 00 12  \t 345     ";
        String actual = NumberString.cleanNumberFromSpacesAndLeadingZeros(number);
        String expected = "12345";
        assertEquals(expected, actual);
    }

    @Test
    public void isPipeSeparatedListOfNumbers() {
        String input = "123|456|78";
        assertTrue(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersEmptyInput() {
        String input = "";
        assertFalse(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersOnlyPipe() {
        String input = "|";
        assertFalse(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersPipeAtStart() {
        String input = "|123";
        assertFalse(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersPipeAtEnd() {
        String input = "323|123|";
        assertFalse(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersMultiplePipes() {
        String input = "323||123";
        assertFalse(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void isPipeSeparatedListOfNumbersOnlyNumber() {
        String input = "123";
        assertTrue(NumberString.isPipeSeparatedListOfNumbers(input));
    }

    @Test
    public void looksLikeNumberPerhapsWithSpaceSlashMinusSimpleNumber() {
        String input = "123";
        assertTrue(NumberString.looksLikeNumberPerhapsWithSpaceSlashMinus(input));
    }

    @Test
    public void looksLikeNumberPerhapsWithSpaceSlashMinusWithSpaces() {
        String input = "123 456 789";
        assertTrue(NumberString.looksLikeNumberPerhapsWithSpaceSlashMinus(input));
    }

    @Test
    public void looksLikeNumberPerhapsWithSpaceSlashMinusWithSpacesAndMore() {
        String input = "0521 / 98 216-27";
        assertTrue(NumberString.looksLikeNumberPerhapsWithSpaceSlashMinus(input));
    }

    @Test
    public void looksLikeNumberPerhapsWithComma() {
        String input = "10,43";
        assertFalse(NumberString.looksLikeNumberPerhapsWithSpaceSlashMinus(input));
    }

    @Test
    public void looksLikeNumberPerhapsWithPoint() {
        String input = "10.432";
        assertFalse(NumberString.looksLikeNumberPerhapsWithSpaceSlashMinus(input));
    }

    @Test
    public void endsWithDigitEmptyInput() {
        String input = "";
        assertFalse(NumberString.endsWithDigit(input));
    }

    @Test
    public void endsWithDigitSingleDigit() {
        String input = "7";
        assertTrue(NumberString.endsWithDigit(input));
    }

    @Test
    public void endsWithDigitCharDigit() {
        String input = "a";
        assertFalse(NumberString.endsWithDigit(input));
    }

    @Test
    public void endsWithDigitSingleDigitWithSpaceBehind() {
        String input = "7 ";
        assertFalse(NumberString.endsWithDigit(input));
    }

    @Test
    public void startsWithDigitEmptyInput() {
        String input = "";
        assertFalse(NumberString.startsWithDigit(input));
    }

    @Test
    public void startsWithDigitSingleDigit() {
        String input = "7";
        assertTrue(NumberString.startsWithDigit(input));
    }

    @Test
    public void startsWithDigitCharDigit() {
        String input = "a";
        assertFalse(NumberString.startsWithDigit(input));
    }

    @Test
    public void startsWithDigitSingleDigitWithSpaceBehind() {
        String input = " 7";
        assertFalse(NumberString.startsWithDigit(input));
    }

    @Test
    public void startsWithFiveDigitAndSpace() {
        String input = "20146 Hamburg";
        assertTrue(NumberString.startsWithFiveDigitAndSpace(input));
    }

    @Test
    public void startsWithFiveDigitAndSpaceWrong() {
        String input = "1234 Bremen";
        assertFalse(NumberString.startsWithFiveDigitAndSpace(input));
    }

    @Test
    public void taupuIfPossibleSingelDigit() {
        String numberString = "7";
        String actual = NumberString.taupuIfPossible(numberString);
        String expected = "7";
        assertEquals(expected, actual);
    }

    @Test
    public void taupuIfPossible1000() {
        String numberString = "1000";
        String actual = NumberString.taupuIfPossible(numberString);
        String expected = "1.000";
        assertEquals(expected, actual);
    }

    @Test
    public void taupuIfPossibleNotANumber() {
        String numberString = "einhundert";
        String actual = NumberString.taupuIfPossible(numberString);
        String expected = "einhundert";
        assertEquals(expected, actual);
    }

    @Test
    public void taupuIfPossible6264201967() {
        String numberString = "6264201967";
        String actual = NumberString.taupuIfPossible(numberString);
        String expected = "6.264.201.967";
        assertEquals(expected, actual);
    }

    @Test
    public void taupuIfPossible62642019679112233456() {
        // auch für Long zu lang
        String numberString = "62642019679112233456";
        String actual = NumberString.taupuIfPossible(numberString);
        String expected = "62642019679112233456";
        assertEquals(expected, actual);
    }

    @Test
    public void numberAndPercent1() {
        int number = 3877;
        int total = 4030;
        int lenghtOfNumberWithDot = 5;
        String actual = NumberString.numberAndPercent(number, total, lenghtOfNumberWithDot);
        String expected = "3.877 ( 96,20%)";
        assertEquals(expected, actual);
    }

    @Test
    public void numberAndPercent2() {
        int number = 385;
        int total = 4030;
        int lenghtOfNumberWithDot = 5;
        String actual = NumberString.numberAndPercent(number, total, lenghtOfNumberWithDot);
        String expected = "  385 (  9,55%)";
        assertEquals(expected, actual);
    }

    @Test
    public void numberAndPercent3() {
        int number = 153;
        int total = 4030;
        int lenghtOfNumberWithDot = 5;
        String actual = NumberString.numberAndPercent(number, total, lenghtOfNumberWithDot);
        String expected = "  153 (  3,80%)";
        assertEquals(expected, actual);
    }

    @Test
    public void numberAndPercent4() {
        int number = 4030;
        int total = 4030;
        int lenghtOfNumberWithDot = 5;
        String actual = NumberString.numberAndPercent(number, total, lenghtOfNumberWithDot);
        String expected = "4.030 (100,00%)";
        assertEquals(expected, actual);
    }

    @Test
    public void parseDoubleWithTaupuAndKomma0_00() {
        String numberString = "0,00";
        double failureValue = Double.MIN_VALUE;
        double actual = NumberString.parseDoubleWithTaupuAndKomma(numberString, failureValue);
        double expected = 0.0d;
        assertEquals(expected, actual, 0.01d);
    }

    @Test
    public void parseDoubleWithTaupuAndKomma500000000_00() {
        String numberString = "500.000.000,00";
        double failureValue = Double.MIN_VALUE;
        double actual = NumberString.parseDoubleWithTaupuAndKomma(numberString, failureValue);
        double expected = 500000000.0d;
        assertEquals(expected, actual, 0.01d);
    }

}
