package de.duehl.basics.text.html;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.io.File;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FineFileReader;
import de.duehl.basics.io.Reader;
import de.duehl.basics.test.TestHelper;
import de.duehl.basics.text.Text;

public class HtmlToolTest {

    private final String path;

    public HtmlToolTest() {
        TestHelper testHelper = TestHelper.createForBasics(getClass(), 7);
        path =  testHelper.getTestDataPath() + File.separator;
    }

    /* -------------------------- TESTS -------------------------- */

    @Test
    public void urlEscape01() {
        String text = "Sieh mal: http://www.duehl.de/";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de/\">http://www.duehl.de/</a>";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape02() {
        String text = "Sieh mal: http://www.duehl.de/.";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de/\">http://www.duehl.de/</a>.";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape03() {
        String text = "Sieh mal: http://www.duehl.de/, boah!";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de/\">http://www.duehl.de/</a>, boah!";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape04() {
        String text = "Sieh mal: http://www.duehl.de/!";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de/\">http://www.duehl.de/</a>!";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape05() {
        String text = "Sieh mal: http://www.duehl.de/?";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de/\">http://www.duehl.de/</a>?";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape06() {
        String text = "Sieh mal: http://www.duehl.de";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de\">http://www.duehl.de</a>";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape07() {
        String text = "Sieh mal: http://www.duehl.de.";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de\">http://www.duehl.de</a>.";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape08() {
        String text = "Sieh mal: http://www.duehl.de, boah!";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de\">http://www.duehl.de</a>, boah!";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape09() {
        String text = "Sieh mal: http://www.duehl.de!";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de\">http://www.duehl.de</a>!";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape10() {
        String text = "Sieh mal: http://www.duehl.de?";
        String expected = "Sieh mal: <a href=\"http://www.duehl.de\">http://www.duehl.de</a>?";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape11() {
        String text = "http://www.duehl.de";
        String expected = "<a href=\"http://www.duehl.de\">http://www.duehl.de</a>";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape12() {
        String text = "www.duehl.de";
        String expected = "<a href=\"www.duehl.de\">www.duehl.de</a>";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape13() {
        String text = "www.duehl.de.";
        String expected = "<a href=\"www.duehl.de\">www.duehl.de</a>.";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEscape14() {
        String text = "Schau da! www.duehl.de!";
        String expected = "Schau da! <a href=\"www.duehl.de\">www.duehl.de</a>!";
        String actual = HtmlTool.textToHtmlConvertingURLsToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape1() {
        String text = "c:\\bla.txt";
        String expected = "<a href=\"file://c:/bla.txt\">c:\\bla.txt</a>";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape2() {
        String text = "Schau da! c:\\bla.txt!";
        String expected = "Schau da! <a href=\"file://c:/bla.txt\">c:\\bla.txt</a>!";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape3() {
        String text = "Schau da! c:\\bla.txt.";
        String expected = "Schau da! <a href=\"file://c:/bla.txt\">c:\\bla.txt</a>.";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape4() {
        String text = "Schau da! c:\\bla.txt,";
        String expected = "Schau da! <a href=\"file://c:/bla.txt\">c:\\bla.txt</a>,";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape5() {
        String text = "C:\\bla.txt";
        String expected = "<a href=\"file://C:/bla.txt\">C:\\bla.txt</a>";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void fileEscape6() {
        String text = "\\\\bla\\fasel\\bla.txt";
        String expected = "<a href=\"file://bla/fasel/bla.txt\">\\\\bla\\fasel\\bla.txt</a>";
        String actual = HtmlTool.textToHtmlConvertingFilesToLinks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void green1() {
        String input = "Irgendein Text...";
        String output = HtmlTool.green(input);
        assertTrue(output.startsWith("<font color=\""));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void green2() {
        String input = "Irgendein Text...";
        String output = HtmlTool.green(input);
        assertTrue(output.startsWith("<font color=\"#00A000\">"));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void red1() {
        String input = "Irgendein Text...";
        String output = HtmlTool.red(input);
        assertTrue(output.startsWith("<font color=\""));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void red2() {
        String input = "Irgendein Text...";
        String output = HtmlTool.red(input);
        assertTrue(output.startsWith("<font color=\"#FF0000\">"));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void orange1() {
        String input = "Irgendein Text...";
        String output = HtmlTool.orange(input);
        assertTrue(output.startsWith("<font color=\""));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void orange2() {
        String input = "Irgendein Text...";
        String output = HtmlTool.orange(input);
        assertTrue(output.startsWith("<font color=\"#FF6600\">"));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void greenNumber() {
        int number = 42;
        String actual = HtmlTool.green(number);
        String expected = "<font color=\"#00A000\">" + number + "</font>";
        assertEquals(expected, actual);
    }

    @Test
    public void redNumber() {
        int number = 42;
        String actual = HtmlTool.red(number);
        String expected = "<font color=\"#FF0000\">" + number + "</font>";
        assertEquals(expected, actual);
    }

    @Test
    public void orangeNumber() {
        int number = 42;
        String actual = HtmlTool.orange(number);
        String expected = "<font color=\"#FF6600\">" + number + "</font>";
        assertEquals(expected, actual);
    }

    @Test
    public void colorText1() {
        String input = "Irgendein Text...";
        String output = HtmlTool.colorText("blabla", input);
        assertTrue(output.startsWith("<font color=\""));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void colorText2() {
        String input = "Irgendein Text...";
        String output = HtmlTool.colorText("blabla", input);
        assertTrue(output.startsWith("<font color=\"blabla\">"));
        assertTrue(output.endsWith("</font>"));
    }

    @Test
    public void createLink() {
        String url = "http://www.example.org/";
        String linkText = "Dies ist ein Link!";
        String actual = HtmlTool.createLink(url, linkText);
        String expected = "<a href=\"http://www.example.org/\">Dies ist ein Link!</a>";
        assertEquals(expected , actual);
    }

    @Test
    public void createAnchor() {
        String actual = HtmlTool.createAnchor("anker");
        String expected = "<a name=\"anker\" />";
        assertEquals(expected , actual);
    }

    @Test
    public void strong() {
        String actual = HtmlTool.strong("foo");
        String expected = "<strong>foo</strong>";
        assertEquals(expected , actual);
    }

    @Test
    public void strongNumber() {
        String actual = HtmlTool.strong(42);
        String expected = "<strong>42</strong>";
        assertEquals(expected , actual);
    }

    @Test
    public void createCenteredBigText() {
        String actual = HtmlTool.createCenteredBigText("foo");
        String expected = "<html><center><p style=\"font-size:large\">foo</p></center></html>";
        assertEquals(expected , actual);
    }

    @Test
    public void htmlify() {
        String input = "Dies ist ein langer Text der umgebrochen und eingerückt werden soll.\n"
                + "\ta = b;\n"
                + "\tc = {\n"
                + "\t\td = e,\n"
                + "\t\tf = g,\n"
                + "\t};";
        String actual = HtmlTool.htmlify(input, 20);
        String expected = "<html>Dies ist ein langer<br/>" + Text.LINE_BREAK
                + "Text der umgebrochen<br/>" + Text.LINE_BREAK
                + "und eingerückt<br/>" + Text.LINE_BREAK
                + "werden soll.<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;a = b;<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;c = {<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;d = e,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;f = g,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;};</html>";
        assertEquals(expected , actual);
    }

    @Test
    public void htmlifyWithoutFrame() {
        String input = "Dies ist ein langer Text der umgebrochen und eingerückt werden soll.\n"
                + "\ta = b;\n"
                + "\tc = {\n"
                + "\t\td = e,\n"
                + "\t\tf = g,\n"
                + "\t};";
        String actual = HtmlTool.htmlifyWithoutFrame(input, 20);
        String expected = "Dies ist ein langer<br/>" + Text.LINE_BREAK
                + "Text der umgebrochen<br/>" + Text.LINE_BREAK
                + "und eingerückt<br/>" + Text.LINE_BREAK
                + "werden soll.<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;a = b;<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;c = {<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;d = e,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;f = g,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;};";
        assertEquals(expected , actual);
    }

    @Test
    public void htmlifyWithoutFrameWithSmallerAndBigger() {
        String input = "Dies ist ein langer Text der umgebrochen und eingerückt werden soll.\n"
                + "\ta = b;\n"
                + "\tc = {\n"
                + "\t\td = e,\n"
                + "\t\tf = g,\n"
                + "\t};\n"
                + "h <= i\n"
                + "j > k"
                ;
        String actual = HtmlTool.htmlifyWithoutFrame(input, 20);
        String expected = "Dies ist ein langer<br/>" + Text.LINE_BREAK
                + "Text der umgebrochen<br/>" + Text.LINE_BREAK
                + "und eingerückt<br/>" + Text.LINE_BREAK
                + "werden soll.<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;a = b;<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;c = {<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;d = e,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;f = g,<br/>" + Text.LINE_BREAK
                + "&nbsp;&nbsp;&nbsp;&nbsp;};<br/>" + Text.LINE_BREAK
                + "h &lt;= i<br/>" + Text.LINE_BREAK
                + "j &gt; k"
                ;
        assertEquals(expected , actual);
    }

    @Test
    public void breakHtml() {
        String text = "Dies ist ein langer <b>Text</b> der umgebrochen und eingerückt werden soll."
                ;
        int charactersPerLine = 20;
        String actual = HtmlTool.breakHtml(text, charactersPerLine);
        String expected = "Dies ist ein langer<br/>" + Text.LINE_BREAK
                + "<b>Text</b> der<br/>" + Text.LINE_BREAK
                + "umgebrochen und<br/>" + Text.LINE_BREAK
                + "eingerückt werden<br/>" + Text.LINE_BREAK
                + "soll."
                ;
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlOnlyLtGt() {
        String input = "&lt;1&gt;";
        String actual = HtmlTool.decodeHtmlOnlyLtGt(input);
        String expected = "<1>";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlOnlyLtGt() {
        String input = "<1>";
        String actual = HtmlTool.encodeHtmlOnlyLtGt(input);
        String expected = "&lt;1&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlAmpLtAndGt() {
        String input = "&lt;1&gt;";
        String actual = HtmlTool.decodeHtmlAmpLtAndGt(input);
        String expected = "<1>";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlAmpLtAndGt() {
        String input = "<1>";
        String actual = HtmlTool.encodeHtmlAmpLtAndGt(input);
        String expected = "&lt;1&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlAmpLtAndGt2() {
        String input = "&lt;p&gt;Müller GmbH &amp; Co. KG&lt;/p&gt;";
        String actual = HtmlTool.decodeHtmlAmpLtAndGt(input);
        String expected = "<p>Müller GmbH & Co. KG</p>";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlAmpLtAndGt2() {
        String input = "<p>Müller GmbH & Co. KG</p>";
        String actual = HtmlTool.encodeHtmlAmpLtAndGt(input);
        String expected = "&lt;p&gt;Müller GmbH &amp; Co. KG&lt;/p&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlLtGt() {
        String input = "<1>";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "&lt;1&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlUmlaute() {
        String input = "Über den ärgsten Österreicher weiß sich "
                + "Zimönchen Äpfeli überhaupt keinen Rat.";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "&Uuml;ber&nbsp;den&nbsp;&auml;rgsten&nbsp;&Ouml;sterreicher&nbsp;"
                + "wei&szlig;&nbsp;sich&nbsp;Zim&ouml;nchen&nbsp;&Auml;pfeli&nbsp;"
                + "&uuml;berhaupt&nbsp;keinen&nbsp;Rat.";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlQuotation() {
        String input = "\"";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "&quot;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlStrasse() {
        String input = "Straße";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "Stra&szlig;e";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlThStrasse() {
        String input = "<th>Straße</th>";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "&lt;th&gt;Stra&szlig;e&lt;/th&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlThStrasseWithSpacesInFront() {
        int numberOfSpaces = 20;
        String decodedSpaces = Text.multipleString(" ", numberOfSpaces);
        String encodedSpaces = Text.multipleString("&nbsp;", numberOfSpaces);
        String input = decodedSpaces + "<th>Straße</th>";
        String actual = HtmlTool.encodeHtml(input);
        String expected = encodedSpaces + "&lt;th&gt;Stra&szlig;e&lt;/th&gt;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeMixed() {
        String input = "ent enthält '<change_date>'";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "ent&nbsp;enth&auml;lt&nbsp;&apos;&lt;change_date&gt;&apos;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeMixedWithoutSpaces() {
        String input = "ent enthält '<change_date>'";
        String actual = HtmlTool.encodeHtmlWithoutSpaces(input);
        String expected = "ent enth&auml;lt &apos;&lt;change_date&gt;&apos;";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlExampleHtml() {
        Reader readerHtml = new FineFileReader(path + "example.html", Charset.UTF_8);
        Reader readerTxt = new FineFileReader(path + "example.txt", Charset.UTF_8);
        String input = readerHtml.readFileToString();
        String expected = readerTxt.readFileToString();
        readerHtml.close();
        readerTxt.close();
        String actual = HtmlTool.encodeHtml(input);
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlHeinsUndPartner() {
        String input = "Heins & Partner GmbH";
        String actual = HtmlTool.encodeHtml(input);
        String expected = "Heins&nbsp;&amp;&nbsp;Partner&nbsp;GmbH";
        assertEquals(expected , actual);
    }

    @Test
    public void encodeHtmlHeinsUndPartnerWithoutSpaces() {
        String input = "Heins & Partner GmbH";
        String actual = HtmlTool.encodeHtmlWithoutSpaces(input);
        String expected = "Heins &amp; Partner GmbH";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlLtGt() {
        String input = "&lt;1&gt;";
        String actual = HtmlTool.decodeHtml(input);
        String expected = "<1>";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlUmlaute() {
        String input = "&Uuml;ber den &auml;rgsten &Ouml;sterreicher wei&szlig; sich "
                + "Zim&ouml;nchen &Auml;pfeli &uuml;berhaupt keinen Rat.";
        String actual = HtmlTool.decodeHtml(input);
        String expected = "Über den ärgsten Österreicher weiß sich "
                + "Zimönchen Äpfeli überhaupt keinen Rat.";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlQuotation() {
        String input = "&quot;";
        String actual = HtmlTool.decodeHtml(input);
        String expected = "\"";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlStrasse() {
        String input = "Stra&szlig;e";
        String actual = HtmlTool.decodeHtml(input);
        String expected = "Straße";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlThStrasse() {
        String input = "&lt;th&gt;Stra&szlig;e&lt;/th&gt;";
        String actual = HtmlTool.decodeHtml(input);
        String expected = "<th>Straße</th>";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlThStrasseWithSpacesInFront() {
        int numberOfSpaces = 20;
        String decodedSpaces = Text.multipleString(" ", numberOfSpaces);
        String encodedSpaces = Text.multipleString("&nbsp;", numberOfSpaces);
        String input = encodedSpaces + "&lt;th&gt;Stra&szlig;e&lt;/th&gt;";
        String actual = HtmlTool.decodeHtml(input);
        String expected = decodedSpaces + "<th>Straße</th>";
        assertEquals(expected , actual);
    }

    @Test
    public void decodeHtmlExampleHtml() {
        Reader readerHtml = new FineFileReader(path + "example.html", Charset.UTF_8);
        Reader readerTxt = new FineFileReader(path + "example.txt");
        String input = readerTxt.readFileToString();
        String expected = readerHtml.readFileToString();
        readerHtml.close();
        readerTxt.close();
        String actual = HtmlTool.decodeHtml(input);
        assertEquals(expected , actual);
    }

    @Test
    public void tt() {
        String input = "foo";
        String actual = HtmlTool.tt(input);
        String expected = "<tt>" + input + "</tt>";
        assertEquals(expected , actual);
    }

    @Test
    public void blockquote() {
        String input = "foo";
        String actual = HtmlTool.blockquote(input);
        String expected = "<blockquote>" + input + "</blockquote>";
        assertEquals(expected , actual);
    }

    @Test
    public void blockquoteTt() {
        String input = "foo";
        String actual = HtmlTool.blockquoteTt(input);
        String expected = "<blockquote><tt>" + input + "</tt></blockquote>";
        assertEquals(expected , actual);
    }

    @Test
    public void removeHtmlCommentsWithoutComment() {
        String content = "<html><body>foo</body></html>";
        String actual = HtmlTool.removeHtmlComments(content);
        String expected = content;
        assertEquals(expected , actual);
    }

    @Test
    public void removeHtmlCommentsWithSimpleComment() {
        String content = "<html><body><!--foo--></body></html>";
        String actual = HtmlTool.removeHtmlComments(content);
        String expected = "<html><body></body></html>";
        assertEquals(expected , actual);
    }

    @Test
    public void removeHtmlCommentsDoNotRemoveWrongComment() {
        String content = "<html><body><!--></body></html>";
        String actual = HtmlTool.removeHtmlComments(content);
        String expected = content;
        assertEquals(expected , actual);
    }

    @Test
    public void removeHtmlCommentsNestedComment() {
        String content = "<html><body><!--foo bar <!-- baz -->--></body></html>";
        String actual = HtmlTool.removeHtmlComments(content);
        String expected = "<html><body>--></body></html>";
        assertEquals(expected , actual);
    }

    @Test
    public void fillWithSpacesNormalCase() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpaces(input, 5);
        String expected = "abc&nbsp;&nbsp;";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesZeroNumber() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpaces(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesNegativeNumber() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpaces(input, -7);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesNumberEqulasInputLength() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpaces(input, input.length());
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesEmptyInput() {
        String input = "";
        String actual = HtmlTool.fillWithSpaces(input, 3);
        String expected = "&nbsp;&nbsp;&nbsp;";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNormalCase() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpacesAtFront(input, 5);
        String expected = "&nbsp;&nbsp;abc";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontZeroNumber() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpacesAtFront(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNegativeNumber() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpacesAtFront(input, -7);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNumberEqulasInputLength() {
        String input = "abc";
        String actual = HtmlTool.fillWithSpacesAtFront(input, input.length());
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontEmptyInput() {
        String input = "";
        String actual = HtmlTool.fillWithSpacesAtFront(input, 3);
        String expected = "&nbsp;&nbsp;&nbsp;";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromEmptyString() {
        String input = "";
        String actual = HtmlTool.removeLinks(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithoutLinks() {
        String input = "<html><head>bla</head><body><h1>foo</h1> bar baz</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithOneNormalLink() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithTwoNormalLink() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + "<a href=\"foobar\">Hier steht was lustiges!</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithOneNormalLinkWithLineBreakAfterOpeningTag() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">\nbaz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithOneNormalLinkWithLineBreakInOpeningTag() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\n\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithOneNormalLinkWithLineBreakInOpeningTagAfterA() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a\n href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksFromStringWithOneNormalLinkWithLineBreakInOpeningTagAfterEqual() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href\n=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinks(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromEmptyString() {
        String input = "";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithoutLinks() {
        String input = "<html><head>bla</head><body><h1>foo</h1> bar baz</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithOneNormalLink() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithSomeNormalLinks() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + "ich "
                + "<A HREF=\"foobar\">Hier steht was lustiges!</a>"
                + "bin "
                + "<a HREF=\"foobar\">Hier steht was lustiges!</A>"
                + "zu "
                + "<a HREF=\"foobar\">Hier steht was lustiges!</A>"
                + "sehen "
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> ich bin zu sehen </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithOneNormalLinkWithLineBreakAfterOpeningTag() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">\nbaz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithOneNormalLinkWithLineBreakInOpeningTag() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\n\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithOneNormalLinkWithLineBreakInOpeningTagAfterA() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a\n href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureFromStringWithOneNormalLinkWithLineBreakInOpeningTagAfterEqual() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href\n=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecure(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromEmptyString() {
        String input = "";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithoutLinks() {
        String input = "<html><head>bla</head><body><h1>foo</h1> bar baz</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithOneNormalLink() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> baz</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithSomeNormalLinks() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">baz</a>"
                + " ich "
                + "<A HREF=\"foobar\">foo</a>"
                + " bin "
                + "<a HREF=\"foobar\">bar</A>"
                + " zu "
                + "<a HREF=\"foobar\">baz</A>"
                + " sehen "
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> baz ich foo bin bar zu baz "
                + "sehen </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithOneNormalLinkWithLBAfterOT() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\"bar\">\nbaz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> \nbaz</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithOneNormalLinkWithLBInOT() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href=\n\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> baz</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithOneNormalLinkWithLBInOTAfterA() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a\n href=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> baz</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLinksSecureButKeepLinkDescriptionFromStringWithOneNormalLinkWithLBInOTAfterE() {
        String input = "<html><head>bla</head><body><h1>foo</h1> "
                + "<a href\n=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeLinksSecureButKeepLinkDescription(input);
        String expected = "<html><head>bla</head><body><h1>foo</h1> baz</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void isHtml5KeyWordHtmlLowerCase() {
        String input = "html";
        boolean isKeyword = HtmlTool.isHtml5KeyWord(input);
        assertTrue(isKeyword);
    }

    @Test
    public void isHtml5KeyWordHtmlUpperCase() {
        String input = "HTML";
        boolean isKeyword = HtmlTool.isHtml5KeyWord(input);
        assertTrue(isKeyword);
    }

    @Test
    public void isHtml5KeyWordNoKeyword() {
        String input = "foo";
        boolean isKeyword = HtmlTool.isHtml5KeyWord(input);
        assertFalse(isKeyword);
    }

    @Test
    public void removeComments() {
        String html = "<html><head>bla</head><body>"
                + "<!--<h1>foo</h1><a href=\"bar\">baz</a>-->"
                + "</body></html>";
        String actual = HtmlTool.removeComments(html);
        String expected = "<html><head>bla</head><body></body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeCommentsWithLineBreaks() {
        String html = "<html><head>bla</head><body>"
                + "<!--\n<h1>foo</h1>\n<a href\n=\"bar\">baz</a>\n-->"
                + "</body></html>";
        String actual = HtmlTool.removeComments(html);
        String expected = "<html><head>bla</head><body></body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeCommentsNoComment() {
        String html = "<html><head>bla</head><body>"
                + "<h1>foo</h1><a href\n=\"bar\">baz</a>"
                + "</body></html>";
        String actual = HtmlTool.removeComments(html);
        String expected = html;
        assertEquals(expected, actual);
    }

    @Test
    public void removeMultipleComments() {
        String html = "<html><head>bla</head><body>"
                + "<!--<h1>foo1</h1><a href=\"bar\">baz</a>-->"
                + "<h1>foo2</h1><a href=\"bar\">baz</a>"
                + "<!--<h1>foo3</h1><a href=\"bar\">baz</a>-->"
                + "</body></html>";
        String actual = HtmlTool.removeComments(html);
        String expected = "<html><head>bla</head><body>"
                + "<h1>foo2</h1><a href=\"bar\">baz</a>"
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeWrongEndedComments() {
        String html = "<html><head>bla</head><body>"
                + "<!--<h1>foo1</h1><a href=\"bar\">baz</a>-- >"
                + "<h1>foo2</h1><a href=\"bar\">baz</a>"
                + "<!--<h1>foo3</h1><a href=\"bar\">baz</a>-->"
                + "</body></html>";
        String actual = HtmlTool.removeComments(html);
        String expected = "<html><head>bla</head><body>"
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void lowerTagNames() {
        String html = "<HTML><head>bla</head><body>"
                + "<h1>foo</h1><A href\n=\"bar\">baz</a>"
                + "</boDy></hTmL>";
        String actual = HtmlTool.lowerTagNames(html);
        String expected = "<html><head>bla</head><body>"
                + "<h1>foo</h1><a href\n=\"bar\">baz</a>"
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void lowerTagNamesButNoParameters() {
        String html = "<HTML><head>bla</head><body>"
                + "<h1>foo</h1><A HREF\n=\"bar\">baz</a>"
                + "</boDy></hTmL>";
        String actual = HtmlTool.lowerTagNames(html);
        String expected = "<html><head>bla</head><body>"
                + "<h1>foo</h1><a HREF\n=\"bar\">baz</a>"
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeImages() {
        String html = "<html><head>bla</head><body><h1>foo</h1> "
                + "<img src=\"c:/bla/fasel.jpg\">"
                + "<img src=\"c:/bla/fasel.jpg\"/>"
                + "<img src=\"c:/bla/fasel.jpg\" />"
                + "</body></html>";
        String actual = HtmlTool.removeImages(html);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeDefinitionLists() {
        String html = "<html><head>bla</head><body><h1>foo</h1> "
                + "<dl>"
                + "    <dt>Common</dt>   <dd>0.2.03</dd>"
                + "    <dt>Frontend</dt> <dd>0.3.45</dd>"
                + "    <dt>Backend</dt>  <dd>0.9.70</dd>"
                + "</dl>"
                + "</body></html>";
        String actual = HtmlTool.removeDefinitionLists(html);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeDefinitionListsWithBlankBehindOpeningTag() {
        String html = "<html><head>bla</head><body><h1>foo</h1> "
                + "<dl >"
                + "    <dt>Common</dt>   <dd>0.2.03</dd>"
                + "    <dt>Frontend</dt> <dd>0.3.45</dd>"
                + "    <dt>Backend</dt>  <dd>0.9.70</dd>"
                + "</dl>"
                + "</body></html>";
        String actual = HtmlTool.removeDefinitionLists(html);
        String expected = "<html><head>bla</head><body><h1>foo</h1> </body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeDefinitionListsOnMultipleLines() {
        String html = "<html><head>bla</head><body><h1>foo</h1>" + Text.LINE_BREAK
                + "<dl>" + Text.LINE_BREAK
                + "    <dt>Common</dt>   <dd>0.2.03</dd>" + Text.LINE_BREAK
                + "    <dt>Frontend</dt> <dd>0.3.45</dd>" + Text.LINE_BREAK
                + "    <dt>Backend</dt>  <dd>0.9.70</dd>" + Text.LINE_BREAK
                + "</dl>" + Text.LINE_BREAK
                + "</body></html>";
        String actual = HtmlTool.removeDefinitionLists(html);
        String expected = "<html><head>bla</head><body><h1>foo</h1>" + Text.LINE_BREAK
                + "" + Text.LINE_BREAK
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void removeDefinitionListsOnMultipleLinesMultipleDls() {
        String html = "<html><head>bla</head><body><h1>foo</h1>" + Text.LINE_BREAK
                + "<dl>" + Text.LINE_BREAK
                + "    <dt>Common</dt>   <dd>0.2.03</dd>" + Text.LINE_BREAK
                + "    <dt>Frontend</dt> <dd>0.3.45</dd>" + Text.LINE_BREAK
                + "    <dt>Backend</dt>  <dd>0.9.70</dd>" + Text.LINE_BREAK
                + "</dl>" + Text.LINE_BREAK
                + "<p>Bla bla bla.</p>" + Text.LINE_BREAK
                + "<dl>" + Text.LINE_BREAK
                + "    <dt>Common</dt>   <dd>0.2.03</dd>" + Text.LINE_BREAK
                + "    <dt>Frontend</dt> <dd>0.3.45</dd>" + Text.LINE_BREAK
                + "    <dt>Backend</dt>  <dd>0.9.70</dd>" + Text.LINE_BREAK
                + "</dl>" + Text.LINE_BREAK
                + "</body></html>";
        String actual = HtmlTool.removeDefinitionLists(html);
        String expected = "<html><head>bla</head><body><h1>foo</h1>" + Text.LINE_BREAK
                + "" + Text.LINE_BREAK
                + "<p>Bla bla bla.</p>" + Text.LINE_BREAK
                + "" + Text.LINE_BREAK
                + "</body></html>";
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsLowercaseScript() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("script");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<script>", "<SCRIPT>",
                "<script ", "<SCRIPT ",
                "<script\n", "<SCRIPT\n",
                "<script\r\n", "<SCRIPT\r\n",
                "<script" + Text.LINE_BREAK, "<SCRIPT" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsUppercaseScript() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("SCRIPT");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<script>", "<SCRIPT>",
                "<script ", "<SCRIPT ",
                "<script\n", "<SCRIPT\n",
                "<script\r\n", "<SCRIPT\r\n",
                "<script" + Text.LINE_BREAK, "<SCRIPT" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlEndingTagsLowercaseScript() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlEndingTags("script");
        List<String> expected = CollectionsHelper.buildListFrom("</script>", "</SCRIPT>");
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlEndingTagsUppercaseScript() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlEndingTags("SCRIPT");
        List<String> expected = CollectionsHelper.buildListFrom("</script>", "</SCRIPT>");
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsLowercaseImage() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("img");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<img>", "<IMG>",
                "<img ", "<IMG ",
                "<img\n", "<IMG\n",
                "<img\r\n", "<IMG\r\n",
                "<img" + Text.LINE_BREAK, "<IMG" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsUppercaseImage() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("IMG");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<img>", "<IMG>",
                "<img ", "<IMG ",
                "<img\n", "<IMG\n",
                "<img\r\n", "<IMG\r\n",
                "<img" + Text.LINE_BREAK, "<IMG" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsLowercaseDl() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("dl");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<dl>", "<DL>",
                "<dl ", "<DL ",
                "<dl\n", "<DL\n",
                "<dl\r\n", "<DL\r\n",
                "<dl" + Text.LINE_BREAK, "<DL" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlTagStartsUppercaseDl() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlTagStarts("DL");
        List<String> expected = CollectionsHelper.buildListFrom(
                "<dl>", "<DL>",
                "<dl ", "<DL ",
                "<dl\n", "<DL\n",
                "<dl\r\n", "<DL\r\n",
                "<dl" + Text.LINE_BREAK, "<DL" + Text.LINE_BREAK);
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlEndingTagsLowercaseDl() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlEndingTags("dl");
        List<String> expected = CollectionsHelper.buildListFrom("</dl>", "</DL>");
        assertEquals(expected, actual);
    }

    @Test
    public void createUpperAndLowerCaseHtmlEndingTagsUppercaseDl() {
        List<String> actual = HtmlTool.createUpperAndLowerCaseHtmlEndingTags("DL");
        List<String> expected = CollectionsHelper.buildListFrom("</dl>", "</DL>");
        assertEquals(expected, actual);
    }

    @Test
    public void removeHtmlMarkup() {
        String input = "<span style=\"color:ba55d3;font-weight:bold\">Birmingham</span>";
        String actual = HtmlTool.removeHtmlMarkup(input);
        String expected = "Birmingham";
        assertEquals(expected, actual);
    }

}
