package de.duehl.basics.text.xml;

/*
 * Copyright 2023 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class XmlAnalyserTest {

    @Test
    public void getLinesInTagSimple() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine bar in Blubb...",
                "    </bar>",
                "</foo>");
        List<String> tagLines = XmlAnalyser.getLinesInTag(lines, "foo");
        assertFalse(tagLines.isEmpty());
        assertEquals(5, tagLines.size());
        assertEquals(lines.get(0), tagLines.get(0));
        assertEquals(lines.get(1), tagLines.get(1));
        assertEquals(lines.get(2), tagLines.get(2));
        assertEquals(lines.get(3), tagLines.get(3));
        assertEquals(lines.get(4), tagLines.get(4));
    }

    @Test
    public void getLinesInTagSimple2() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine bar in Blubb...",
                "    </bar>",
                "</foo>");
        List<String> tagLines = XmlAnalyser.getLinesInTag(lines, "bar");
        assertFalse(tagLines.isEmpty());
        assertEquals(3, tagLines.size());
        assertEquals(lines.get(1), tagLines.get(0));
        assertEquals(lines.get(2), tagLines.get(1));
        assertEquals(lines.get(3), tagLines.get(2));
    }

    @Test
    public void getLinesInTag1() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "",
                "<foo>",
                "    <bar>",
                "        Eine bar in Blubb...",
                "    </bar>",
                "</foo>",
                "Sonst noch irgendwas...",
                "<foobar>",
                "    Bla bla...",
                "</foobar>");
        List<String> tagLines = XmlAnalyser.getLinesInTag(lines, "foo");
        assertFalse(tagLines.isEmpty());
        assertEquals(5, tagLines.size());
        assertEquals(lines.get(1), tagLines.get(0));
        assertEquals(lines.get(2), tagLines.get(1));
        assertEquals(lines.get(3), tagLines.get(2));
        assertEquals(lines.get(4), tagLines.get(3));
        assertEquals(lines.get(5), tagLines.get(4));
    }

    @Test
    public void getLinesInTagDoNotGetOtherTagsLater() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "",
                "<foo>",
                "    <bar>",
                "        Eine bar in Blubb...",
                "    </bar>",
                "</foo>",
                "Sonst noch irgendwas...",
                "<foobar>",
                "    Bla bla...",
                "    <foo>",
                "        <bar>",
                "            Eine andere Bar in Bla-Blubb...",
                "        </bar>",
                "    </foo>",
                "</foobar>"
                );
        List<String> tagLines = XmlAnalyser.getLinesInTag(lines, "foo");
        assertFalse(tagLines.isEmpty());
        assertEquals(5, tagLines.size());
        assertEquals(lines.get(1), tagLines.get(0));
        assertEquals(lines.get(2), tagLines.get(1));
        assertEquals(lines.get(3), tagLines.get(2));
        assertEquals(lines.get(4), tagLines.get(3));
        assertEquals(lines.get(5), tagLines.get(4));
    }

    @Test (expected = RuntimeException.class)
    public void getLinesInTagNotWrongTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foobar>",
                "    Bla bla...",
                "</foobar>");
        XmlAnalyser.getLinesInTag(lines, "foo");
    }

    @Test
    public void getLinesInTagWithParameter() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bla=\"1\" blubb=\"hugo\">",
                "    <bar>",
                "        Eine bar in Blubb...",
                "    </bar>",
                "</foo>");
        List<String> tagLines = XmlAnalyser.getLinesInTag(lines, "foo");
        assertFalse(tagLines.isEmpty());
        assertEquals(5, tagLines.size());
        assertEquals(lines.get(0), tagLines.get(0));
        assertEquals(lines.get(1), tagLines.get(1));
        assertEquals(lines.get(2), tagLines.get(2));
        assertEquals(lines.get(3), tagLines.get(3));
        assertEquals(lines.get(4), tagLines.get(4));
    }


    @Test (expected = RuntimeException.class)
    public void getFullTagInLineNoEndTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>bar-content</baz>",
                "</foo>");
        XmlAnalyser.getFullTagInLine(lines, "bar");
    }

    @Test (expected = RuntimeException.class)
    public void getFullTagInLineNoStartTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <baz>bar-content</bar>",
                "</foo>");
        XmlAnalyser.getFullTagInLine(lines, "bar");
    }

    @Test
    public void getFullTagInLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>bar-content</bar>",
                "</foo>");
        String line = XmlAnalyser.getFullTagInLine(lines, "bar");
        assertEquals(lines.get(1), line);
    }

    @Test (expected = RuntimeException.class)
    public void getContentInTagNoEndTag() {
        XmlAnalyser.getContentInTag("<foo>Hier!</bar>", "foo");
    }

    @Test (expected = RuntimeException.class)
    public void getContentInTagNoStartTag() {
        XmlAnalyser.getContentInTag("<bar>Hier!</foo>", "foo");
    }

    @Test (expected = RuntimeException.class)
    public void getContentInTagNoBiggerAfterStartTag() {
        XmlAnalyser.getContentInTag("<foo Hier!</foo>", "foo");
    }

    @Test (expected = RuntimeException.class)
    public void getContentInTagNoBiggerAfterStartTag2() {
        XmlAnalyser.getContentInTag("<foo param=\"1\" Hier!</foo>", "foo");
    }

    @Test
    public void getContentInTagWithoutParameter() {
        String actual = XmlAnalyser.getContentInTag("<foo>Hier!</foo>", "foo");
        String expected = "Hier!";
        assertEquals(expected, actual);
    }

    @Test
    public void getContentInTagWithParameter() {
        String actual = XmlAnalyser.getContentInTag("<foo param=\"1\">Hier!</foo>", "foo");
        String expected = "Hier!";
        assertEquals(expected, actual);
    }

    @Test
    public void getContentInFullTagLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>bar-content</bar>",
                "</foo>");
        String content = XmlAnalyser.getContentInFullTagLine(lines, "bar");
        assertEquals("bar-content", content);
    }

    @Test
    public void getContentInFullTagLineErrorToBlankSuccess() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>bar-content</bar>",
                "</foo>");
        String content = XmlAnalyser.getContentInFullTagLineErrorToBlank(lines, "bar");
        assertEquals("bar-content", content);
    }

    @Test
    public void getContentInFullTagLineErrorToBlankEmptyInput() {
        List<String> lines = CollectionsHelper.buildListFrom();
        String actual = XmlAnalyser.getContentInFullTagLineErrorToBlank(lines, "bar");
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getContentInFullTagLineErrorToBlankNotContainedInInput() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>bar-content</bar>",
                "</foo>");
        String actual = XmlAnalyser.getContentInFullTagLineErrorToBlank(lines, "baz");
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getMultipleLinesInTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> listOfLists = XmlAnalyser.getMultipleLinesInTag(lines, "bar");
        assertEquals(3, listOfLists.size());

        List<String> barLines1 = listOfLists.get(0);
        assertTrue(3 == barLines1.size());
        assertEquals(lines.get(1), barLines1.get(0));
        assertEquals(lines.get(2), barLines1.get(1));
        assertEquals(lines.get(3), barLines1.get(2));

        List<String> barLines2 = listOfLists.get(1);
        assertTrue(3 == barLines2.size());
        assertEquals(lines.get(4), barLines2.get(0));
        assertEquals(lines.get(5), barLines2.get(1));
        assertEquals(lines.get(6), barLines2.get(2));

        List<String> barLines3 = listOfLists.get(2);
        assertTrue(3 == barLines3.size());
        assertEquals(lines.get(7), barLines3.get(0));
        assertEquals(lines.get(8), barLines3.get(1));
        assertEquals(lines.get(9), barLines3.get(2));
    }

    @Test (expected = RuntimeException.class)
    public void getMultipleLinesInTagEmpty() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "</foo>");
        XmlAnalyser.getMultipleLinesInTag(lines, "bar");
    }

    @Test
    public void getMultipleLinesInTagAllowEmpty() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "</foo>");
        List<List<String>> listOfLists = XmlAnalyser.getMultipleLinesInTagAllowEmptyList(lines, "bar");
        assertEquals(0, listOfLists.size());
    }

    @Test
    public void getMultipleLinesInTagWithClosingTagOnSameLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>Eine Bar in Versmold</bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> listOfLists = XmlAnalyser.getMultipleLinesInTag(lines, "bar");
        assertEquals(3, listOfLists.size());

        List<String> barLines1 = listOfLists.get(0);
        assertTrue(3 == barLines1.size());
        assertEquals(lines.get(1), barLines1.get(0));
        assertEquals(lines.get(2), barLines1.get(1));
        assertEquals(lines.get(3), barLines1.get(2));

        List<String> barLines2 = listOfLists.get(1);
        assertTrue(1 == barLines2.size());
        assertEquals(lines.get(4), barLines2.get(0));

        List<String> barLines3 = listOfLists.get(2);
        assertTrue(3 == barLines3.size());
        assertEquals(lines.get(5), barLines3.get(0));
        assertEquals(lines.get(6), barLines3.get(1));
        assertEquals(lines.get(7), barLines3.get(2));
    }

    @Test
    public void getMultipleLinesInTagWithoutOpeningAndClosingTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> listOfLists =
                XmlAnalyser.getMultipleLinesInTagWithoutOpeningAndClosingTag(lines, "bar");
        assertEquals(3, listOfLists.size());

        List<String> barLines1 = listOfLists.get(0);
        assertEquals(1, barLines1.size());
        assertEquals(lines.get(2), barLines1.get(0));

        List<String> barLines2 = listOfLists.get(1);
        assertEquals(1, barLines2.size());
        assertEquals(lines.get(5), barLines2.get(0));

        List<String> barLines3 = listOfLists.get(2);
        assertEquals(1, barLines3.size());
        assertEquals(lines.get(8), barLines3.get(0));
    }

    @Test
    public void getMultipleLinesInTagWithoutOpeningAndClosingTagWithClosingTagOnSameLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>Eine Bar in Versmold</bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> listOfLists =
                XmlAnalyser.getMultipleLinesInTagWithoutOpeningAndClosingTag(lines, "bar");
        assertEquals(3, listOfLists.size());

        List<String> barLines1 = listOfLists.get(0);
        assertEquals(1, barLines1.size());
        assertEquals(lines.get(2), barLines1.get(0));

        List<String> barLines2 = listOfLists.get(1);
        assertEquals(1, barLines2.size());
        assertEquals("Eine Bar in Versmold", barLines2.get(0));

        List<String> barLines3 = listOfLists.get(2);
        assertEquals(1, barLines3.size());
        assertEquals(lines.get(6), barLines3.get(0));
    }

    @Test (expected = RuntimeException.class)
    public void getParametersFromOpeningTagNoStartTag() {
        XmlAnalyser.getParametersFromOpeningTag("< foo param=\"1\">Hier!</foo>", "foo");
    }

    @Test (expected = RuntimeException.class)
    public void getParametersFromOpeningTagNoBiggerAfterStartTag() {
        XmlAnalyser.getParametersFromOpeningTag("<foo param=\"1\"", "foo");
    }

    @Test
    public void getParametersFromOpeningTag() {
        List<NamedXmlParameter> list = XmlAnalyser.getParametersFromOpeningTag(
                "<foo param=\"1\">Hier!</foo>", "foo");
        assertEquals(1, list.size());

        NamedXmlParameter param1 = list.get(0);
        assertEquals("param", param1.getName());
        assertEquals("1", param1.getContent());
    }

    @Test
    public void getParametersFromOpeningTag2() {
        List<NamedXmlParameter> list = XmlAnalyser.getParametersFromOpeningTag(
                "<foo param=\"1\" blubb=\"zwei\">", "foo");
        assertEquals(2, list.size());

        NamedXmlParameter param1 = list.get(0);
        assertEquals("param", param1.getName());
        assertEquals("1", param1.getContent());

        NamedXmlParameter param2 = list.get(1);
        assertEquals("blubb", param2.getName());
        assertEquals("zwei", param2.getContent());
    }

    @Test
    public void determineLineWithOpeningTagInFirstLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        String actual = XmlAnalyser.determineLineWithOpeningTag(lines, "foo");
        String expected = lines.get(0);
        assertEquals(expected, actual);
    }

    @Test
    public void determineLineWithOpeningTagInOtherLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        String actual = XmlAnalyser.determineLineWithOpeningTag(lines, "bar");
        String expected = lines.get(1);
        assertEquals(expected, actual);
    }

    @Test (expected = RuntimeException.class)
    public void determineLineWithOpeningTagNoTagFound() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        XmlAnalyser.determineLineWithOpeningTag(lines, "haben wir gar nicht!");
    }

    @Test
    public void existsOpeningTagInFirstLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertTrue(XmlAnalyser.existsOpeningTag(lines, "foo"));
    }

    @Test
    public void existsOpeningTagInOtherLine() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertTrue(XmlAnalyser.existsOpeningTag(lines, "bar"));
    }

    @Test
    public void existsOpeningTagInLine() {
        String line = "    <bar>";
        assertTrue(XmlAnalyser.existsOpeningTag(line, "bar"));
        assertFalse(XmlAnalyser.existsOpeningTag(line, "foo"));
    }

    @Test
    public void existsClosingTagInLine() {
        String line = "    </bar>";
        assertTrue(XmlAnalyser.existsClosingTag(line, "bar"));
        assertFalse(XmlAnalyser.existsClosingTag(line, "foo"));
    }

    @Test
    public void existsOpeningTagNoTagFound() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertFalse(XmlAnalyser.existsOpeningTag(lines, "haben wir gar nicht!"));
    }

    @Test
    public void getFirstOpeningTagIndex() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertEquals( 1, XmlAnalyser.getFirstOpeningTagIndex(lines, "bar"));
        assertEquals( 0, XmlAnalyser.getFirstOpeningTagIndex(lines, "foo"));
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "baz"));
    }

    @Test
    public void getFirstOpeningTagIndex2() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">bar</foo>",
                "<foo>baz</foo>"
                );
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "bar"));
        assertEquals( 0, XmlAnalyser.getFirstOpeningTagIndex(lines, "foo"));
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "baz"));
    }

    @Test
    public void getFirstOpeningTagIndex3() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo>baz</foo>",
                "<foo bar=\"1\" baz=\"zwei\">bar</foo>"
                );
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "bar"));
        assertEquals( 0, XmlAnalyser.getFirstOpeningTagIndex(lines, "foo"));
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "baz"));
    }

    @Test
    public void getFirstOpeningTagIndexWithStartIndex() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        int startIndex = 5;
        assertEquals( 7, XmlAnalyser.getFirstOpeningTagIndex(lines, "bar", startIndex));
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "foo", startIndex));
        assertEquals(-1, XmlAnalyser.getFirstOpeningTagIndex(lines, "baz", startIndex));
    }

    @Test
    public void getFirstClosingTagIndex() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertEquals( 3, XmlAnalyser.getFirstClosingTagIndex(lines, "bar"));
        assertEquals(10, XmlAnalyser.getFirstClosingTagIndex(lines, "foo"));
        assertEquals(-1, XmlAnalyser.getFirstClosingTagIndex(lines, "baz"));
    }

    @Test
    public void getFirstClosingTagIndexWithStartIndex() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertEquals( 6, XmlAnalyser.getFirstClosingTagIndex(lines, "bar", 4));
        assertEquals(10, XmlAnalyser.getFirstClosingTagIndex(lines, "foo", 4));
        assertEquals(-1, XmlAnalyser.getFirstClosingTagIndex(lines, "baz", 4));
    }

    @Test
    public void getAllOpeningTagIndices() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<Integer> indices = XmlAnalyser.getAllOpeningTagIndices(lines, "bar");
        assertEquals(3, indices.size());
        assertEquals(Integer.valueOf(1), indices.get(0));
        assertEquals(Integer.valueOf(4), indices.get(1));
        assertEquals(Integer.valueOf(7), indices.get(2));
    }

    @Test
    public void getAllOpeningTagIndicesNotExisting() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<Integer> indices = XmlAnalyser.getAllOpeningTagIndices(lines, "gibts-nicht");
        assertEquals(0, indices.size());
    }

    @Test
    public void cutAtOpeningTagWithoutContainedTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> partsList = XmlAnalyser.cutAtOpeningTag(lines, "gibts-nicht");
        assertEquals(1, partsList.size());
        assertEquals(lines, partsList.get(0));
    }

    @Test
    public void cutAtOpeningTagSingleLines() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>Eine Bar in Bielefeld</bar>",
                "    <bar>Eine Bar in Versmold</bar>",
                "    <bar>Eine Bar in Steinhagen</bar>",
                "</foo>");
        List<List<String>> partsList = XmlAnalyser.cutAtOpeningTag(lines, "bar");
        assertEquals(4, partsList.size());
        assertEquals(CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">"), partsList.get(0));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>Eine Bar in Bielefeld</bar>"), partsList.get(1));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>Eine Bar in Versmold</bar>"), partsList.get(2));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>Eine Bar in Steinhagen</bar>",
                "</foo>"), partsList.get(3));
    }

    @Test
    public void cutAtOpeningTagDirectSingleLines() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<bar>Eine Bar in Bielefeld</bar>",
                "<bar>Eine Bar in Versmold</bar>",
                "<bar>Eine Bar in Steinhagen</bar>");
        List<List<String>> partsList = XmlAnalyser.cutAtOpeningTag(lines, "bar");
        assertEquals(3, partsList.size());
        assertEquals(CollectionsHelper.buildListFrom(
                "<bar>Eine Bar in Bielefeld</bar>"), partsList.get(0));
        assertEquals(CollectionsHelper.buildListFrom(
                "<bar>Eine Bar in Versmold</bar>"), partsList.get(1));
        assertEquals(CollectionsHelper.buildListFrom(
                "<bar>Eine Bar in Steinhagen</bar>"), partsList.get(2));
    }

    @Test
    public void cutAtOpeningTagEmptyLines() {
        List<String> lines = CollectionsHelper.buildListFrom();
        List<List<String>> partsList = XmlAnalyser.cutAtOpeningTag(lines, "bar");
        assertEquals(1, partsList.size());
        assertEquals(CollectionsHelper.buildListFrom(), partsList.get(0));
    }

    @Test
    public void cutAtOpeningTag1() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<List<String>> partsList = XmlAnalyser.cutAtOpeningTag(lines, "bar");
        assertEquals(4, partsList.size());
        assertEquals(CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">"), partsList.get(0));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>"), partsList.get(1));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>"), partsList.get(2));
        assertEquals(CollectionsHelper.buildListFrom(
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>"), partsList.get(3));
    }

    @Test
    public void getInnerPartIfStartsAndEndsWithTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<String> innerPartList = XmlAnalyser.getInnerPartIfStartsAndEndsWithTag(lines, "foo");
        assertEquals(9, innerPartList.size());
        assertEquals(lines.get(1), innerPartList.get(0));
        assertEquals(lines.get(2), innerPartList.get(1));
        assertEquals(lines.get(3), innerPartList.get(2));
        assertEquals(lines.get(4), innerPartList.get(3));
        assertEquals(lines.get(5), innerPartList.get(4));
        assertEquals(lines.get(6), innerPartList.get(5));
        assertEquals(lines.get(7), innerPartList.get(6));
        assertEquals(lines.get(8), innerPartList.get(7));
        assertEquals(lines.get(9), innerPartList.get(8));
    }

    @Test
    public void getInnerPartIfStartsAndEndsWithTagWithNotContainedTag() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        List<String> innerPartList = XmlAnalyser.getInnerPartIfStartsAndEndsWithTag(lines,
                "gibt-es-nicht");
        assertEquals(0, innerPartList.size());
    }

    @Test
    public void getInnerPartIfStartsAndEndsWithTagWithEmptyLines() {
        List<String> lines = CollectionsHelper.buildListFrom();
        assertThrows(RuntimeException.class, () ->
                XmlAnalyser.getInnerPartIfStartsAndEndsWithTag(lines,"foo"));
    }

    @Test
    public void linesStartsWithTagCorrect() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertTrue(XmlAnalyser.linesStartsWithTag(lines, "foo"));
    }

    @Test
    public void linesStartsWithTagWrong() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertFalse(XmlAnalyser.linesStartsWithTag(lines, "bar"));
    }

    @Test
    public void linesEndsWithTagCorrect() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertTrue(XmlAnalyser.linesEndsWithTag(lines, "foo"));
    }

    @Test
    public void linesEndsWithTagWrong() {
        List<String> lines = CollectionsHelper.buildListFrom(
                "<foo bar=\"1\" baz=\"zwei\">",
                "    <bar>",
                "        Eine Bar in Bielefeld",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Versmold",
                "    </bar>",
                "    <bar>",
                "        Eine Bar in Steinhagen",
                "    </bar>",
                "</foo>");
        assertFalse(XmlAnalyser.linesEndsWithTag(lines, "bar"));
    }

}
