package de.duehl.basics.date;

import static org.junit.Assert.*;

import org.junit.Test;

import de.duehl.basics.date.MutualDate;
import de.duehl.basics.date.Weekday;

@SuppressWarnings("deprecation")
public class MutualDateTest {

    @Test
    public void create1() {
        MutualDate date = new MutualDate();
        assertNotNull(date);
    }

    @Test
    public void create2() {
        MutualDate date = new MutualDate(1, 2, 3);
        assertNotNull(date);
    }

    @Test
    public void create3() {
        MutualDate date = new MutualDate("foo");
        assertNotNull(date);
    }

    @Test
    public void createFromWrongString() {
        MutualDate date = new MutualDate("foo");

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(0, day);
        assertEquals(0, month);
        assertEquals(0, year);
    }

    @Test
    public void createFromString01() {
        MutualDate date = new MutualDate("11. Februar 1967");

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(11,   day  );
        assertEquals(2,    month);
        assertEquals(1967, year );
    }

    @Test
    public void createFromString02() {
        MutualDate date = new MutualDate("11.02.1967");

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(11,   day  );
        assertEquals(2,    month);
        assertEquals(1967, year );
    }

    @Test
    public void createFromString03() {
        MutualDate date = new MutualDate("Wed Jan 18 12:01:50 CET 2012");
        // dow mon dd hh:mm:ss zzz yyyy

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(18,   day  );
        assertEquals(1,    month);
        assertEquals(2012, year );
    }


    @Test
    public void leapYear001() {
        MutualDate date = new MutualDate(29, 2, 1899);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear002() {
        MutualDate date = new MutualDate(29, 2, 1900);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear003() {
        MutualDate date = new MutualDate(29, 2, 1904);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear004() {
        MutualDate date = new MutualDate(29, 2, 1908);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear005() {
        MutualDate date = new MutualDate(29, 2, 1996);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear006() {
        MutualDate date = new MutualDate(29, 2, 2000);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear007() {
        MutualDate date = new MutualDate(29, 2, 2001);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear008() {
        MutualDate date = new MutualDate(29, 2, 2012);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void validDate001() {
        MutualDate date = new MutualDate(29, 2, 2011);
        assertFalse(date.isValid());
    }

    @Test
    public void validDate002() {
        MutualDate date = new MutualDate(29, 2, 2012);
        assertTrue(date.isValid());
    }

    @Test
    public void validDate003() {
        MutualDate date = new MutualDate(4, 1, 2012);
        assertTrue(date.isValid());
    }

    @Test
    public void validDate004() {
        MutualDate date = new MutualDate(32, 1, 2012);
        assertFalse(date.isValid());
    }

    @Test
    public void validDate005() {
        assertFalse(new MutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate006() {
        assertFalse(new MutualDate( 0,  1, 2012).isValid());
    }

    @Test
    public void validDate007() {
        assertFalse(new MutualDate( 2,  4,  999).isValid());
    }

    @Test
    public void validDate008() {
        assertFalse(new MutualDate( 1,  5,10000).isValid());
    }

    @Test
    public void validDate009() {
        assertFalse(new MutualDate(31,  6, 2012).isValid());
    }

    @Test
    public void validDate010() {
        assertFalse(new MutualDate(31,  9, 2012).isValid());
    }

    @Test
    public void validDate011() {
        assertFalse(new MutualDate(31, 11, 2012).isValid());
    }

    @Test
    public void validDate012() {
        assertFalse(new MutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate013() {
        assertFalse(new MutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate014() {
        assertFalse(new MutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate015() {
        assertFalse(new MutualDate(30,  2, 2012).isValid());
    }

    @Test
    public void validDate016() {
        assertFalse(new MutualDate(29,  2, 2011).isValid());
    }

    @Test
    public void validDate017() {
        assertFalse(new MutualDate("Unsinn").isValid());
    }

    @Test
    public void validDateZero01() {
        assertFalse(new MutualDate(29,  2, 2011).isValidWithYearZero());
    }

    @Test
    public void validDateZero02() {
        assertFalse(new MutualDate(29,  1, -1).isValidWithYearZero());
    }

    @Test
    public void validDateZero03() {
        assertTrue(new MutualDate(29,  1, 0).isValidWithYearZero());
    }

    @Test
    public void calculateDayDifference001() {
        MutualDate date1 = new MutualDate(4, 1, 2012);
        int abstand = date1.calculateDayDifference(date1);
        assertEquals(0, abstand);
    }

    @Test
    public void calculateDayDifference002() {
        MutualDate date1 = new MutualDate(4, 1, 2012);
        MutualDate date2 = new MutualDate(17, 1, 2012);
        int abstand = date1.calculateDayDifference(date2);
        assertEquals(13, abstand);
    }

    @Test
    public void calculateDayDifference003() {
        MutualDate date1 = new MutualDate(4, 1, 2012);
        MutualDate date2 = new MutualDate(17, 1, 2012);
        int abstand = date2.calculateDayDifference(date1);
        assertEquals(-13, abstand);
    }

    @Test
    public void calculateDayDifference004() {
        MutualDate date1 = new MutualDate(4, 1, 2011);
        MutualDate date2 = new MutualDate(4, 1, 2012);
        int abstand = date1.calculateDayDifference(date2);
        assertEquals(365, abstand);
    }

    @Test
    public void calculateDayDifference005() {
        MutualDate date1 = new MutualDate(4, 1, 2012);
        MutualDate date2 = new MutualDate(4, 1, 2013);
        int abstand = date1.calculateDayDifference(date2);
        assertEquals(366, abstand);
    }

    @Test
    public void addDays001() {
        MutualDate date = new MutualDate(4, 1, 2012);
        date.addDays(1);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(5,    day  );
        assertEquals(1,    month);
        assertEquals(2012, year );
    }

    @Test
    public void addDays002() {
        MutualDate date = new MutualDate(1, 5, 2012);
        date.addDays(-1);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(30,   day  );
        assertEquals(4,    month);
        assertEquals(2012, year );
    }

    @Test
    public void addDays003() {
        MutualDate date = new MutualDate(1, 5, 2012);
        date.addDays(-4);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(27,   day  );
        assertEquals(4,    month);
        assertEquals(2012, year );
    }

    @Test
    public void addDays004() {
        MutualDate date = new MutualDate(31, 5, 2012);
        date.addDays(1);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(1,    day  );
        assertEquals(6,    month);
        assertEquals(2012, year );
    }

    @Test
    public void addDays005() {
        MutualDate date = new MutualDate(1, 1, 2011);
        date.addDays(365);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(1,    day  );
        assertEquals(1,    month);
        assertEquals(2012, year );
    }

    @Test
    public void addDays006() {
        MutualDate date = new MutualDate(1, 1, 2012);
        date.addDays(365);

        int day   = date.getDay();
        int month = date.getMonth();
        int year  = date.getYear();

        assertEquals(31,   day  );
        assertEquals(12,   month);
        assertEquals(2012, year );
    }

    @Test
    public void addDaysAndCalculate() {
        /* Hier wird getestet, ob addDays und calculateDayDifference zueinander
         * passen, indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    MutualDate date1 = new MutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            MutualDate date2 = date1.copy();
                            date2.addDays(abstand);
                            int abstand2 = date1.calculateDayDifference(date2);
                            assertEquals(abstand, abstand2);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid1() {
        /* Hier wird getestet, ob alle Tage zwischen 1 und 28 für alle
         * Datumswerte zwischen dem 1.1.1895 und dem 28.12.2020 valide sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=28; ++day) {
                    MutualDate date = new MutualDate(day, month, year);
                    assertTrue(date.isValid());
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid2() {
        /* Hier wird getestet, ob alle 29. und 30. für alle Datumswerte
         * zwischen dem 29.1.1895 und dem 30.12.2020 mit Ausnahme des Februars
         * valide sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                if (month != 2) {
                    for (int day=29; day<=30; ++day) {
                        MutualDate date = new MutualDate(day, month, year);
                        assertTrue(date.isValid());
                    }
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid3() {
        /* Hier wird getestet, ob alle 31. für alle passenden Monate
         * zwischen dem 31.1.1895 und dem 31.12.2020 valide sind.
         */
        int day = 31;
        for (int year=1895; year<=2020; ++year) {
            MutualDate date1  = new MutualDate(day,  1, year);
            MutualDate date3  = new MutualDate(day,  3, year);
            MutualDate date5  = new MutualDate(day,  5, year);
            MutualDate date7  = new MutualDate(day,  7, year);
            MutualDate date8  = new MutualDate(day,  8, year);
            MutualDate date10 = new MutualDate(day, 10, year);
            MutualDate date12 = new MutualDate(day, 12, year);
            assertTrue(date1.isValid());
            assertTrue(date3.isValid());
            assertTrue(date5.isValid());
            assertTrue(date7.isValid());
            assertTrue(date8.isValid());
            assertTrue(date10.isValid());
            assertTrue(date12.isValid());
        }
    }

    @Test
    public void mupltipleDaysInvalid1() {
        /* Hier wird getestet, ob alle Tage zwischen -4. und 0. für alle
         * Datumswerte zwischen dem Januar 1895 und dem Dezember 2020 invalide
         * sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=-4; day<=0; ++day) {
                    MutualDate date = new MutualDate(day, month, year);
                    assertFalse(date.isValid());
                }
            }
        }
    }

    @Test
    public void mupltipleDaysInvalid2() {
        /* Hier wird getestet, ob alle Tage zwischen 32. und 45. für alle
         * Datumswerte zwischen dem Januar 1895 und dem Dezember 2020 invalide
         * sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=32; day<=45; ++day) {
                    MutualDate date = new MutualDate(day, month, year);
                    assertFalse(date.isValid());
                }
            }
        }
    }


    @Test
    public void dayOfTheWeek001() {
        MutualDate date = new MutualDate(14, 7, 1789); // Erstürmung der Bastille
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.TUESDAY, weekday);
    }

    @Test
    public void dayOfTheWeek002() {
        MutualDate date = new MutualDate(23, 5, 1949); // Gründung der Bundesrepublik
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.MONDAY, weekday);
    }

    @Test
    public void dayOfTheWeek003() {
        MutualDate date = new MutualDate(18, 1, 1892); // * Oliver Hardy
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.MONDAY, weekday);
    }

    @Test
    public void dayOfTheWeek004() {
        MutualDate date = new MutualDate(9, 11, 1989); // Mauerfall
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek005() {
        MutualDate date = new MutualDate(24, 4, 1982); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.SATURDAY, weekday);
    }

    @Test
    public void dayOfTheWeek006() {
        MutualDate date = new MutualDate(18, 9, 1783); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek007() {
        MutualDate date = new MutualDate(1, 1, 2000); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.SATURDAY, weekday);
    }

    @Test
    public void dayOfTheWeek008() {
        MutualDate date = new MutualDate(19, 6, 2054); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.FRIDAY, weekday);
    }

    @Test
    public void dayOfTheWeek009() {
        MutualDate date = new MutualDate(5, 1, 2012); // heute
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek010() {
        MutualDate date = new MutualDate(20, 12, 2012); // 70. Geb. Gerd
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void isWorkDay001() {
        MutualDate date = new MutualDate(5, 1, 2012); // heute
        boolean isWorkDay = date.isWorkDay();
        assertTrue(isWorkDay);
    }

    @Test
    public void isWorkDay002() {
        MutualDate date = new MutualDate(6, 1, 2012); // morgen
        boolean isWorkDay = date.isWorkDay();
        assertTrue(isWorkDay);
    }

    @Test
    public void isWorkDay003() {
        MutualDate date = new MutualDate(7, 1, 2012); // übermorgen
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay004() {
        MutualDate date = new MutualDate(31, 12, 2011);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay005() {
        MutualDate date = new MutualDate(31, 12, 2010);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay006() {
        MutualDate date = new MutualDate(1, 1, 2013);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay007() {
        MutualDate date = new MutualDate(17, 5, 2012); // Christi Himmelfahrt
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay008() {
        MutualDate date = new MutualDate(16, 3, 2013); // Samstag
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay009() {
        MutualDate date = new MutualDate(17, 3, 2013); // Sonntag
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void before01() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(17, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before02() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before03() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(16, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before04() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 4, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before05() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 5, 2011);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual01() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(17, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual02() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual03() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(16, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual04() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 4, 2012);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual05() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 5, 2011);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void after01() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(17, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.after(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void after02() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.after(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual01() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(17, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual02() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual03() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(18, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual04() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(17, 6, 2012);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual05() {
        MutualDate date1 = new MutualDate(17, 5, 2012);
        MutualDate date2 = new MutualDate(16, 5, 2013);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void between01() {
        MutualDate date  = new MutualDate(17, 5, 2012);
        MutualDate first = new MutualDate(16, 5, 2012);
        MutualDate last  = new MutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between02() {
        MutualDate date  = new MutualDate(16, 5, 2012);
        MutualDate first = new MutualDate(16, 5, 2012);
        MutualDate last  = new MutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between03() {
        MutualDate date  = new MutualDate(16, 5, 2012);
        MutualDate first = new MutualDate(16, 5, 2012);
        MutualDate last  = new MutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between04() {
        MutualDate date  = new MutualDate(18, 5, 2012);
        MutualDate first = new MutualDate(19, 5, 2012);
        MutualDate last  = new MutualDate(22, 5, 2012);
        boolean expected = false;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDayDistanceTo001() {
        MutualDate date1 = new MutualDate(6, 1, 2012); // heute
        MutualDate date2 = new MutualDate(6, 1, 2012); // heute
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 0;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo002() {
        MutualDate date1 = new MutualDate(6, 1, 2012); // heute
        MutualDate date2 = new MutualDate(7, 1, 2012); // morgen
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 1;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo003() {
        MutualDate date1 = new MutualDate(6, 1, 2012); // heute
        MutualDate date2 = new MutualDate(5, 1, 2012); // gestern
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = -1;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo004() {
        MutualDate date1 = new MutualDate(6, 1, 2012); // heute
        MutualDate date2 = new MutualDate(8, 1, 2012); // übermorgen
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 2;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo005() {
        MutualDate date1 = new MutualDate(6, 1, 2012); // heute
        MutualDate date2 = new MutualDate(4, 1, 2012); // vorgestern
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = -2;
        assertEquals(expected, distance);
    }

    @Test
    public void addDaysAndCalculateDistance() {
        /* Hier wird getestet, ob addDays und calculateDayDistanceTo zueinander
         * passen, indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    MutualDate date1 = new MutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            MutualDate date2 = date1.copy();
                            date2.addDays(abstand);
                            int abstand2 = date1.calculateDayDistanceTo(date2);
                            assertEquals(abstand, abstand2);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void addDaysAndCompareCalculatings() {
        /* Hier wird getestet, ob calculateDayDifference und
         * calculateDayDistanceTo die gleichen Ergebnisse liefern,
         * indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    MutualDate date1 = new MutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            MutualDate date2 = date1.copy();
                            date2.addDays(abstand);
                            int abstand2 = date1.calculateDayDistanceTo(date2);
                            int abstand3 = date1.calculateDayDifference(date2);
                            assertEquals(abstand2, abstand3);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void normalise1() {
        // 00.05.2003 -> 30.04.2003
        MutualDate wrongDate = new MutualDate( 0,  5, 2003);
        MutualDate rightDate = new MutualDate(30,  4, 2003);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise2() {
        // -3.05.2003 -> 27.04.2003
        MutualDate wrongDate = new MutualDate(-3,  5, 2003);
        MutualDate rightDate = new MutualDate(27,  4, 2003);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise3() {
        // 32.05.2003 -> 01.06.2003
        MutualDate wrongDate = new MutualDate(32,  5, 2003);
        MutualDate rightDate = new MutualDate( 1,  6, 2003);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise4() {
        // 12.00.2011 -> 12.12.2010
        MutualDate wrongDate = new MutualDate(12,  0, 2011);
        MutualDate rightDate = new MutualDate(12, 12, 2010);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise5() {
        // 23.-1.2012 -> 23.11.2011
        MutualDate wrongDate = new MutualDate(23, -1, 2012);
        MutualDate rightDate = new MutualDate(23, 11, 2011);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise6() {
        // 03.13.2011 -> 03.01.2012
        MutualDate wrongDate = new MutualDate( 3, 13, 2011);
        MutualDate rightDate = new MutualDate( 3,  1, 2012);
        wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void addMonths1() {
        // 31.01.2012 + 1 Monat -> 62.01.2012 -> 02.03.2012
        MutualDate startDate  = new MutualDate(31,  1, 2012);
        MutualDate targetDate = new MutualDate( 2,  3, 2012);
        startDate.addMonths(1);
        assertEquals(targetDate, startDate);
    }

    @Test
    public void addMonths2() {
        // 31.01.2013 + 1 Monat -> 62.01.2013 -> 03.03.2013
        MutualDate startDate  = new MutualDate(31,  1, 2013);
        MutualDate targetDate = new MutualDate( 3,  3, 2013);
        startDate.addMonths(1);
        assertEquals(targetDate, startDate);
    }

    @Test
    public void addMonths3() {
        // 31.03.2012 + 1 Monat -> 62.03.2012 -> 01.05.2012
        MutualDate startDate  = new MutualDate(31,  3, 2012);
        MutualDate targetDate = new MutualDate( 1,  5, 2012);
        startDate.addMonths(1);
        assertEquals(targetDate, startDate);
    }

    @Test
    public void toStringTest1() {
        MutualDate date  = new MutualDate(31, 3, 2012);
        String excpeted = "31.03.2012";
        String actual = date.toString();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStringTest2() {
        MutualDate date  = new MutualDate(31, 3, 2012);
        String excpeted = "2012/03/31";
        String actual = date.toStringInternational();
        assertEquals(excpeted, actual);
    }

    @Test
    public void startDateTest01() {
        String input = "26. 10./ 02. 11. 2000";
        MutualDate date  = new MutualDate(input);
        assertEquals("00.00.0000", date.toString());
        assertEquals(false, date.isValid());
    }

    @Test
    public void copy01() {
        MutualDate date1 = new MutualDate("11. Februar 1967");
        MutualDate date2 = date1.copy();

        int day   = date2.getDay();
        int month = date2.getMonth();
        int year  = date2.getYear();

        assertEquals(11,   day  );
        assertEquals(2,    month);
        assertEquals(1967, year );
    }

}
