package de.duehl.html.download.data;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

/**
 * Diese Klasse stellt das Ergebnis vom Herunterladen von Seiten aus dem WWW dar.
 *
 * @version 1.01     2017-06-21
 * @author Christian Dühl
 */

public class DownloadInfo {

    /** HTTP-Status als Enum aufbereitet. */
    private DownloadStatus status;

    /** Original zurückgelieferter HTTP-Status. */
    private int originalHttpStatusCode;
    /*
     * Dieser Statuswert wird zusätzlich aufbewahrt, weil lange nicht alle Statuswerte in den
     * obigen status einfließen und man so die Chance hat zu sehen, warum eine Seite nicht
     * heruntergeladen wurde.
     */

    /** Url nach der gesucht wurde. */
    private final String url;

    /** Url, die vom "Browser" zurückgegeben wurde. */
    private String returnedUrl;

    /** Inhalt der Seite in UTF-8. */
    private String content;

    /** Url der im Falle eines Redirects zu folgen ist. */
    private String redirectUrlToFollow;

    /**
     * Konstruktor.
     *
     * @param status
     *            HTTP-Status.
     * @param originalHttpStatusCode
     *            Original zurückgelieferter HTTP-Status.
     * @param url
     *            Url nach der gesucht wurde.
     * @param returnedUrl
     *            Url, die nach dem Befolgen von Redirects letztlich zurückgegeben wurde.
     * @param contents
     *            Inhalt der Seite in UTF-8.
     */
    public DownloadInfo(String url) {
        this.url = url;

        status = DownloadStatus.NOT_LOADED;
        originalHttpStatusCode = -1;
        returnedUrl = "";
        redirectUrlToFollow = "";
        content = "";
    }

    /**
     * Wird aufgerufen, wenn die Abfrage Erfolg hatte.
     *
     * @param status
     *            HTTP-Status.
     * @param originalHttpStatusCode
     *            Original zurückgelieferter HTTP-Status.
     * @param returnedUrl
     *            Url, die vom "Browser" zurückgegeben wurde.
     * @param content
     *            Inhalt der Seite in UTF-8.
     * @param redirectUrlToFollow
     *            Url der im Falle eines Redirects zu folgen ist.
     */
    public void storeResult(DownloadStatus status, int originalHttpStatusCode,
            String returnedUrl, String content, String redirectUrlToFollow) {
        this.status = status;
        this.originalHttpStatusCode = originalHttpStatusCode;
        this.returnedUrl = returnedUrl;
        this.content = content;
        this.redirectUrlToFollow = redirectUrlToFollow;
    }

    /** Wird aufgerufen, wenn es bei dem Download zu einer Ausnahme kam. */
    public void failed() {
        status = DownloadStatus.FAILURE;
    }

    /** Wird aufgerufen, wenn die Seite nicht gefunden wurde. */
    public void pageNotFound() {
        status = DownloadStatus.PAGE_NOT_FOUND;
    }

    /** Wird aufgerufen, wenn der Download zu lange dauerte und deshalb beendet wurde. */
    public void downloadCanceled() {
        status = DownloadStatus.DOWNLOAD_CANCELED;
    }

    /** Getter für den HTTP-Status. */
    public DownloadStatus getStatus() {
        return status;
    }

    /** Original zurückgelieferter HTTP-Status. */
    public int getOriginalHttpStatusCode() {
        return originalHttpStatusCode;
    }

    /** Getter für die Url nach der gesucht wurde. */
    public String getUrl() {
        return url;
    }

    /** Getter für die Url, die vom "Browser" zurückgegeben wurde. */
    public String getReturnedUrl() {
        return returnedUrl;
    }

    /** Getter für den Inhalt der Seite in UTF-8. */
    public String getContent() {
        return content;
    }

    /** Gibt an, ob der Download erfolgreich war. */
    public boolean isDownloadSuccessfull() {
        return status == DownloadStatus.OK;
    }

    /** Gibt an, ob der Download wegen Zeitüberschreitung abgebrochen wurde. */
    public boolean isDownloadCanceled() {
        return status == DownloadStatus.DOWNLOAD_CANCELED;
    }

    /** Gibt an, ob es sich um einen Redirect handelt. */
    public boolean isRedirect() {
        return originalHttpStatusCode == 301
                || originalHttpStatusCode == 302
                || originalHttpStatusCode == 303
                || originalHttpStatusCode == 307
                || originalHttpStatusCode == 200 && !redirectUrlToFollow.isEmpty(); // meta header
    }

    /** Getter für die Url, der im Falle eines Redirects zu folgen ist. */
    public String getRedirectUrlToFollow() {
        return redirectUrlToFollow;
    }

    @Override
    public String toString() {
        return "DownloadInfo [status=" + status + ", originalHttpStatusCode="
                + originalHttpStatusCode + ", url=" + url + ", returnedUrl=" + returnedUrl
                + ", content=" + content + ", redirectUrlToFollow=" + redirectUrlToFollow + "]";
    }

    public String toStringWithoutContent() {
        return "DownloadInfo [status=" + status + ", originalHttpStatusCode="
                + originalHttpStatusCode + ", url=" + url + ", returnedUrl=" + returnedUrl
                + ", redirectUrlToFollow=" + redirectUrlToFollow + "]";
    }

}
