package de.duehl.swing.debug.dialog.table;

import java.util.List;

import javax.swing.table.AbstractTableModel;

import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.datetime.time.ImmutualTime;
import de.duehl.swing.debug.data.DebugMessage;

/**
 * Diese Klasse ist ein eigenes Tabellenmodell für die Ansicht der Debugmeldungen.
 *
 * Es hält die Daten der Tabelle in Form einer Liste von Debugmeldungen vor und weiß diese
 * anzuzeigen.
 *
 * @version 1.01     2025-07-29
 * @author Christian Dühl
 */

public class DebugMessagesTableModel extends AbstractTableModel {

    private static final int NUMBER_OF_COLUMNS_WITHOUT_CATEGORY = 6;

    private static final long serialVersionUID = 1L;


    /** Die eingegangenen Meldungen der hier dargestellten Kategorie. */
    private final List<DebugMessage> messages;

    /** Gibt an, ob in der Tabelle auch die Kategorie angezeigt werden soll. */
    private boolean showCategory;

    /**
     * Konstruktor.
     *
     * @param messages
     *            Die eingegangenen Meldungen der hier dargestellten Kategorie.
     */
    public DebugMessagesTableModel(List<DebugMessage> messages) {
        super();

        this.messages = messages;

        showCategory = false;
    }

    /** Legt fest, dass in der Tabelle auch die Kategorie angezeigt werden soll. */
    public void showCategory() {
        showCategory = true;

    }

    /** Ermittelt die Zeilenzahl. */
    @Override
    public int getRowCount() {
        return messages.size();
    }

    /** Ermittelt die Spaltenzahl. */
    @Override
    public int getColumnCount() {
        if (showCategory) {
            return NUMBER_OF_COLUMNS_WITHOUT_CATEGORY + 1;
        }
        else {
            return NUMBER_OF_COLUMNS_WITHOUT_CATEGORY;
        }
    }

    /**
     * Ermittelt den Tabelleninhalt der angegebenen Zelle.
     *
     * Die Tabelle enthält die folgenden Spalten:
     *     - Datum
     *     - Uhrzeit
     *     - Kategorie (sofern sie angezeigt werden soll)
     *     - Klassenname
     *     - Methodenname
     *     - Zeilennummer
     *     - Meldung
     *
     * @param rowIndex
     *            Zeilenindex (die erste hat die 0)
     * @param columnIndex
     *            Spaltenindex (die erste hat die 0)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        if (showCategory) {
            return getValueAtWithCategory(rowIndex, columnIndex);
        }
        else {
            return getValueAtWithoutCategory(rowIndex, columnIndex);
        }
    }

    private Object getValueAtWithCategory(int rowIndex, int columnIndex) {
        DebugMessage message = messages.get(rowIndex);

        switch (columnIndex) {
            case 0:
                return message.getDate();
            case 1:
                return message.getTime();
            case 2:
                return message.getCategory();
            case 3:
                return message.getClassName();
            case 4:
                return message.getMethodName();
            case 5:
                return message.getLineNumber();
            case 6:
                return message.getMessage();
            default:
                throw new RuntimeException("Unzuläsiger Spaltenindex '" + columnIndex + "'.");
        }
    }

    private Object getValueAtWithoutCategory(int rowIndex, int columnIndex) {
        DebugMessage message = messages.get(rowIndex);

        switch (columnIndex) {
            case 0:
                return message.getDate();
            case 1:
                return message.getTime();
            case 2:
                return message.getClassName();
            case 3:
                return message.getMethodName();
            case 4:
                return message.getLineNumber();
            case 5:
                return message.getMessage();
            default:
                throw new RuntimeException("Unzuläsiger Spaltenindex '" + columnIndex + "'.");
        }
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        switch (columnIndex) {
            case 0:
                return ImmutualDate.class;
            case 1:
                return ImmutualTime.class;
            default:
                return String.class;
        }
    }

}
