package de.duehl.swing.ui.buttons.painted;

import java.awt.Color;

import de.duehl.swing.ui.geometry.PixelPoint;

/**
 * Diese Klasse stellt einen kleinen Schalter zum Pausieren dar.
 *
 * @version 1.01     2025-07-23
 * @author Christian Dühl
 */

final public class PauseButton extends PaintedButton {

    private static final long serialVersionUID = 1L;

    /** Konstruktor mit schwarzer Farbe. */
    public PauseButton() {
        this(Color.BLACK);
    }

    /**
     * Konstruktor.
     *
     * @param symbolColor
     *            Die Farbe des Symbols.
     */
    public PauseButton(Color symbolColor) {
        super(symbolColor);
    }

    /** Zeichnet das Symbol in horizontaler und rechteckiger Anordnung. */
    @Override
    protected void paintHorizontalSymbolRectangle() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in vertikaler und rechteckiger Anordnung. */
    @Override
    protected void paintVerticalSymbolRectangle() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in horizontaler und quadratischer Anordnung. */
    @Override
    protected void paintHorizontalSymbolSquare() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in vertikaler und quadratischer Anordnung. */
    @Override
    protected void paintVerticalSymbolSquare() {
        paintSymbol();
    }

    /**
     * Zeichnet das farbige Pause-Symbol. Da es dabei egal ist, ob horizontal oder vertikal, wird
     * es in beiden Fällen verwendet.
     *
     * Hier wird davon ausgegangen, dass die Breite eines Striche im Verhältnis zum Platz
     * dazwischen 2:1 beträgt. Allerdings ist der Platz dann nicht mehr zu sehen, vermutlich
     * durch irgendwelche Strichbreiten oder dergleichen. Daher teile ich nicht durch 5, sondern
     * durch eine größere Zahl.
     *
     *     1------2   5------6
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     |      |   |      |
     *     4------3   8------7
     *
     */
    private void paintSymbol() {
        int width = getWidth();
        int height = getHeight();

        if (isForceSquare()) {
            width = Math.min(width, height);
            height = width;
        }

        int delta = 6;

        int firstLeftPixel = (getWidth() - width)/2;
        int left = firstLeftPixel + delta;

        int lastRightPixel = firstLeftPixel + width - 1;
        int right = lastRightPixel - delta;

        int firstUpperPixel = (getHeight() - height)/2;
        int upper = firstUpperPixel + delta;

        int lastLowerPixel = firstUpperPixel + height - 1;
        int lower = lastLowerPixel - delta;

        int spaceWidth = (int) (width / 7.5);
        int rectanglesWidth = spaceWidth * 2;


        paintPolygon(
                new PixelPoint(left,                   upper), //  1
                new PixelPoint(left + rectanglesWidth, upper), //  2
                new PixelPoint(left + rectanglesWidth, lower), //  3
                new PixelPoint(left,                   lower)  //  4
                );

        paintPolygon(
                new PixelPoint(right - rectanglesWidth, upper), //  5
                new PixelPoint(right,                   upper), //  6
                new PixelPoint(right,                   lower), //  7
                new PixelPoint(right - rectanglesWidth, lower)  //  8
                );
    }

}
