package de.duehl.swing.ui.dialogs.values;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zum Eingeben oder auswählen eines Wertes mit Hilfe einer
 * JComboBox (DropDwon) aus einer Liste möglicher Werte.
 *
 * @version 1.02     2020-10-30
 * @author Christian Dühl
 */

public class ComboBoxDialog extends ModalDialogBase {

    /** Feld in das der Benutzer seine Daten eingibt. */
    private final JComboBox<String> comboBox;

    /** Text der Frage über dem Eingabefeld. */
    private final String question;

    /** Vorgabe die im Eingabefeld angezeigt werden soll. */
    private final String presetting;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für den Dialog.
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param question
     *            Text der Frage über dem Eingabefeld.
     * @param presetting
     *            Vorgabe die im Eingabefeld angezeigt werden soll.
     * @param values
     *            Werte in der DropDown-Box
     */
    public ComboBoxDialog(Point parentLocation, Image programImage, String dialogTitle,
            String question, String presetting, List<String> values) {
        super(parentLocation, programImage, dialogTitle);
        this.question = question;
        this.presetting = presetting;

        String[] valuesArray = CollectionsHelper.stringListToArray(values);
        comboBox = new JComboBox<>(valuesArray);

        fillDialog();

        addClosingWindowListener(() -> escape());
    }

    public void allowEnteringOwnValues() {
        comboBox.setEditable(true);
    }

    /** Erstellt die Dialog-Gui. */
    @Override
    protected void populateDialog() {
        add(createQuestionPart(), BorderLayout.NORTH);
        add(createEnterValuePart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);
        setKeyBindings();
    }

    /** Erzeugt den Bereich mit der Frage. */
    private Component createQuestionPart() {
        JLabel label = new JLabel(question);

        return label;
    }

    /** Erzeugt den Bereich mit dem Textfeld. */
    private Component createEnterValuePart() {
        comboBox.addKeyListener(new KeyListener() {
            @Override
            public void keyPressed(KeyEvent arg0) {
            }
            @Override
            public void keyReleased(KeyEvent event) {
                if (KeyEvent.VK_ENTER == event.getKeyCode()) {
                    closeDialog();
                }
            }
            @Override
            public void keyTyped(KeyEvent arg0) {
            }
        });
        comboBox.setSelectedItem(presetting);
        return comboBox;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private JButton createQuitButton() {
        JButton button = new JButton("Abbrechen");
        button.addActionListener(e -> escape());
        return button;
    }

    private JButton createOkButton() {
        JButton button = new JButton("OK");
        button.addActionListener(e -> closeDialog());
        return button;
    }

    /** Erstellt Tastenkombinationen. */
    private void setKeyBindings() {
        setKeyBindingEscape(() -> escape());
    }

    private void escape() {
        comboBox.setSelectedItem(""); // Leer bei Abbruch
        closeDialog();
    }

    /** Getter für den Vom Benutzer eingegebener Text. */
    public String getEnteredValue() {
        String item;
        if (comboBox.isEditable()) {
            item = comboBox.getEditor().getItem().toString();
        }
        else {
            item = comboBox.getSelectedItem().toString();
        }
        return item.trim();
    }

}
