package de.duehl.swing.ui.dialogs.values;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zum Eingeben eines Wertes.
 *
 * @version 1.02     2022-01-19
 * @author Christian Dühl
 */

public class EnterLongTextDialog extends ModalDialogBase {

    private static final Dimension DIALOG_DIMENSION = new Dimension(800, 300);

    /** Feld in das der Benutzer seine Daten eingibt. */
    private final JTextArea valueArea;

    /** Text der Frage über dem Eingabefeld. */
    private final String question;

    /** Vorgabe die im Eingabefeld angezeigt werden soll. */
    private final String presetting;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für den Dialog.
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param question
     *            Text der Frage über dem Eingabefeld.
     * @param presetting
     *            Vorgabe die im Eingabefeld angezeigt werden soll.
     */
    public EnterLongTextDialog(Point parentLocation, Image programImage, String dialogTitle,
            String question, String presetting) {
        this(null, parentLocation, programImage, dialogTitle, question, presetting);
    }

    /**
     * Konstruktor.
     *
     * @param colorizer
     *            Farbverwaltung für die Gui.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für den Dialog.
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param question
     *            Text der Frage über dem Eingabefeld.
     * @param presetting
     *            Vorgabe die im Eingabefeld angezeigt werden soll.
     */
    public EnterLongTextDialog(Colorizer colorizer, Point parentLocation, Image programImage,
            String dialogTitle, String question, String presetting) {
        super(parentLocation, programImage, dialogTitle, DIALOG_DIMENSION, colorizer);
        this.question = question;
        this.presetting = presetting;

        valueArea = new JTextArea();

        fillDialog();

        addClosingWindowListener(() -> escape());
    }

    /** Erstellt die Dialog-Gui. */
    @Override
    protected void populateDialog() {
        initElements();

        add(createQuestionPart(), BorderLayout.NORTH);
        add(createEnterValuePart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);

        setKeyBindings();
    }

    private void initElements() {
        setColors(valueArea);
        valueArea.setCaretColor(Color.RED);
        valueArea.setLineWrap(true);
        valueArea.setWrapStyleWord(true);
        GuiTools.biggerFont(valueArea, 5);

        /* Das machen wir nicht, denn Enter soll einen Umbruch im text einfügen.
        valueArea.addKeyListener(new KeyListener() {
            @Override
            public void keyPressed(KeyEvent arg0) {
            }
            @Override
            public void keyReleased(KeyEvent event) {
                if (KeyEvent.VK_ENTER == event.getKeyCode()) {
                    closeDialog();
                }
            }
            @Override
            public void keyTyped(KeyEvent arg0) {
            }
        });
        */

        valueArea.setText(presetting);
        valueArea.selectAll();
    }

    /** Erzeugt den Bereich mit der Frage. */
    private Component createQuestionPart() {
        JLabel label = new JLabel(question);
        setColors(label);

        return label;
    }

    /** Erzeugt den Bereich mit dem Textfeld. */
    private Component createEnterValuePart() {
        return new JScrollPane(valueArea,
                JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
    }

    /** Erstellt Tastenkombinationen. */
    private void setKeyBindings() {
        setKeyBindingEscape(() -> escape());
    }

    private void escape() {
        valueArea.setText(""); // Leer bei Abbruch
        closeDialog();
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private JButton createQuitButton() {
        JButton button = new JButton("Abbrechen");
        button.addActionListener(e -> escape());
        return button;
    }

    private JButton createOkButton() {
        JButton button = new JButton("OK");
        button.addActionListener(e -> closeDialog());
        return button;
    }

    /** Getter für den Vom Benutzer eingegebener Text. */
    public String getEnteredValue() {
        return valueArea.getText().trim();
    }

}
