package de.duehl.swing.ui.error;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt einen UncaughtExceptionHandler dar, welcher Fehler abfängt und mittels
 * eines <tt>ErrorDialog</tt>s anzeigt.
 *
 * @version 1.04     2022-01-27
 * @author Christian Dühl
 */

public class GuiExceptionHandler implements UncaughtExceptionHandler {

    private static final boolean REMOVE_NESTED_EXCEPTION_CLASS = false;

    private static final AfterErrorHandlingCallable NO_CALLABLE = new AfterErrorHandlingCallable() {
        @Override
        public void callMeAfterHandledUiError() {
            // nichts ...
        }
    };

    /**
     * Liste mit Fehlermeldungen, zu der kein Fehlerdialog angezeigt wird, sondern der Benutzer nur
     * informiert wird.
     */
    private final List<String> noDialogErrorMessages;

    /** Aktion nach Anzeige des Fehlers. */
    private AfterErrorHandlingCallable callable;

    /** Konstruktor (zunächst ohne eine Aktion nach Anzeige des Fehlers). */
    public GuiExceptionHandler() {
        this.callable = NO_CALLABLE;
        noDialogErrorMessages = new ArrayList<>();
    }

    /** Setter für die Aktion nach Anzeige des Fehlers. */
    public void setAfterErrorHandlingCallable(AfterErrorHandlingCallable callable) {
        this.callable = callable;
    }

    /**
     * Fügt eine Fehlermeldung hinzu, zu der kein Fehlerdialog angezeigt wird, sondern der Benutzer
     * nur informiert wird.
     */
    public void addNoDialogErrorMessage(String noDialogErrorMessage) {
        noDialogErrorMessages.add(noDialogErrorMessage);
    }

    /** Reagiert auf eine unbehandelte Ausnahme. */
    @Override
    public void uncaughtException(Thread thread, Throwable throwable) {
        String message = throwable.getMessage();
        String pureMessage = determinePureMessage(message);

        if (noDialogErrorMessages.contains(pureMessage)) {
            System.out.println(pureMessage);
        }
        else {
            ErrorDialog dialog = new ErrorDialog(
                    "Nicht abgefangenes Problem im Thread " + thread.getName() + "!",
                    throwable);
            dialog.setVisible(true);
        }

        if (callable != NO_CALLABLE) {
            callable.callMeAfterHandledUiError();
        }
    }

    private String determinePureMessage(String message) {
        if (REMOVE_NESTED_EXCEPTION_CLASS) {
            return Text.removePartBeforeSearchTextIncludingSearchText(message, ":").trim();
        }
        else {
            return message;
        }
    }

}
