package de.duehl.swing.ui.menu.collection;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.swing.ui.menu.collection.tools.MyMenuItemCollectionHelper;

/**
 * Diese Klasse sammelt die gemerkten Eigenschaften von der Menüeinträgen.
 *
 * @version 1.01     2020-07-08
 * @author Christian Dühl
 */

class MyMenuItemCollection {

    /** Statische Liste aller im Projekt hinterlegten Menüeinträge. */
    private final List<StoredMyMenuItem> collectedMyMenuItems = new ArrayList<>();

    /** Gibt an, ob die Menüeinträge gesammelt werden sollen. */
    private boolean collectMyMenuItems = true;

    /** Sammelt einen Menüeintrag. */
    public void collect(StoredMyMenuItem storedMyMenuItem) {
        if (collectMyMenuItems) {
            /*
             * An dieser Stelle kann man leider nicht einfach automatisch prüfen, ob
             * die Tastenkombination schon vergeben wurde, da diese Information im neuen
             * Menüeintrag noch nicht gesetzt wurde.
             *
             * Dafür gibt es die Methode checkAccelerator().
             */
            collectedMyMenuItems.add(storedMyMenuItem);
        }
    }

    /**
     * Überprüft die Tastenbelegung dieses Menüeintrags auf das bereits Vorhandensein bei anderen
     * Menüeinträgen.
     *
     * Hier befindet sich das übergebene Objekt bereits in der Liste, daher wird es natürlich nicht
     * mit sich selbst verglichen!
     */
    public void checkAccelerator(StoredMyMenuItem storedMyMenuItem) {
        checkNewMyMenuItemForKeyboardDuplicates(storedMyMenuItem);
    }

    private void checkNewMyMenuItemForKeyboardDuplicates(StoredMyMenuItem testingStoredMyMenuItem) {
        if (testingStoredMyMenuItem.isKeyCodeAndModifiersStored()) {
            for (StoredMyMenuItem storedMyMenuItem : collectedMyMenuItems) {
                if (storedMyMenuItem != testingStoredMyMenuItem
                        && storedMyMenuItem.isKeyCodeAndModifiersStored()) {
                    checkNewMyMenuItemForKeyboardDuplicates(testingStoredMyMenuItem,
                            storedMyMenuItem);
                }
            }
        }
    }

    private void checkNewMyMenuItemForKeyboardDuplicates(StoredMyMenuItem testingStoredMyMenuItem,
            StoredMyMenuItem storedMyMenuItem) {
        if (testingStoredMyMenuItem.getKeyCode() == storedMyMenuItem.getKeyCode()
                && testingStoredMyMenuItem.getModifiers() == storedMyMenuItem.getModifiers()) {
            throw new IllegalArgumentException("Es gibt zwei Menüeinträge mit dem gleichen "
                    + "Tastatur-Kommando:\n"
                    + "\t" + "Tastatur-Kommando    : '"
                            + MyMenuItemCollectionHelper.generateKeyboardCommandDescription(
                                    storedMyMenuItem)
                            + "'\n"
                    + "\t" + "Vorhandener Menütext : '" + storedMyMenuItem.getText() + "'\n"
                    + "\t" + "Neuer Menütext       : '" + testingStoredMyMenuItem.getText() + "'\n");
        }
    }

    /** Schaltet die Sammlung von Menüeinträgen ab. */
    public void switchOffMyMenuItemCollection() {
        collectMyMenuItems = false;
    }

    /** Schaltet die Sammlung von Menüeinträgen an (nur sinnvoll, falls vorher abgeschaltet). */
    public void switchOnMyMenuItemCollection() {
        collectMyMenuItems = true;
    }

    /** Entfernt alle bisher gemerkten Menüeinträge. */
    public void clearCollection() {
        collectedMyMenuItems.clear();
    }

    /** Getter für die Liste aller im Projekt hinterlegten Menüeinträge. */
    public List<StoredMyMenuItem> getListOfCollectedMyMenuItems() {
        return CollectionsHelper.copyList(collectedMyMenuItems);
    }

}
