package de.duehl.swing.ui.timer;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.Timer;
import java.util.TimerTask;

/**
 * Diese Klasse stellt einen Timer für grafische Oberflächen dar, der abläuft, bevor auf
 * Usereingaben reagiert wird, damit der Benutzer in Ruhe tippen kann, ohne gestört zu werden.
 *
 * @version 1.01     2017-11-15
 * @author Christian Dühl
 */

public class WaitingUserInputTimer {

    /** Die zu erledigende Aufgabe, nachdem der Benutzer aufgehört hat zu tippen. */
    private final Runnable runnable;

    /** Millisekunden, die gewartet werden sollen, nachdem die letzte Benutzereingabe erfolgte. */
    private final long milliseconds;

    /** Timer zur ... */
    private Timer timer;

    /** Gibt an, ob die Wartezeit zwischen Benutzereingabe und Ausführung der Aufgabe noch läuft. */
    private boolean running;

    /**
     * Konstruktor.
     *
     * @param runnable
     *            Die zu erledigende Aufgabe, nachdem der Benutzer aufgehört hat zu tippen.
     * @param milliseconds
     *            Millisekunden, die gewartet werden sollen, nachdem die letzte Benutzereingabe
     *            erfolgte.
     */
    public WaitingUserInputTimer(Runnable runnable, long milliseconds) {
        this.milliseconds = milliseconds;
        this.runnable = runnable;
        running = false;

        startTimer();
    }

    private synchronized void startTimer() {
        timer = new Timer();
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                running = false;
                runnable.run();
            }
        }, milliseconds);
        running = true;
    }

    /**
     * Gibt an, ob die Wartezeit zwischen Benutzereingabe und Ausführung der Aufgabe bereits
     * abgelaufen ist.
     */
    public synchronized boolean hasStopped() {
        return !running;
    }

    /** Hält den Timer an und startet ihn erneut. */
    public synchronized void restart() {
        timer.cancel();
        startTimer();
    }

    public void cancel() {
        timer.cancel();
        running = false;
    }

}
