package de.duehl.swing.text.html;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class KnownValueHandlingTest {

    @Test
    public void firstAndContained() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "foo";
        List<String> knownValues = CollectionsHelper.buildListFrom();
        assertTrue(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void firstAndNotContained() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "Apfel";
        List<String> knownValues = CollectionsHelper.buildListFrom();
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void secondAndContained() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "foo";
        List<String> knownValues = CollectionsHelper.buildListFrom("foo");
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void secondContainedAndPart() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "fo";
        List<String> knownValues = CollectionsHelper.buildListFrom("foo");
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void secondContainedAndPartOtherWay() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "foo";
        List<String> knownValues = CollectionsHelper.buildListFrom("fo");
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void empty() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "";
        List<String> knownValues = CollectionsHelper.buildListFrom();
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void blank() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "    ";
        List<String> knownValues = CollectionsHelper.buildListFrom();
        assertFalse(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void blankInList() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String perhapsNewValue = "foo";
        List<String> knownValues = CollectionsHelper.buildListFrom("   ");
        assertTrue(handling.isValueNewAndGood(perhapsNewValue, knownValues));
    }

    @Test
    public void perhapsAdd() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String value = "foo";
        List<String> values = CollectionsHelper.buildListFrom();
        handling.perhapsAdd(value, values);
        assertEquals(1, values.size());
        assertEquals("foo", values.get(0));
    }

    @Test
    public void perhapsAddWithOtherDiffereentKnownValue() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String value = "foo";
        List<String> values = CollectionsHelper.buildListFrom();
        String knownValue = "bar";
        handling.perhapsAdd(value, values, knownValue);
        assertEquals(1, values.size());
        assertEquals("foo", values.get(0));
    }

    @Test
    public void perhapsAddWithOtherSameKnownValue() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String value = "foo";
        List<String> values = CollectionsHelper.buildListFrom();
        String knownValue = "foo";
        handling.perhapsAdd(value, values, knownValue);
        assertEquals(0, values.size());
    }

    @Test
    public void perhapsAddWithOtherKnownValuesContainingTheValue() {
        String text = "foo bar baz";
        KnownValueHandling handling = new KnownValueHandling(text);
        String value = "foo";
        List<String> values = CollectionsHelper.buildListFrom();
        List<String> knownValues = CollectionsHelper.buildListFrom("foo", "bar");
        handling.perhapsAdd(value, values, knownValues);
        assertEquals(0, values.size());
    }

    @Test
    public void torstenLiebe() {
        String text = "Öffentliche Bekanntmachung Amtsgericht Leipzig, Aktenzeichen: 405 IN 396/21 "
                + "In dem Insolvenzverfahren über das Vermögen des Torsten Liebe, geb. 04.01.1977, "
                + "Leibnitzstraße 5, 04288 Leipzig Inhaber der Firma Torsten Liebe Rohbaumeister, "
                + "- hat der Insolvenzverwalter am 29.04.2021 gemäß § 35 Abs. 2 InsO erklärt, dass "
                + "Vermögen aus der selbständigen Tätigkeit des Schuldners die "
                + "weiterwirtschaftende Tätigkeit des Torsten Liebe, handelnd als Bauunternehmer, "
                + "sowie alle damit in Zusammenhang stehenden Verträge (§ 35 Abs. 2 Satz 1 InsO) "
                + "nicht zur Insolvenzmasse gehört und Ansprüche aus dieser Tätigkeit im "
                + "Insolvenzverfahren nicht geltend gemacht werden können. 405 IN 396/21 "
                + "Amtsgericht Leipzig, Insolvenzgericht, 03.05.2021";
        KnownValueHandling handling = new KnownValueHandling(text);
        List<String> values = CollectionsHelper.buildListFrom();
        String perhapsNewValue = "Torsten Liebe";
        String knownValue = "Torsten Liebe Rohbaumeister";
        handling.perhapsAdd(perhapsNewValue, values, knownValue);
        assertEquals(0, values.size());
        /*
         * Anmerkung:
         *
         * Der Name "Torsten Liebe" (rechts) wird zum Namen "Torsten Liebe Rohbaumeister" (links)
         * nicht hervorghehoben!
         *
         * Das funktioniert alledings wie geplant, der Begriff "Torsten Liebe" wird nicht
         * aufgenommen, da er in "Torsten Liebe Rohbaumeister" enthalten ist, damit nicht
         * längere Auszeichnungen durch kürzere kaputt gemacht werden.
         */
    }

    @Test
    public void torstenLiebeRohbaumeister() {
        String text = "Öffentliche Bekanntmachung Amtsgericht Leipzig, Aktenzeichen: 405 IN 396/21 "
                + "In dem Insolvenzverfahren über das Vermögen des Torsten Liebe, geb. 04.01.1977, "
                + "Leibnitzstraße 5, 04288 Leipzig Inhaber der Firma Torsten Liebe Rohbaumeister, "
                + "- hat der Insolvenzverwalter am 29.04.2021 gemäß § 35 Abs. 2 InsO erklärt, dass "
                + "Vermögen aus der selbständigen Tätigkeit des Schuldners die "
                + "weiterwirtschaftende Tätigkeit des Torsten Liebe, handelnd als Bauunternehmer, "
                + "sowie alle damit in Zusammenhang stehenden Verträge (§ 35 Abs. 2 Satz 1 InsO) "
                + "nicht zur Insolvenzmasse gehört und Ansprüche aus dieser Tätigkeit im "
                + "Insolvenzverfahren nicht geltend gemacht werden können. 405 IN 396/21 "
                + "Amtsgericht Leipzig, Insolvenzgericht, 03.05.2021";
        KnownValueHandling handling = new KnownValueHandling(text);
        List<String> values = CollectionsHelper.buildListFrom();
        String perhapsNewValue = "Torsten Liebe Rohbaumeister";
        String knownValue = "Torsten Liebe";
        handling.perhapsAdd(perhapsNewValue, values, knownValue);
        assertEquals(0, values.size());
        /*
         * Anmerkung:
         *
         * Der Name "Torsten Liebe" (links) wird zum Namen "Torsten Liebe Rohbaumeister" (rechts)
         * nicht hervorghehoben!
         *
         * Das funktioniert alledings wie geplant, der Begriff "Torsten Liebe" wird nicht
         * aufgenommen, da er in "Torsten Liebe Rohbaumeister" enthalten ist, damit nicht
         * längere Auszeichnungen durch kürzere kaputt gemacht werden.
         */
    }

    @Test
    public void streetAndStr() {
        String text = "Öffentliche Bekanntmachung61 IN 204/14: In dem Restschuldbefreiungsverfahren "
                + "des Eckhardt Kurfiss, geb. am 18.07.1957, Kirchgönser Straße 8, 35510 Butzbach, "
                + "Inhaber der Firma \"\"Carafun\"\", Europastraße 2, 61200 Wölfersheim, ist dem "
                + "Schuldner am 28.01.2020 gem. § 300 Abs. 1 Satz 2 InsO vorzeitig "
                + "Restschuldbefreiung erteilt worden. Von der Restschuldbefreiung sind diejenigen "
                + "Gläubiger betroffen, die im Zeitpunkt der Eröffnung des Insolvenzverfahrens über "
                + "das Vermögen des Schuldners am 08.12.2014 Insolvenzgläubiger im Sinne des § 38 "
                + "InsO waren, auch wenn sie nicht an dem Insolvenzverfahren teilgenommen haben"
                + ". Von der Restschuldbefreiung ausgenommen sind die in § 302 InsO aufgeführten "
                + "Forderungen. Der vollständige Beschluss kann von den Beteiligten in der "
                + "Geschäftsstelle des Insolvenzgerichts eingesehen werden. Amtsgericht Friedberg "
                + "(Hessen), 28.01.2020";
        String knownValue = "Europastraße 2";
        String  perhapsNewValue = "Kirchgönser Str. 8";
        List<String> values = CollectionsHelper.buildListFrom();
        KnownValueHandling handling = new KnownValueHandling(text);
        handling.perhapsAdd(perhapsNewValue, values, knownValue);
        assertEquals(0, values.size());
        /*
         * Anmerkung: Kirchgönser Str. 8 kommt nicht im Text vor!
         */
    }

    @Test
    public void streetAndStrSkipCheckNewValueIsContainedInText() {
        String text = "Öffentliche Bekanntmachung61 IN 204/14: In dem Restschuldbefreiungsverfahren "
                + "des Eckhardt Kurfiss, geb. am 18.07.1957, Kirchgönser Straße 8, 35510 Butzbach, "
                + "Inhaber der Firma \"\"Carafun\"\", Europastraße 2, 61200 Wölfersheim, ist dem "
                + "Schuldner am 28.01.2020 gem. § 300 Abs. 1 Satz 2 InsO vorzeitig "
                + "Restschuldbefreiung erteilt worden. Von der Restschuldbefreiung sind diejenigen "
                + "Gläubiger betroffen, die im Zeitpunkt der Eröffnung des Insolvenzverfahrens über "
                + "das Vermögen des Schuldners am 08.12.2014 Insolvenzgläubiger im Sinne des § 38 "
                + "InsO waren, auch wenn sie nicht an dem Insolvenzverfahren teilgenommen haben"
                + ". Von der Restschuldbefreiung ausgenommen sind die in § 302 InsO aufgeführten "
                + "Forderungen. Der vollständige Beschluss kann von den Beteiligten in der "
                + "Geschäftsstelle des Insolvenzgerichts eingesehen werden. Amtsgericht Friedberg "
                + "(Hessen), 28.01.2020";
        String knownValue = "Europastraße 2";
        String  perhapsNewValue = "Kirchgönser Str. 8";
        List<String> values = CollectionsHelper.buildListFrom();
        KnownValueHandling handling = new KnownValueHandling(text);
        handling.skipCheckNewValueIsContainedInText();
        handling.perhapsAdd(perhapsNewValue, values, knownValue);
        assertEquals(1, values.size());
        assertEquals(perhapsNewValue, values.get(0));
        /*
         * Anmerkung: Kirchgönser Str. 8 kommt nicht im Text vor!
         */
    }

}
