package de.duehl.vocabulary.japanese.grammar;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse hält eine sortierte Liste der Verbformen vor und kann eine Liste von Vokabeln nach
 * diesen sortieren.
 *
 * @version 1.01     2025-11-26
 * @author Christian Dühl
 */

public class VerbSearchWords {

    public static final String WOERERBUCHFORM = "Wörterbuchform";

    /** Die sortierte Liste der Verbformen. */
    public static final List<String> SORTED_VERB_SEARCH_WORDS = CollectionsHelper.buildListFrom(
            WOERERBUCHFORM,
            "höflich Gegenwart/Zukunft",
            "höflich Gegenwart/Zukunft verneint",
            "höflich Vergangenheit",
            "höflich Vergangenheit verneint",
            "normal Gegenwart/Zukunft",
            "normal Gegenwart/Zukunft verneint", // Nai-Form
            "normal Vergangenheit", // Ta-Form
            "normal Vergangenheit verneint",
            "Willensform", // shou
            "normale Willensform", // yo
            "Aufforderungsform", // Te-Form für Bitten und Aufforderungen
            "Aufforderungsform verneint", // naide
            "höfliche Wunschform Gegenwart/Zukunft", // tai-Form mit desu
            "höfliche Wunschform Gegenwart/Zukunft verneint",
            "höfliche Wunschform Vergangenheit",
            "höfliche Wunschform Vergangenheit verneint",
            "normale Wunschform Gegenwart/Zukunft", // tai-Form
            "normale Wunschform Gegenwart/Zukunft verneint",
            "normale Wunschform Vergangenheit",
            "normale Wunschform Vergangenheit verneint",
            "höfliche Verlaufsform Gegenwart", // Te-imas-Form
            "höfliche Verlaufsform Gegenwart verneint",
            "höfliche Verlaufsform Vergangenheit",
            "höfliche Verlaufsform Vergangenheit verneint",
            "normale Verlaufsform Gegenwart", // Te-iru-Form
            "normale Verlaufsform Gegenwart verneint",
            "normale Verlaufsform Vergangenheit",
            "normale Verlaufsform Vergangenheit verneint",
            "höfliche Potentialform Gegenwart/Zukunft",
            "höfliche Potentialform Gegenwart/Zukunft verneint",
            "höfliche Potentialform Vergangenheit",
            "normale Potentialform Gegenwart/Zukunft",
            "normale Potentialform Gegenwart/Zukunft verneint",
            "normale Potentialform Vergangenheit",
            "Suru-Verb"
            );

    /** Sortiert die übergebene Liste von Vokabeln nach den Suchbegriffen für Verbformen. */
    public static void sortVocablesByVerbForms(List<Vocable> vocables) {
        Collections.sort(vocables, new Comparator<Vocable>() {
            @Override
            public int compare(Vocable v1, Vocable v2) {
                int index1 = determineLowestExistingVerbFormIndex(v1.getSearchWords());
                int index2 = determineLowestExistingVerbFormIndex(v2.getSearchWords());
                if (index1 == -1) {
                    index1 = Integer.MAX_VALUE;
                }
                if (index2 == -1) {
                    index2 = Integer.MAX_VALUE;
                }
                return index1 - index2;
            }});
    }

    /**
     * Bestimmt den kleinsten Index der vorkommenden Verbformen unter den Suchbegriffen. Wird keine
     * Verbform darin gefunden, so wird die größte Integerzahl zurückgegeben.
     */
    private static int determineLowestExistingVerbFormIndex(List<String> searchWords) {
        int lowestIndex = Integer.MAX_VALUE;

        for (String searchWord : searchWords) {
            int index = SORTED_VERB_SEARCH_WORDS.indexOf(searchWord);
            if (index != -1 && index < lowestIndex) {
                lowestIndex = index;
            }
        }

        return lowestIndex;
    }

}
