package de.duehl.vocabulary.japanese.logic.success;

import java.awt.Color;
import java.util.List;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanaData;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.symbol.Katakana;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.InternalKanaDataRequester;

/**
 * Diese Klasse ermittelt den Erfolg bei den letzten zehn Abfragen eines Liste von Katakana und
 * erzeugt daraus die passende Farbe sowie die Prozentzahl.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

public class KatakanaListTestSuccesssCalculator {

    /** Die Liste mit den Katakana, zu welchen der Erfolg berechnet werden soll. */
    private final List<Katakana> katakanaList;

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die Anzahl der maximal zehn Abfragen aller Katakana des Vokabulars. */
    private int lastCountSum;

    /** Die Anzahl der korrekten maximal zehn Abfragen aller Katakana des Vokabulars. */
    private int lastCorrectCountSum;

    /** Die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    private Color foregroundColor;

    /** Der Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar. */
    private String percentText;

    /** Die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    private double percent;

    /**
     * Der Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur Anzeige in der
     * VocabularyBar.
     */
    private String percentTextWithTwoDigitsAfterComma;

    /**
     * Konstruktor.
     *
     * @param katakanaList
     *            Die Liste mit den Katakana,  zu welchen der Erfolg berechnet werden soll.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     */
    public KatakanaListTestSuccesssCalculator(List<Katakana> katakanaList,
            FumikoDataStructures dataStructures) {
        this.katakanaList = katakanaList;
        this.dataStructures = dataStructures;
    }

    /** Führt die Berechnung durch. */
    public void calculate() {
        calculateCounts();
        foregroundColor = determineForegroundColor();
        determinePercentAndPercentText();
    }

    private void calculateCounts() {
        lastCountSum = 0;
        lastCorrectCountSum = 0;

        for (Katakana katakana : katakanaList) {
            calculateCountsForKatakana(katakana);
        }
    }

    private void calculateCountsForKatakana(Katakana katakana) {
        InternalKanaDataRequester requester = dataStructures.getInternalKanaDataRequester();
        InternalAdditionalKanaData data = requester.getInternalDataForKatakana(katakana);
        int lastCount;
        int lastCorrectCount;
        lastCount = data.getLastTenTestsCount();
        lastCorrectCount = data.getLastCorrectTestsCount();
        lastCountSum += lastCount;
        lastCorrectCountSum += lastCorrectCount;

        /* Damit nicht abgefragte Katakana nicht gut aussehen: */
        if (lastCount == 0) {
            lastCountSum += 10;
        }
    }

    private Color determineForegroundColor() {
        Options options = dataStructures.getOptions();
        if (options.isColorVocabularyDependingOnLastSuccess()) {
            VocableColors vocableColors = new VocableColors(options);
            return vocableColors.determineForegroundColor(lastCountSum, lastCorrectCountSum);
        }
        else {
            return Color.BLACK;
        }
    }

    private void determinePercentAndPercentText() {
        percentTextWithTwoDigitsAfterComma = NumberString.percent(lastCorrectCountSum, lastCountSum);

        percent = NumberString.percentAsNumber(lastCorrectCountSum, lastCountSum);

        String percentString = String.format("%.0f", percent);
        percentText = Text.fillWithSpacesAtFront(percentString, 3);
    }

    /**
     * Getter für die Vordergrundfarbe.
     *
     * Bei der entsprechenden Klasse für die Berechnung des Erfolgs für Vokabeln wird diese Methide
     * dazu benutzt, den Namen des Vokabulars in der VocabularyBar einzufärben. Ob es hier bei den
     * Katakana für diese Methode eine Verwendung gibt, muss man sehen.
     */
    public Color getForegroundColor() {
        return foregroundColor;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar.
     */
    public String getPercentText() {
        return percentText;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur
     * Anzeige in der VocabularyBar.
     */
    public String getPercentTextWithTwoDigitsAfterComma() {
        return percentTextWithTwoDigitsAfterComma;
    }

    /** Getter für die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    public double getPercent() {
        return percent;
    }

}
