package de.duehl.vocabulary.japanese.logic.success;

import java.awt.Color;
import java.util.List;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;

/**
 * Diese Klasse ermittelt den Erfolg bei den letzten zehn Abfragen eines Vokabulars und erzeugt
 * daraus die passende Farbe sowie die Prozentzahl zur Anzeige in der VocabularyBar.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

public class VocabularyTestSuccesssCalculator {

    /** Die Liste mit den Vokabeln, zu welchen der Erfolg berechnet werden soll. */
    private final List<Vocable> vocables;

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die Anzahl der maximal zehn Abfragen aller Vokabeln des Vokabulars. */
    private int lastCountSum;

    /** Die Anzahl der korrekten maximal zehn Abfragen aller Vokabeln des Vokabulars. */
    private int lastCorrectCountSum;

    /** Die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    private Color foregroundColor;

    /** Der Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar. */
    private String percentText;

    /** Die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    private double percent;

    /**
     * Der Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur Anzeige in der
     * VocabularyBar.
     */
    private String percentTextWithTwoDigitsAfterComma;

    /**
     * Konstruktor.
     *
     * @param vocabulary
     *            Das Vokabular zu welchem der Erfolg berechnet werden soll.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     */
    public VocabularyTestSuccesssCalculator(Vocabulary vocabulary,
            FumikoDataStructures dataStructures) {
        this(vocabulary.getVocables(), dataStructures);
    }

    /**
     * Konstruktor.
     *
     * @param vocables
     *            Die Liste mit den Vokabeln,  zu welchen der Erfolg berechnet werden soll.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     */
    public VocabularyTestSuccesssCalculator(List<Vocable> vocables,
            FumikoDataStructures dataStructures) {
        this.vocables = vocables;
        this.dataStructures = dataStructures;
    }

    /** Führt die Berechnung durch. */
    public void calculate() {
        calculateCounts();
        foregroundColor = determineForegroundColor();
        determinePercentAndPercentText();
    }

    private void calculateCounts() {
        lastCountSum = 0;
        lastCorrectCountSum = 0;

        for (Vocable vocable : vocables) {
            calculateCountsForVocable(vocable);
        }
    }

    private void calculateCountsForVocable(Vocable vocable) {
        InternalDataRequester requester = dataStructures.getInternalDataRequester();
        InternalAdditionalVocableData data = requester.getInternalDataForVocable(vocable);
        int lastCount;
        int lastCorrectCount;
        Options options = dataStructures.getOptions();
        TranslationDirection translationDirection = options.getTranslationDirection();
        if (translationDirection == TranslationDirection.JAPANESE_TO_GERMAN) {
            lastCount = data.getLastTenJapaneseToGermanTestsCount();
            lastCorrectCount = data.getLastCorrectJapaneseToGermanTestsCount();
        }
        else if (translationDirection == TranslationDirection.GERMAN_TO_JAPANESE) {
            lastCount = data.getLastGermanToJapaneseTestsCount();
            lastCorrectCount = data.getLastCorrectGermanToJapaneseTestsCount();
        }
        else {
            throw new RuntimeException("Unbekannte Übersetzungsrichtung " + translationDirection);
        }
        lastCountSum += lastCount;
        lastCorrectCountSum += lastCorrectCount;

        /* Damit nicht abgefragte Vokabeln (und Vokabularien) nicht gut aussehen: */
        if (lastCount == 0) {
            lastCountSum += 10;
        }
    }

    private Color determineForegroundColor() {
        Options options = dataStructures.getOptions();
        if (options.isColorVocabularyDependingOnLastSuccess()) {
            VocableColors vocableColors = new VocableColors(options);
            return vocableColors.determineForegroundColor(lastCountSum, lastCorrectCountSum);
        }
        else {
            return Color.BLACK;
        }
    }

    private void determinePercentAndPercentText() {
        percentTextWithTwoDigitsAfterComma = NumberString.percent(lastCorrectCountSum, lastCountSum);

        percent = NumberString.percentAsNumber(lastCorrectCountSum, lastCountSum);

        String percentString = String.format("%.0f", percent);
        percentText = Text.fillWithSpacesAtFront(percentString, 3);
    }

    /** Getter für die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    public Color getForegroundColor() {
        return foregroundColor;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar.
     */
    public String getPercentText() {
        return percentText;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur
     * Anzeige in der VocabularyBar.
     */
    public String getPercentTextWithTwoDigitsAfterComma() {
        return percentTextWithTwoDigitsAfterComma;
    }

    /** Getter für die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    public double getPercent() {
        return percent;
    }

}
