package de.duehl.vocabulary.japanese.ui.components;

import java.awt.GridLayout;

import javax.swing.ButtonGroup;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.ui.VocabularyTrainerGui;

/**
 * Diese Klasse stellt die grafische Oberfläche der Auswahl der Übersetzungsrichtung im
 * Vokabel-Trainers dar.
 *
 * @version 1.01     2024-08-07
 * @author Christian Dühl
 */

public class TranslationDirectionGuiWithRadioButtons {

    /** Die Programmoptionen. */
    private final Options options;

    /** Die grafische Oberfläche des Vokabel-Trainers. */
    private final VocabularyTrainerGui gui;

    /** Der Panel auf dem die Auswahl dargestellt wird. */
    private final JPanel panel;

    /** Der RadioButton für die Übersetzung von Japanisch in Deutsch. */
    private final JRadioButton japaneseToGermanTranslation;

    /** Der RadioButton für die Übersetzung von Japanisch in Deutsch. */
    private final JRadioButton germanToJapaneseTranslation;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param gui
     *            Die grafische Oberfläche des Vokabel-Trainers.
     */
    public TranslationDirectionGuiWithRadioButtons(Options options, VocabularyTrainerGui gui) {
        this.options = options;
        this.gui = gui;

        panel = new JPanel();
        japaneseToGermanTranslation = new JRadioButton("Japanisch - Deutsch");
        germanToJapaneseTranslation = new JRadioButton("Deutsch - Japanisch");

        init();
        selectRadioButton();
    }

    private void init() {
        initRadioButtons();
        initButtonGroup();
        initPanel();
    }

    private void initRadioButtons() {
        japaneseToGermanTranslation.addActionListener(e -> toggled());
        germanToJapaneseTranslation.addActionListener(e -> toggled());
    }

    private void toggled() {
        TranslationDirection direction;
        if (japaneseToGermanTranslation.isSelected()) {
            direction = TranslationDirection.JAPANESE_TO_GERMAN;
        }
        else {
            direction = TranslationDirection.GERMAN_TO_JAPANESE;
        }
        options.setTranslationDirection(direction);
        gui.showTranslationDirectionOnBarButtons();
        gui.setCorrectForegroundColorOfVocabularyBarsLater();
    }

    private void initButtonGroup() {
        ButtonGroup buttonGroup = new ButtonGroup();
        buttonGroup.add(japaneseToGermanTranslation);
        buttonGroup.add(germanToJapaneseTranslation);
    }

    private void initPanel() {
        panel.setLayout(new GridLayout(0, 1, 2, 2));
        GuiTools.createTitle(panel);

        panel.add(japaneseToGermanTranslation);
        panel.add(germanToJapaneseTranslation);
    }

    private void selectRadioButton() {
        TranslationDirection direction = options.getTranslationDirection();

        if (direction == TranslationDirection.JAPANESE_TO_GERMAN) {
            japaneseToGermanTranslation.setSelected(true);
        }
        else if (direction == TranslationDirection.GERMAN_TO_JAPANESE) {
            germanToJapaneseTranslation.setSelected(true);
        }
        else {
            throw new RuntimeException("Unbekannte TranslationDirection " + direction + ".");
        }
    }

    /** Getter für den Panel auf dem die Auswahl dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
