package de.duehl.vocabulary.japanese.ui.components.bars;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.GridLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.additional.LocationGetter;
import de.duehl.vocabulary.japanese.common.ui.resources.IconDefinitions;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.logic.VocabularyTrainerLogic;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjiset.KanjiSetBarsRefresher;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjiset.KanjiSetEditor;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.table.KanjiTableDialog;
import de.duehl.vocabulary.japanese.ui.tools.VocabularyTrainerUiTools;

/**
 * Diese Klasse stellt eine Kanji-Mengen im Dialog zur Verwaltung der Kanji-Mengen
 * (KanjiSetManagementDialog) dar.
 *
 * Man kann die Kanji-Menge bearbeiten oder löschen.
 *
 * Sortiert wird hier nicht!
 * So viele Kanji-Mengen braucht man ja gar nicht, dass das nötig wäre.
 *
 * @version 1.01     2025-11-27
 * @author Christian Dühl
 */

public class KanjiSetBar {

    /** Die benutzerdefinierte Kanji-Menge. */
    private final KanjiSet kanjiSet;

    /** Kann die Bars mit den Kanji-Mengen neu zeichnen. */
    private final KanjiSetBarsRefresher kanjiSetBarsRefresher;

    /** Die Logik des Vokabeltrainers. */
    private final VocabularyTrainerLogic logic;

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private final FumikoUiObjects uiObjects;

    /**
     * Das Objekt, das die aktuelle Position der Oberfläche des nutzenden Dialogs zurück geben
     * kann.
     */
    private final LocationGetter locationGetter;

    /** Die Liste mit den nicht erlaubten Namen für das Umbenennen dieser Kanji-Menge. */
    private final List<String> notAllowedNames;

    /** Das Panel auf dem die Bar mit der Kanji-Menge dargestellt wird. */
    private final JPanel panel;

    /**
     * Konstruktor.
     *
     * @param kanjiSet
     *            Die benutzerdefinierte Kanji-Menge.
     * @param allNames
     *            Die Liste mit den Namen aller Kanji-Mengen.
     * @param kanjiSetBarsRefresher
     *            Kann die Bars mit den Kanji-Mengen neu zeichnen.
     * @param logic
     *            Die Logik des Vokabeltrainers.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param locationGetter
     *            Das Objekt, das die aktuelle Position der Oberfläche des nutzenden Dialogs zurück geben kann.
     */
    public KanjiSetBar(KanjiSet kanjiSet, List<String> allNames,
            KanjiSetBarsRefresher kanjiSetBarsRefresher, VocabularyTrainerLogic logic,
            FumikoDataStructures dataStructures, FumikoUiObjects uiObjects,
            LocationGetter locationGetter) {
        this.kanjiSet = kanjiSet;
        this.kanjiSetBarsRefresher = kanjiSetBarsRefresher;
        this.logic = logic;
        this.dataStructures = dataStructures;
        this.uiObjects = uiObjects;
        this.locationGetter = locationGetter;

        notAllowedNames = new ArrayList<>();
        for (String name : allNames) {
            if (!name.equals(kanjiSet.getName())) {
                notAllowedNames.add(name);
            }
        }

        panel = new JPanel();

        createBarUi();
    }

    private void createBarUi() {
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(createNameLabel(), BorderLayout.CENTER);
        panel.add(createButtonsPart(), BorderLayout.EAST);
    }

    private Component createNameLabel() {
        JLabel label = new JLabel(kanjiSet.getName());
        GuiTools.biggerFont(label, 10);
        return label;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(1, 0, 2, 2));

        panel.add(createViewAsListButon());
        panel.add(createEditButon());
        panel.add(createDeleteButon());

        return panel;
    }

    private Component createViewAsListButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.SHOW_VOCABULARY_LIST, e -> viewKanjiSetAsKanjiList(),
                "Vokabeln anzeigen");
        if (kanjiSet.getSet().isEmpty()) {
            button.setEnabled(false);
        }
        return button;
    }

    private void viewKanjiSetAsKanjiList() {
        String title = "Alle Kanji aus der Menge '" + kanjiSet.getName() + "'";
        KanjiTableDialog dialog = new KanjiTableDialog(kanjiSet.getSet(), dataStructures, uiObjects,
                uiObjects.getInformer(), title, locationGetter.getLocation());
        dialog.setVisible(true);
    }

    private Component createEditButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.EDIT_OWN_LIST,
                e -> edit(),
                "Kanji-Menge bearbeiten");
        return button;
    }

    private void edit() {
        KanjiSetEditor editor = new KanjiSetEditor(kanjiSet, notAllowedNames, logic, dataStructures,
                uiObjects, locationGetter.getLocation());
        editor.setVisible(true);
        kanjiSetBarsRefresher.refreshKanjiSetBars();
    }

    private Component createDeleteButon() {
        JButton button = VocabularyTrainerUiTools.createPicturedButton(
                IconDefinitions.DELETE_OWN_LIST,
                e -> delete(),
                "Kanji-Menge löschen");
        button.setForeground(Color.RED);
        return button;
    }

    private void delete() {
        String name = kanjiSet.getName();
        String title = "Wirklich löschen?";
        String message = "Soll die Kanji-Menge '" + name + "' wirklich gelöscht werden?";
        boolean delete = GuiTools.askUser(panel, title, message);
        if (delete) {
            kanjiSetBarsRefresher.deleteKanjiSetFromListAndFileFromDisk(kanjiSet);
            kanjiSetBarsRefresher.refreshKanjiSetBars();
        }
    }

    /** Getter für das Panel auf dem die Bar mit der Kanji-Menge dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
