package de.duehl.vocabulary.japanese.ui.dialog.kanji.display;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.basics.text.Text;
import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.components.selections.SelectionsHelper;
import de.duehl.swing.ui.components.selections.StringSelection;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanjiData;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.display.findlists.SetsWithKanjiFinder;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.table.components.AllVocablesWithKanjiShower;

/**
 * Diese Klasse stellt ein einzelnes Kanji Panel dar.
 *
 * Gezeigt werden das Zeichen selbst, die deutsche Bedeutung, die ON-Lesungen und die kun-Lesungen.
 *
 * @version 1.01     2025-11-21
 * @author Christian Dühl
 */

public class KanjiPanel {

    /** Das angezeigte Kanji-Zeichen. */
    private final Kanji kanji;

    /** Der modale oder nicht modale Dialog bzw., Frame, der diesen Panel nutzt. */
    private final Component thisPanelUsingDialog;

    /** Erzeugt die Komponente, die das Kanji groß dargestellt. */
    private final BigKanjiDisplay bigKanjiDisplay;

    /** Gibt an, ob auch die internen Daten zum Kanji angezeigt werden sollen. */
    private boolean showInternalKanjiData;

    /**
     * Gibt an, ob das Zeichen auch nochmal in einem Viewer angezeigt werden soll, damit es
     * heraus kopiert werden kann.
     */
    private boolean showKanjiAsViewerToo;

    /** Der Panel auf dem das Kanji dargestellt wird. */
    private final JPanel panel;

    private final StringSelection kanjiAsCharacterSeletion;
    private final StringSelection germanMeaningSelection;
    private final StringSelection onLesungenSelection;
    private final StringSelection kunLesungenSelection;

    private final StringSelection numberInKanjiAndKanaBookSelection;
    private final StringSelection pageNumberInKanjiAndKanaBookSelection;

    private final StringSelection firstSeenDateSelection;
    private final StringSelection testCountSelection;
    private final StringSelection correctTestCountSelection;
    private final StringSelection lastTestDatetSelection;
    private final StringSelection lastCorrectTestDateSelection;
    private final StringSelection lastTenTestResultsSelection;

    private final JButton showAllVocableWithKanjiButton;
    private final JButton showAllKanjiSetsWithKanjiButton;

    /** Gibt an, ob auch die Buttons am Ende angezeigt werden sollen. */
    private boolean showButtonsToo;

    /** Die Datenstrukturen des Vokabeltrainers. */
    private FumikoDataStructures dataStructures;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private FumikoUiObjects uiObjects;

    /** Die Position des nutzenden Dialogs. */
    private Point location;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private LongTimeProcessInformer informer;

    /**
     * Das Objekt, welches zu einem Kanji alle Kanji-Mengen findet, in denen dieses Kanji vorkommt.
     */
    private SetsWithKanjiFinder setsWithKanjiFinder;

    /**
     * Konstruktor.
     *
     * @param kanji
     *            Das angezeigte Kanji-Zeichen.
     * @param thisPanelUsingDialog
     *            Der modale oder nicht modale Dialog bzw., Frame, der diesen Panel nutzt.
     */
    public KanjiPanel(Kanji kanji, Component thisPanelUsingDialog) {
        this.kanji = kanji;
        this.thisPanelUsingDialog = thisPanelUsingDialog;

        bigKanjiDisplay = new BigKanjiDisplay();
        bigKanjiDisplay.setBiggerFontSize(200); // Default: 400

        showInternalKanjiData = false;
        showKanjiAsViewerToo = false;
        showButtonsToo = false;


        panel = new JPanel();

        kanjiAsCharacterSeletion = new StringSelection("Das Kanji (noch einmal zum rauskopieren)");
        germanMeaningSelection = new StringSelection("Deutsche Bedeutung");
        onLesungenSelection = new StringSelection("ON-Lesungen");
        kunLesungenSelection = new StringSelection("kun-Lesungen");

        numberInKanjiAndKanaBookSelection = new StringSelection(
                "Nummer im im Buch 'Kanji und Kana' von Wolfgang Hadamitzky");
        pageNumberInKanjiAndKanaBookSelection = new StringSelection(
                "Seite im im Buch 'Kanji und Kana' von Wolfgang Hadamitzky");

        firstSeenDateSelection = new StringSelection("Zuerst gesehen");
        testCountSelection = new StringSelection("Anzahl Test");
        correctTestCountSelection = new StringSelection("Anzahl erfolgreicher Test");
        lastTestDatetSelection = new StringSelection("zuletzt getestet am");
        lastCorrectTestDateSelection = new StringSelection("zuletzt erfolgreich getestet am");
        lastTenTestResultsSelection = new StringSelection("Erfolg der letzten Tests");

        showAllVocableWithKanjiButton = new JButton("Zeige alle Vokabeln mit diesem Kanji an");
        showAllKanjiSetsWithKanjiButton = new JButton(
                "Zeige alle Kanji-Mengen mit diesem Kanji an");

        init();
    }

    private void init() {
        initAssetFocusableViewer(kanjiAsCharacterSeletion);
        initAssetFocusableViewer(kanjiAsCharacterSeletion);
        initAssetFocusableViewer(germanMeaningSelection);
        initAssetFocusableViewer(onLesungenSelection);
        initAssetFocusableViewer(kunLesungenSelection);

        initAssetFocusableViewer(numberInKanjiAndKanaBookSelection);
        initAssetFocusableViewer(pageNumberInKanjiAndKanaBookSelection);

        initAssetFocusableViewer(firstSeenDateSelection);
        initAssetFocusableViewer(testCountSelection);
        initAssetFocusableViewer(correctTestCountSelection);
        initAssetFocusableViewer(lastTestDatetSelection);
        initAssetFocusableViewer(lastCorrectTestDateSelection);
        initAssetFocusableViewer(lastTenTestResultsSelection);
    }

    private void initAssetFocusableViewer(StringSelection selection) {
        SelectionsHelper.initSelectionAsViewer(selection);
        selection.setFocusable(true);
    }

    /**
     * Legt fest, dass das Zeichen auch nochmal in einem Viewer angezeigt werden soll, damit es
     * heraus kopiert werden kann.
     */
    public void showKanjiAsViewerToo() {
        showKanjiAsViewerToo = true;
    }

    /**
     * Legt fest, dass auch die internen Daten zum Kanji angezeigt werden sollen.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     */
    public void showInternalKanjiDataToo(FumikoDataStructures dataStructures) {
        showInternalKanjiData = true;
        this.dataStructures = dataStructures;
    }

    /**
     * Legt fest, dass auch die Buttons am Ende angezeigt werden sollen.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param location
     *            Die Position des nutzenden Dialogs.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public void showButtonsToo(FumikoDataStructures dataStructures, FumikoUiObjects uiObjects,
            Point location, LongTimeProcessInformer informer) {
        this.dataStructures = dataStructures;
        this.uiObjects = uiObjects;
        this.informer = informer;
        this.location = location;

        setsWithKanjiFinder = new SetsWithKanjiFinder(dataStructures, thisPanelUsingDialog);

        showButtonsToo = true;

        showAllVocableWithKanjiButton.addActionListener(e -> showAllVocablesWithKanji());
        showAllKanjiSetsWithKanjiButton.addActionListener(e -> showAllKanjiSetsWithKanji());
    }

    private void showAllVocablesWithKanji() {
        if (showButtonsToo) {
            AllVocablesWithKanjiShower vocablesShower = new AllVocablesWithKanjiShower(
                    dataStructures, uiObjects, location, informer);
            vocablesShower.showKanji(kanji);
        }
    }

    private void showAllKanjiSetsWithKanji() {
        if (showButtonsToo) {
            setsWithKanjiFinder.findKanjiSetsWithKanji(kanji);
            // XXX
        }
    }

    /**
     * Erzeugt die Gui.
     *
     * Möchte man die internen Daten anzeigen, so muss man zuvor die Methode
     * showInternalKanjiDataToo(requester) aufrufen!
     */
    public void createGui() {
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel); Das sollte man wohl außerhalb entscheiden.

        panel.add(bigKanjiDisplay.getComponent(), BorderLayout.CENTER);
        panel.add(createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart(),
                BorderLayout.SOUTH);
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        if (showKanjiAsViewerToo) {
            panel.add(kanjiAsCharacterSeletion.getPanel());
        }
        panel.add(createGermanMeaningOnLesungenAndKunLesungenPart());
        if (showInternalKanjiData) {
            panel.add(createInternalKanjiDataPart());
        }
        if (showButtonsToo) {
            panel.add(showAllVocableWithKanjiButton);
            panel.add(showAllKanjiSetsWithKanjiButton);
        }

        return panel;
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(germanMeaningSelection.getPanel());
        panel.add(onLesungenSelection.getPanel());
        panel.add(kunLesungenSelection.getPanel());

        return panel;
    }

    private Component createInternalKanjiDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(numberInKanjiAndKanaBookSelection.getPanel());
        panel.add(pageNumberInKanjiAndKanaBookSelection.getPanel());

        panel.add(firstSeenDateSelection.getPanel());
        panel.add(testCountSelection.getPanel());
        panel.add(correctTestCountSelection.getPanel());
        panel.add(lastTestDatetSelection.getPanel());
        panel.add(lastCorrectTestDateSelection.getPanel());
        panel.add(lastTenTestResultsSelection.getPanel());

        return panel;
    }

    /** Getter für den Panel auf dem das Kanji dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Zeigt das Kanji an. Kann mehrfach aufgerufen werden. */
    public void showKanji() {
        bigKanjiDisplay.showKanji(kanji);

        kanjiAsCharacterSeletion.setText(kanji.getCharacter());
        germanMeaningSelection.setText(kanji.getGermanMeaning());
        onLesungenSelection.setText(Text.joinWithCommaAndBlank(kanji.getOnLesungen()));
        kunLesungenSelection.setText(Text.joinWithCommaAndBlank(kanji.getKunLesungen()));

        if (showInternalKanjiData) {
            numberInKanjiAndKanaBookSelection.setText(kanji.getNumberInKanjiAndKanaBook());
            pageNumberInKanjiAndKanaBookSelection.setText(kanji.getPageNumberInKanjiAndKanaBook());

            InternalKanjiDataRequester kanjiRequester =
                    dataStructures.getInternalKanjiDataRequester();
            InternalAdditionalKanjiData data = kanjiRequester.getInternalDataForKanji(kanji);

            firstSeenDateSelection.setText(data.getFirstSeenDate().toString());
            testCountSelection.setText(data.getTestCount());
            correctTestCountSelection.setText(data.getCorrectTestCount());
            lastTestDatetSelection.setText(data.getLastTestDate().toString());
            lastCorrectTestDateSelection.setText(data.getLastCorrectTestDate().toString());
            lastTenTestResultsSelection.setText(data.getLastTenTestResultsAsStorageString());
        }
    }

}
