package de.duehl.vocabulary.japanese.ui.dialog.kanji.table.components;

import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.vocables.lister.VocabularyListerDialog;

/**
 * Diese Klasse zeigt zu einem Kanji alle Vokabeln, in denen das Kanji vorkommt, an.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

public class AllVocablesWithKanjiShower {

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private final FumikoUiObjects uiObjects;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer informer;

    /** Die Position des nutzenden Dialogs. */
    private final Point parentLocation;

    /**
     * Konstruktor.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public AllVocablesWithKanjiShower(FumikoDataStructures dataStructures,
            FumikoUiObjects uiObjects, Point parentLocation, LongTimeProcessInformer informer) {
        this.dataStructures = dataStructures;
        this.uiObjects = uiObjects;
        this.parentLocation = parentLocation;
        this.informer = informer;
    }

    /** Zeigt zum übergebenen Kanji die Vokabeln, in denen es vorkommt, an. */
    public void showKanji(Kanji kanji) {
        List<Vocable> vocables = determineVocablesWithKanji(kanji);
        if (!vocables.isEmpty()) {
            VocabularyListerDialog dialog = new VocabularyListerDialog(vocables,
                    "Vokabeln mit Kanji " + kanji.getCharacter(), dataStructures, uiObjects,
                    parentLocation, informer);
            dialog.setVisible(true);
        }
    }

    private List<Vocable> determineVocablesWithKanji(Kanji kanji) {
        String searchedKanjiCharacter = kanji.getCharacter();

        List<Vocable> vocables = new ArrayList<>();

        List<Vocabulary> vocabularies = dataStructures.getVocabularies();
        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                if (vocable.getKanji().contains(searchedKanjiCharacter)) {
                    vocables.add(vocable);
                }
            }
        }

        return vocables;
    }

}
