package de.duehl.vocabulary.japanese.ui.dialog.vocables.detail.related;

import java.awt.Point;
import java.util.List;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.grammar.VerbSearchWords;
import de.duehl.vocabulary.japanese.grammar.find.RelatedVocablesFinder;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.vocables.lister.VocabularyListerDialog;

/**
 * Diese Klasse findet zu einer Vokabel zugehörige Verb- und Adjektivformen zu finden, wenn die
 * Vokabel ein Verb bzw. ein Adjektiv ist.
 *
 * @version 1.01     2025-11-26
 * @author Christian Dühl
 */

class VerbsAndAdjectivesFinder {

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private final FumikoUiObjects uiObjects;

    /**
     * Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese Klasse
     * verwendet.
     */
    private final Point location;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer longTimeProcessInformer;

    /**
     * Konstruktor.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     * @param longTimeProcessInformer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public VerbsAndAdjectivesFinder(FumikoDataStructures dataStructures, FumikoUiObjects uiObjects,
            Point location, LongTimeProcessInformer longTimeProcessInformer) {
        this.dataStructures = dataStructures;
        this.uiObjects = uiObjects;
        this.location = location;
        this.longTimeProcessInformer = longTimeProcessInformer;
    }

    /**
     * Findet zur übergebenen Vokabel, welche ein Verb sein muss, alle Vokabeln mit den bekannten
     * Verbformen zu diesem Verb.
     *
     * @param vocable
     *            Die Verb-Vokabel.
     */
    public void findRelatedVerbs(Vocable vocable) {
        RelatedVocablesFinder finder = new RelatedVocablesFinder(dataStructures.getVocabularies(),
                vocable, v -> v.isVerb());
        finder.find();
        List<Vocable> relatedVerbs = finder.getRelatedVocables();
        String description = "Verbformen zum Verb" + " "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces();

        VerbSearchWords.sortVocablesByVerbForms(relatedVerbs);

        listVocables(relatedVerbs, description);
    }

    /**
     * Findet zur übergebenen Vokabel, welche ein Adjektiv sein muss, alle Vokabeln mit den
     * bekannten Adjektivformen zu diesem Adjektiv.
     *
     * @param vocable
     *            Die Adjektiv-Vokabel.
     */
    public void findRelatedAdjectives(Vocable vocable) {
        RelatedVocablesFinder finder = new RelatedVocablesFinder(dataStructures.getVocabularies(),
                vocable, v -> v.isAdjective());
        finder.find();
        List<Vocable> relatedAdjectives = finder.getRelatedVocables();
        String description = "Adjektivformen zum Adjektiv" + " "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces();

        VerbSearchWords.sortVocablesByVerbForms(relatedAdjectives);

        listVocables(relatedAdjectives, description);
    }

    /**
     * Zeigt die übergebenen Vokabeln als Liste an.
     *
     * @param vocables
     *            Die anzuzeigenden Vokabeln.
     * @param description
     *            Die Beschreibung der anzuzeigenden Vokabeln.
     */
    private void listVocables(List<Vocable> vocables, String description) {
        VocabularyListerDialog dialog = new VocabularyListerDialog(vocables,
                description, dataStructures, uiObjects, location, longTimeProcessInformer);
        dialog.setVisible(true);
    }

}
