package de.duehl.vocabulary.japanese.ui.tools;

import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.resources.IconLoader;
import de.duehl.vocabulary.japanese.common.ui.resources.IconDefinitions;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.symbol.Hiragana;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.data.symbol.Katakana;
import de.duehl.vocabulary.japanese.logic.success.HiraganaListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.KanjiListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.KatakanaListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.VocabularyTestSuccesssCalculator;

/**
 * Diese Klasse stellt Hilfsmethoden für die Gui zur Verfügung.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

public class VocabularyTrainerUiTools {

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Vokabeln erzielt wurde.
     *
     * @param vocables
     *            Die Liste mit den Vokabeln, zu welchen der Erfolg berechnet werden soll.
     * @param frontTitle
     *            Der vordere Teil des Titels.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Vokabeln in
     *         Klammern dahinter.
     */
    public static String generateTitleWithVocabularyTestSuccesss(List<Vocable> vocables,
            String frontTitle, FumikoDataStructures dataStructures) {
        VocabularyTestSuccesssCalculator calculator =
                new VocabularyTestSuccesssCalculator(vocables, dataStructures);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return frontTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Kanji erzielt wurde.
     *
     * @param vocables
     *            Die Liste mit den abzufragenden Kanji.
     * @param frontTitle
     *            Der vordere Teil des Titels.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Kanji in
     *         Klammern dahinter.
     */
    public static String generateTitleWithKanjiTestSuccesss(List<Kanji> kanjiToTest,
            String frontTitle, FumikoDataStructures dataStructures) {
        KanjiListTestSuccesssCalculator calculator = new KanjiListTestSuccesssCalculator(
                kanjiToTest, dataStructures);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return frontTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Hiragana erzielt wurde.
     *
     * @param hiraganaToTest
     *            Die Liste mit den abzufragenden Hiragana.
     * @param testTitle
     *            Der Titels.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Hiragana in
     *         Klammern dahinter.
     */
    public static String generateTitleWithHiraganaTestSuccesss(List<Hiragana> hiraganaToTest,
            String testTitle, FumikoDataStructures dataStructures) {
        HiraganaListTestSuccesssCalculator calculator = new HiraganaListTestSuccesssCalculator(
                hiraganaToTest, dataStructures);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return testTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Katakana erzielt wurde.
     *
     * @param hiraganaToTest
     *            Die Liste mit den abzufragenden Katakana.
     * @param testTitle
     *            Der Titels.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Katakana in
     *         Klammern dahinter.
     */
    public static String generateTitleWithKatakanaTestSuccesss(List<Katakana> katakanaToTest,
            String testTitle, FumikoDataStructures dataStructures) {
        KatakanaListTestSuccesssCalculator calculator = new KatakanaListTestSuccesssCalculator(
                katakanaToTest, dataStructures);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return testTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Vokabeln
     * erzielt wurde.
     *
     * @param vocables
     *            Die Liste mit den abzufragenden Vokabeln.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Vokabeln.
     */
    public static double createLastTenTestsPercent(List<Vocable> vocables,
            FumikoDataStructures dataStructures) {
        VocabularyTestSuccesssCalculator calculator =
                new VocabularyTestSuccesssCalculator(vocables, dataStructures);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Kanji
     * erzielt wurde.
     *
     * @param kanjiList
     *            Die Liste mit den abzufragenden Kanji.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Kanji.
     */
    public static double createLastTenKanjiTestsPercent(List<Kanji> kanjiList,
            FumikoDataStructures dataStructures) {
        KanjiListTestSuccesssCalculator calculator = new KanjiListTestSuccesssCalculator(kanjiList,
                dataStructures);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Hiragana
     * erzielt wurde.
     *
     * @param hiraganaList
     *            Die Liste mit den abzufragenden Hiragana.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Hiragana.
     */
    public static double createLastTenHiraganaTestsPercent(List<Hiragana> hiraganaList,
            FumikoDataStructures dataStructures) {
        HiraganaListTestSuccesssCalculator calculator = new HiraganaListTestSuccesssCalculator(
                hiraganaList, dataStructures);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Katakana
     * erzielt wurde.
     *
     * @param katakanaList
     *            Die Liste mit den abzufragenden Katakana.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Katakana.
     */
    public static double createLastTenKatakanaTestsPercent(List<Katakana> katakanaList,
            FumikoDataStructures dataStructures) {
        KatakanaListTestSuccesssCalculator calculator = new KatakanaListTestSuccesssCalculator(
                katakanaList, dataStructures);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Erzeugt einen Button mit einem Icon.
     *
     * @param pictureIdentifier
     *            Bezeichner des Bildes, im Normalfall eine der öffentlichen Konstanten aus
     *            IconDefinitions.
     * @param listener
     *            Wird beim Betätigen des Buttons ausgeführt.
     * @param toolTipText
     *            Wird angezeigt, wenn die Maus eine Weile über dem Button verweilt.
     * @return Der erzeugte Button.
     */
    public static JButton createPicturedButton(String pictureIdentifier, ActionListener listener,
            String toolTipText) {
        JButton button = new JButton(loadIcon(pictureIdentifier));
        button.setBorder(BorderFactory.createRaisedBevelBorder());
        button.addActionListener(listener);
        button.setToolTipText(toolTipText);
        return button;
    }

    private static Icon loadIcon(String imageIdentifier) {
        IconDefinitions iconDefinitions = new IconDefinitions();
        IconLoader iconLoader = iconDefinitions.createIconLoader();
        return iconLoader.loadIcon(imageIdentifier);
    }

    /**
     * Erzeugt für Abschlussdialoge von Abfragen die richtige Beschreibung.
     *
     * TODO: Später soll auch eine passende Fumiko-Chan mitgeliefert werden, dann muss ich eine
     * kleine Datenklasse mit der Nachricht und dem Bild programmieren und zurückgeben.
     *
     * @param whatWasTestet
     *            Was getestet wurde,. z.B. "Vokabeln" oder "Kanji".
     * @param testVerb
     *            Wie man das beantworten des Tests bezeichnet, z.B. "übersetzt" oder
     *            "beantwortet".
     * @param total
     *            Die Anzahl aller abgefragten Dinge.
     * @param correct
     *            Die Anzahl der richtig abgefragten Dinge.
     * @return Die Beschreibung der Leistung des Benutzers.
     */
    public static String createAllTestsMessage(String whatWasTestet, String testVerb, int total,
            int correct) {
        String correctFromTotal = "Sie haben " + NumberString.taupu(correct) + " von "
                + NumberString.taupu(total) + " " + whatWasTestet + " richtig " + testVerb + ".";

        String successDescription;
        if (total == correct) {
            successDescription = "Sehr gut, alles richtig " + testVerb + "!";
        }
        else {
            double percent = NumberString.percentAsNumber(correct, total);
            if (percent >= 90) {
                successDescription = "Das war schon sehr gut!";
            }
            else if (percent >= 75) {
                successDescription = "Ein gutes Ergebnis!";
            }
            else if (percent >= 50) {
                successDescription = "Immerhin war mindestens die Hälfte richtig.";
            }
            else if (percent >= 25) {
                successDescription = "Es wird, aber es gibt noch viel Luft nach oben.";
            }
            else {
                successDescription = "Üben, üben, üben ...";
            }
        }

        return correctFromTotal + "<br/>" + successDescription;
    }

}
