package de.duehl.basics.datetime.data;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.Objects;

import de.duehl.basics.datetime.DateAndTime;

/**
 * Diese Klasse steht für ein Datum mit einer Uhrzeit in den Formaten "DD.MM.YYYY" und "HH:MM:SS".
 *
 * @version 1.01     2024-04-24
 * @author Christian Dühl
 */

public class DateAndTimeString {

    /** Das Datum im Format "DD.MM.YYYY". */
    private final String date;

    /** Die Uhrzeit im Format "HH:MM:SS". */
    private final String time;

    /**
     * Konstruktor.
     *
     * @param date
     *            Das Datum im Format "DD.MM.YYYY".
     * @param time
     *            Die Uhrzeit im Format "HH:MM:SS".
     */
    public DateAndTimeString(String date, String time) {
        this.date = date;
        this.time = time;

        checkDate(date);
        checkTime(time);
    }

    /**
     * Konstruktor.
     *
     * @param dateAndTime
     *            Datum und Uhrzeit in einem DateAndTime-Objekt.
     */
    public DateAndTimeString(DateAndTime dateAndTime) {
        this(dateAndTime.getDate().toString(), dateAndTime.getTime().toString());
    }

    /** Konstruktor mit dem aktuellen Datum und der aktuellen Uhrzeit. */
    public DateAndTimeString() {
        this(new DateAndTime());
    }

    /** Getter für das Datum im Format "DD.MM.YYYY". */
    public String getDate() {
        return date;
    }

    /** Getter für die Uhrzeit im Format "HH:MM:SS". */
    public String getTime() {
        return time;
    }

    @Override
    public String toString() {
        return "DateAndTimeString [date=" + date + ", time=" + time + "]";
    }

    @Override
    public int hashCode() {
        return Objects.hash(date, time);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        DateAndTimeString other = (DateAndTimeString) obj;
        return Objects.equals(date, other.date) && Objects.equals(time, other.time);
    }

    /**
     * Überprüft das Format des übergebenen Datums und wirft eine Ausnahme, wenn es nicht in
     * Ordnung ist.
     */
    public static void checkDate(String date) {
        if (!isDateOk(date)) {
            throw new RuntimeException("Das Datum hat nicht das Format 'DD.MM.YYYY'.\n"
                    + "\t" + "date = " + date + "\n");
        }
    }

    /**
     * Überprüft das Format der übergebenen Uhrzeit und wirft eine Ausnahme, wenn es nicht in
     * Ordnung ist.
     */
    public static void checkTime(String time) {
        if (!isTimeOk(time)) {
            throw new RuntimeException("Die Uhrzeit hat nicht das Format 'HH:MM:SS'.\n"
                    + "\t" + "time = " + time + "\n");
        }
    }

    /** Überprüft das Format des übergebenen Datums. */
    public static boolean isDateOk(String date) {
        return date.matches("\\d{2}\\.\\d{2}\\.\\d{4}");
    }

    /** Überprüft das Format der übergebenen Uhrzeit. */
    public static boolean isTimeOk(String time) {
        return time.matches("\\d{2}:\\d{2}:\\d{2}");
    }

}
