package de.duehl.basics.regex;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Diese Klasse stellt Hilfsmethoden rund um reguläre Ausdrücke zur Verfügung.
 *
 * @version 1.01     2024-11-19
 * @author Christian Dühl
 */

public class RegexHelper {

    /** Erstellt einen regulären Ausdruck aus einer Liste von möglichen Ausprägungen. */
    public static String buildOrRegexFromList(List<String> list) {
        return buildOrRegexFromListInternal(list, true);
    }

    /** Erstellt einen regulären Ausdruck aus einer Liste von möglichen Ausprägungen. */
    public static String buildNotQuotingOrRegexFromList(List<String> list) {
        return buildOrRegexFromListInternal(list, false);
    }

    public static String buildOrRegexFromListInternal(List<String> list, boolean quote) {
        StringBuilder regex = new StringBuilder();
        regex.append("(?:");
        boolean first = true;
        for (String thing : list) {
            if (first) {
                first = false;
            }
            else {
                regex.append("|");
            }
            if (quote) {
                regex.append("\\Q");
            }
            regex.append(thing);
            if (quote) {
                regex.append("\\E");
            }
        }
        regex.append(")");
        return regex.toString();
    }

    /**
     * Gibt eine Liste mit den gequoteten Elementen der Liste zurück.
     *
     * @param list
     *            Liste, deren Elemente gequotet werden sollen.
     * @return Liste mit den gequoteten Elementen.
     */
    public static List<String> createQuotedList(List<String> list) {
        List<String> quotedList = new ArrayList<>();

        for (String element : list) {
            String quoted = Pattern.quote(element);
            quotedList.add(quoted);
        }

        return quotedList;
    }

    /** Erstellt aus einer Liste von regulären Ausdrücken eine Liste von kompilierten Pattern. */
    public static List<Pattern> compileListOfRegexes(List<String> regexes) {
        List<Pattern> patterns = new ArrayList<>();

        for (String regex : regexes) {
            Pattern pattern = Pattern.compile(regex);
            patterns.add(pattern);
        }

        return patterns;
    }

}
