package de.duehl.basics.regex;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Diese Klasse steht für eine Pattern und einen Ersetzungstext. Analog zu s///
 * in Perl.
 *
 * @version 1.01     2015-02-26
 * @author Christian Dühl
 */

public class Substitution {

    /** Kompiliertes Pattern. */
    private final Pattern pattern;

    /** Ersetzungstext (kann $1, ... enthalten). */
    private final String exchange;

    /**
     * Konstruktor.
     *
     * @param regex
     *            Regulärer Ausdruck.
     * @param exchange
     *            Ersetzungstext (kann $1, ... enthalten).
     */
    public Substitution(final String regex, final String exchange) {
        this(regex, 0, exchange);
    }

    /**
     * Konstruktor.
     *
     * @param regex
     *            Regulärer Ausdruck.
     * @param flags
     *            Flags für den regulären Ausdruck.
     * @param exchange
     *            Ersetzungstext (kann $1, ... enthalten).
     */
    public Substitution(final String regex, final int flags,
            final String exchange) {
        pattern = Pattern.compile(regex, flags);
        this.exchange = exchange;
    }

    /**
     * Führt auf der Eingabe die Ersetzung durch und gibt das Ergebnis zurück.
     *
     * @param input
     *            Eingabe.
     * @return Ergebnis.
     */
    public String substitute(final String input) {
        Matcher matcher = pattern.matcher(input);
        return matcher.replaceAll(exchange);
    }

    /** Getter für den regulären Ausdruck. */
    public String getRegex() {
        return pattern.pattern();
    }

}
