package de.duehl.basics.text.xml.own;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.duehl.basics.text.Lines;
import de.duehl.basics.text.xml.XmlAnalyser;

/**
 * Diese Klasse stellt Hilfsmethoden rund um sehr einfache, eigene XML-Dateien bereit.
 *
 * Sie sollten nicht für XML-Dateien aus anderen Quellen benutzt werden!
 *
 * @version 1.01     2021-11-23
 * @author Christian Dühl
 */

public class OwnXmlHelper {

    /**
     * Hüllt den übergebenen einfachen Inhalt in einen XML-Tag der Art <tag>content</tag> ein.
     *
     * @param tag
     *            Bezeichner des Elements (im Beispiel 'tag').
     * @param content
     *            Einzubettender Inhalt (im Beispiel 'content').
     * @return Tag mit dem Inhalt (im Beispiel <tag>content</tag>).
     */
    public static String singleUniqueXmlElementToXmlString(String tag, String content) {
        return "<" + tag + ">" + content + "</" + tag + ">";
    }

    /**
     * Liest den Inhalt eines einfachen XML-Elements der Form <tag>content</tag> aus der
     * übergebenen Zeile einer XML-Datei aus.
     *
     * @param tag
     *            Bezeichner des Elements (im Beispiel 'tag').
     * @param line
     *            Aus einer eigenen XML-Datei eingelesene Zeile mit den Daten eines Objektes.
     * @return Inhalt des XML-Elements (im Beispiel 'content').
     */
    public static String readContentOfSingleUniqueXmlElementInXmlLine(String tag, String line) {
        Pattern pattern = Pattern.compile(
                "\\Q<" + tag + ">\\E(.*)\\Q</" + tag + ">\\E");

        Matcher matcher = pattern.matcher(line);
        if (matcher.find()) {
            return matcher.group(1);
        }
        else {
            throw new IllegalArgumentException("Kein <" + tag + ">...</" + tag
                    + "> im der XML-Zeile gefunden!\n\tline = " + line);
        }
    }

    /**
     * Liest den Inhalt eines einfachen XML-Elements der Form <tag>content</tag> aus den
     * zusammengesetzten, übergebenen Zeilen einer XML-Datei aus.
     *
     * @param tag
     *            Bezeichner des Elements (im Beispiel 'tag').
     * @param line
     *            Aus einer eigenen XML-Datei eingelesene Zeilen (in einem String) mit den Daten
     *            eines Objektes.
     * @return Inhalt des XML-Elements (im Beispiel 'content').
     */
    public static String readContentOfSingleUniqueXmlElementInXmlMultipleLine(String tag,
            String line) {
        Lines lines = Lines.createFromMultipleLineInOneString(line);
        Lines contentLines = XmlAnalyser.getLinesInTag(lines, tag);
        return contentLines.joinWithLineBreak();
    }

}
