package de.duehl.basics.replacements;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class ReplacementSorterTest {

    @Test
    public void normalSort() {
        /*
         * Gedachter ursprünglicher Text:
         *     Die Birne liegt auf dem Tisch.
         *     Der Apfel hängt am Apfelbaum.
         */

        String text = "Die <<obst:3>> liegt auf dem Tisch."
                + "Der <<obst:1>> hängt am <<obst:2>>baum.";
        String tag = "obst";
        List<String> foundDataList = CollectionsHelper.buildListFrom("Apfel", "Apfel", "Birne");

        assertEquals(3, foundDataList.size());
        assertEquals("Apfel", foundDataList.get(0));
        assertEquals("Apfel", foundDataList.get(1));
        assertEquals("Birne", foundDataList.get(2));

        ReplacementSorter<String> sorter = new ReplacementSorter<>(text, tag, foundDataList);
        sorter.sort();
        String actualText = sorter.getText();
        String expectedText = "Die <<obst:1>> liegt auf dem Tisch."
                + "Der <<obst:2>> hängt am <<obst:3>>baum.";
        assertEquals(expectedText, actualText);
        assertEquals(3, foundDataList.size());
        assertEquals("Birne", foundDataList.get(0));
        assertEquals("Apfel", foundDataList.get(1));
        assertEquals("Apfel", foundDataList.get(2));
    }

    @Test
    public void sortWithoutFoundDataList() {
        /*
         * Gedachter ursprünglicher Text:
         *     Die Birne liegt auf dem Tisch.
         *     Der Apfel hängt am Apfelbaum.
         */

        String text = "Die <<obst:3>> liegt auf dem Tisch."
                + "Der <<obst:1>> hängt am <<obst:2>>baum.";
        String tag = "obst";

        ReplacementSorter<String> sorter =
                ReplacementSorter.createSorterWithoutFoundDataList(text, tag);
        sorter.sort();
        String actualText = sorter.getText();
        String expectedText = "Die <<obst:1>> liegt auf dem Tisch."
                + "Der <<obst:2>> hängt am <<obst:3>>baum.";
        assertEquals(expectedText, actualText);
    }

    @Test
    public void normalSortLonger() {
        /*
         * Gedachter ursprünglicher Text:
         *     Das Fahrrad lehnt am Gartenzaun, mit einer Orange im Korb.
         *     Im Auto liegen reife Birnen.
         *     Am Lenker des Rollers hängen Kirschen.
         *     Apfelsinen schmücken den Sitz des Dreirades.
         */

        String text = "Das <<fahrzeug:3>> lehnt am Gartenzaun, mit einer Orange im Korb."
                + "Im <<fahrzeug:1>> liegen reife Birnen."
                + "Am Lenker des <<fahrzeug:4>>s hängen Kirschen."
                + "Apfelsinen schmücken den Sitz des <<fahrzeug:2>>es.";
        String tag1 = "fahrzeug";
        List<String> foundDataList1 = CollectionsHelper.buildListFrom("Auto", "Dreirad", "Fahrrad",
                "Roller");

        assertEquals(4, foundDataList1.size());
        assertEquals("Auto", foundDataList1.get(0));
        assertEquals("Dreirad", foundDataList1.get(1));
        assertEquals("Fahrrad", foundDataList1.get(2));
        assertEquals("Roller", foundDataList1.get(3));

        ReplacementSorter<String> sorter1 = new ReplacementSorter<>(text, tag1, foundDataList1);
        sorter1.sort();
        String actualText = sorter1.getText();
        String expectedText = "Das <<fahrzeug:1>> lehnt am Gartenzaun, mit einer Orange im Korb."
                + "Im <<fahrzeug:2>> liegen reife Birnen."
                + "Am Lenker des <<fahrzeug:3>>s hängen Kirschen."
                + "Apfelsinen schmücken den Sitz des <<fahrzeug:4>>es.";
        assertEquals(expectedText, actualText);
        assertEquals(4, foundDataList1.size());
        assertEquals("Fahrrad", foundDataList1.get(0));
        assertEquals("Auto", foundDataList1.get(1));
        assertEquals("Roller", foundDataList1.get(2));
        assertEquals("Dreirad", foundDataList1.get(3));
    }

    @Test
    public void normalSortMixed() {
        /*
         * Gedachter ursprünglicher Text:
         *     Das Fahrrad lehnt am Gartenzaun, mit einer Orange im Korb.
         *     Im Auto liegen reife Birnen.
         *     Am Lenker des Rollers hängen Kirschen.
         *     Bananen schmücken den Sitz des Dreirades.
         */

        String text = "Das <<fahrzeug:3>> lehnt am Gartenzaun, mit einer <<obst:4>> im Korb."
                + "Im <<fahrzeug:1>> liegen reife <<obst:2>>n."
                + "Am Lenker des <<fahrzeug:4>>s hängen <<obst:3>>n."
                + "<<obst:1>>n schmücken den Sitz des <<fahrzeug:2>>es.";
        String tag1 = "fahrzeug";
        List<String> foundDataList1 = CollectionsHelper.buildListFrom("Auto", "Dreirad", "Fahrrad",
                "Roller");

        assertEquals(4, foundDataList1.size());
        assertEquals("Auto", foundDataList1.get(0));
        assertEquals("Dreirad", foundDataList1.get(1));
        assertEquals("Fahrrad", foundDataList1.get(2));
        assertEquals("Roller", foundDataList1.get(3));


        String tag2 = "obst";
        List<String> foundDataList2 = CollectionsHelper.buildListFrom("Banane", "Birne", "Kirsche",
                "Orange");

        assertEquals(4, foundDataList2.size());
        assertEquals("Banane", foundDataList2.get(0));
        assertEquals("Birne", foundDataList2.get(1));
        assertEquals("Kirsche", foundDataList2.get(2));
        assertEquals("Orange", foundDataList2.get(3));

        ReplacementSorter<String> sorter1 = new ReplacementSorter<>(text, tag1, foundDataList1);
        sorter1.sort();
        text = sorter1.getText();
        ReplacementSorter<String> sorter2 = new ReplacementSorter<>(text, tag2, foundDataList2);
        sorter2.sort();

        String actualText = sorter2.getText();
        String expectedText = ""
                + "Das <<fahrzeug:1>> lehnt am Gartenzaun, mit einer <<obst:1>> im Korb."
                + "Im <<fahrzeug:2>> liegen reife <<obst:2>>n."
                + "Am Lenker des <<fahrzeug:3>>s hängen <<obst:3>>n."
                + "<<obst:4>>n schmücken den Sitz des <<fahrzeug:4>>es.";
        assertEquals(expectedText, actualText);

        assertEquals(4, foundDataList1.size());
        assertEquals("Fahrrad", foundDataList1.get(0));
        assertEquals("Auto", foundDataList1.get(1));
        assertEquals("Roller", foundDataList1.get(2));
        assertEquals("Dreirad", foundDataList1.get(3));

        assertEquals(4, foundDataList2.size());
        assertEquals("Orange", foundDataList2.get(0));
        assertEquals("Birne", foundDataList2.get(1));
        assertEquals("Kirsche", foundDataList2.get(2));
        assertEquals("Banane", foundDataList2.get(3));
    }

}
