package de.duehl.basics.text;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;
import static de.duehl.basics.text.Text.LINE_BREAK;
import static de.duehl.basics.text.Text.LINE_BREAK_PLACE_HOLDER;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.junit.Ignore;
import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.logic.Pair;
import de.duehl.basics.text.data.FoundSearch;

import static de.duehl.basics.collections.CollectionsHelper.printAssertEquals;

@SuppressWarnings("unused")
public class TextTest {

    @Test
    public void break001() {
        String text = "Mal sehen was so passiert! ";
        String expected = "Mal sehen was so passiert! ";
        int lineLength = 80;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break002() {
        String text = "Mal sehen was so passiert! ";
        String expected = "Mal sehen was" + LINE_BREAK + "so passiert! ";
        int lineLength = 15;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break003() {
        String text = "Mal sehen was so passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was" + LINE_BREAK
                + "so" + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break004() {
        String text = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was" + LINE_BREAK
                + "so" + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES" + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break005() {
        String text = "Mal_sehen_was_so_AUSSERGEWÖHNLICHLANGES_passiert!";
        String expected = text;
        int lineLength = 5;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break006() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK + "cd" + LINE_BREAK + "ef";
        int lineLength = 2;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break007() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK + "cd" + LINE_BREAK + "ef";
        int lineLength = 1;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break008() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK + "cd" + LINE_BREAK + "ef";
        int lineLength = 3;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break009() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK + "cd" + LINE_BREAK + "ef";
        int lineLength = 4;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break010() {
        String text = "ab cd ef";
        String expected = "ab cd" + LINE_BREAK + "ef";
        int lineLength = 5;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break011() {
        String text = "ab cd ef gh ij kl mn op qr";
        String expected = "ab cd" + LINE_BREAK + "ef gh" + LINE_BREAK + "ij kl"
                + LINE_BREAK + "mn op" + LINE_BREAK + "qr";
        int lineLength = 5;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void break012() {
        String text = "Mal\nsehen\r\nwas so AUSSERGEWÖHNLICHLANGES passiert! ";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was so" + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES" + LINE_BREAK
                + "passiert! ";
        int lineLength = 10;
        String actual = Text.addLineBreaks(text, lineLength);
        assertEquals(expected, actual);
    }

    @Test
    public void countCharInString01() {
        String text = "foobarbazo";
        String search = "o";
        int expected = 3;
        int actual = Text.countPartInString(text, search);
        assertEquals(expected, actual);
    }

    @Test
    public void countCharInString02() {
        String text = "hahahahahahahahahaha";
        String search = "haha";
        int expected = 5;
        int actual = Text.countPartInString(text, search);
        assertEquals(expected, actual);
    }

    @Test
    public void countCharInString03() {
        String text = "10.234.642,12";
        String search = ".";
        int expected = 2;
        int actual = Text.countPartInString(text, search);
        assertEquals(expected, actual);
    }

    @Test
    public void countCharInStringPipes() {
        String text = "(Softwareentwicklung sowie IT-Consulting).|dmaic software GmbH & Co. KG";
        String search = "|";
        int expected = 1;
        int actual = Text.countPartInString(text, search);
        assertEquals(expected, actual);
    }

    @Test
    public void stripWhitespaceEmptyInput() {
        String input = "";
        String expected = "";
        String actual = Text.stripWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void stripWhitespaceSpaceBeforeAndBehind() {
        String input = " abc ";
        String expected = "abc";
        String actual = Text.stripWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void stripWhitespaceMultipleSpaces() {
        String input = " \r\na     b\tc \n";
        String expected = "a b c";
        String actual = Text.stripWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void removeWhitespaceEmptyInput() {
        String input = "";
        String expected = "";
        String actual = Text.removeWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void removeWhitespaceSpaceBeforeAndBehind() {
        String input = " abc ";
        String expected = "abc";
        String actual = Text.removeWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void removeWhitespaceMultipleSpaces() {
        String input = " \r\na     b\tc \n";
        String expected = "abc";
        String actual = Text.removeWhitespace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsAreEqual01() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = 5;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual02() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = 4;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual03() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = 3;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual04() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = 1;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual05() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = 0;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test (expected = IllegalArgumentException.class)
    public void firstCharsAreEqual06() {
        String text1 = "abcd";
        String text2 = "abcd";
        int numberOfChars = -1;
        Text.firstCharsAreEqual(text1, text2, numberOfChars);
    }

    @Test
    public void firstCharsAreEqual07() {
        String text1 = "abcd";
        String text2 = "abce";
        int numberOfChars = 3;
        assertTrue(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual08() {
        String text1 = "abcd";
        String text2 = "abee";
        int numberOfChars = 3;
        assertFalse(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual09() {
        String text1 = "abcd";
        String text2 = "aece";
        int numberOfChars = 3;
        assertFalse(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void firstCharsAreEqual10() {
        String text1 = "abcd";
        String text2 = "ebce";
        int numberOfChars = 3;
        assertFalse(Text.firstCharsAreEqual(text1, text2, numberOfChars));
    }

    @Test
    public void findAllPositionsEmptySearch() {
        String searchText = "";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositions(searchText, text);
        assertTrue(null != indices);
        assertTrue(10 == indices.size());
        assertEquals(0, (int) indices.get(0));
        assertEquals(1, (int) indices.get(1));
        assertEquals(2, (int) indices.get(2));
        assertEquals(3, (int) indices.get(3));
        assertEquals(4, (int) indices.get(4));
        assertEquals(5, (int) indices.get(5));
        assertEquals(6, (int) indices.get(6));
        assertEquals(7, (int) indices.get(7));
        assertEquals(8, (int) indices.get(8));
        assertEquals(9, (int) indices.get(9));
    }

    @Test
    public void findAllPositionsMuliple() {
        List<String> searchTexts = CollectionsHelper.buildListFrom(" ", ",", ";", ".");
        String text = "Foo, bar und baz; außerdem blubb.";
        List<Integer> indices = Text.findAllPositions(searchTexts, text);
        assertTrue(null != indices);
        assertEquals(8, indices.size());
        assertEquals(3, (int) indices.get(0));
        assertEquals(4, (int) indices.get(1));
        assertEquals(8, (int) indices.get(2));
        assertEquals(12, (int) indices.get(3));
        assertEquals(16, (int) indices.get(4));
        assertEquals(17, (int) indices.get(5));
        assertEquals(26, (int) indices.get(6));
        assertEquals(32, (int) indices.get(7));
    }

    @Test
    public void findAllPositions01() {
        String searchText = "foofoo";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositions(searchText, text);
        assertTrue(null != indices);
        assertTrue(2 == indices.size());
        assertEquals(0, (int) indices.get(0));
        assertEquals(3, (int) indices.get(1));
    }

    @Test
    public void findAllPositions02() {
        String searchText = "foo";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositions(searchText, text);
        assertTrue(null != indices);
        assertTrue(3 == indices.size());
        assertEquals(0, (int) indices.get(0));
        assertEquals(3, (int) indices.get(1));
        assertEquals(6, (int) indices.get(2));
    }

    @Test
    public void findAllPositionsWithoutOverlappingEmptySearch() {
        String searchText = "";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositionsWithoutOverlapping(searchText, text);
        assertTrue(null != indices);
        assertTrue(10 == indices.size());
        assertEquals(0, (int) indices.get(0));
        assertEquals(1, (int) indices.get(1));
        assertEquals(2, (int) indices.get(2));
        assertEquals(3, (int) indices.get(3));
        assertEquals(4, (int) indices.get(4));
        assertEquals(5, (int) indices.get(5));
        assertEquals(6, (int) indices.get(6));
        assertEquals(7, (int) indices.get(7));
        assertEquals(8, (int) indices.get(8));
        assertEquals(9, (int) indices.get(9));
    }

    @Test
    public void findAllPositionsWithoutOverlapping01() {
        String searchText = "foofoo";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositionsWithoutOverlapping(searchText, text);
        assertTrue(null != indices);
        assertTrue(1 == indices.size());
        assertEquals(0, (int) indices.get(0));
    }

    @Test
    public void findAllPositionsWithoutOverlapping02() {
        String searchText = "foo";
        String text = "foofoofoo";
        List<Integer> indices = Text.findAllPositionsWithoutOverlapping(searchText, text);
        assertTrue(null != indices);
        assertTrue(3 == indices.size());
        assertEquals(0, (int) indices.get(0));
        assertEquals(3, (int) indices.get(1));
        assertEquals(6, (int) indices.get(2));
    }

    @Test
    public void splitByWhitespaceEmptyInput() {
        String input = "";
        List<String> parts = Text.splitByWhitespace(input);
        assertEquals(0, parts.size());
    }

    @Test
    public void splitByWhitespace01() {
        String input = "foo";
        List<String> parts = Text.splitByWhitespace(input);
        assertTrue(1 == parts.size());
        assertEquals(input, parts.get(0));
    }

    @Test
    public void splitByWhitespace02() {
        String input = "foo bar";
        List<String> parts = Text.splitByWhitespace(input);
        assertTrue(2 == parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
    }

    @Test
    public void splitByWhitespace03() {
        String input = "foo bar       baz";
        List<String> parts = Text.splitByWhitespace(input);
        assertTrue(3 == parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void lineBreakHiding01() {
        String input = "a" + LINE_BREAK + "b" + LINE_BREAK + "c";
        String expected = "a" + LINE_BREAK_PLACE_HOLDER + "b" + LINE_BREAK_PLACE_HOLDER + "c";
        String actual = Text.hideLineBreaks(input);
        assertEquals(expected, actual);
    }

    @Test
    public void lineBreakHiding02() {
        String input = "a" + LINE_BREAK_PLACE_HOLDER + "b" + LINE_BREAK_PLACE_HOLDER + "c";
        String expected = "a" + LINE_BREAK + "b" + LINE_BREAK + "c";
        String actual = Text.showLineBreaksAfterHiding(input);
        assertEquals(expected, actual);
    }

    @Test
    public void lineBreaksToSpace01() {
        String input = "a" + LINE_BREAK + "b" + LINE_BREAK + "c";
        String expected = "a b c";
        String actual = Text.lineBreaksToSpace(input);
        assertEquals(expected, actual);
    }

    @Test
    public void removeLlineBreaksAndTabulators() {
        String input = "a" + LINE_BREAK + " b\t  c    " + LINE_BREAK + "d";
        String expected = "a b c d";
        String actual = Text.removeLineBreaksAndTabulators(input);
        assertEquals(expected, actual);
    }

    @Test
    public void removeLlineBreaksAndTabulatorsLineBreakAtEnd() {
        String input = "a" + LINE_BREAK;
        String expected = "a";
        String actual = Text.removeLineBreaksAndTabulators(input);
        assertEquals(expected, actual);
    }

    @Test
    public void stripContentsFromLines01() {
        String line1 = "#Dies";
        String line2 = "ist";
        String line3 = "# ein   ";
        String line4 = "Test";
        List<String> lines = new ArrayList<>();
        lines.add(line1);
        lines.add(line2);
        lines.add(line3);
        lines.add(line4);
        List<String> stripped = Text.stripCommentsFromLines(lines);
        assertEquals(2, stripped.size());
        assertEquals(line2, stripped.get(0));
        assertEquals(line4, stripped.get(1));
    }

    @Test
    public void stripContentsFromLines02() {
        String line1 = "#Dies";
        String line2 = "ist";
        String line3 = "# ein   ";
        String line4 = " # Test";
        List<String> lines = new ArrayList<>();
        lines.add(line1);
        lines.add(line2);
        lines.add(line3);
        lines.add(line4);
        List<String> stripped = Text.stripCommentsFromLines(lines);
        assertEquals(2, stripped.size());
        assertEquals(line2, stripped.get(0));
        assertEquals(line4, stripped.get(1));
    }

    @Test
    public void removeLastCharactersEmptyString() {
        String input = "";
        String actual = Text.removeLastCharacters(input, 5);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test (expected = NullPointerException.class)
    public void removeLastCharactersNullString() {
        String input = null;
        Text.removeLastCharacters(input, 5);
    }

    @Test
    public void removeLastCharactersFromTooSmallString() {
        String input = "abc";
        String actual = Text.removeLastCharacters(input, 5);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLastCharactersFromEqualLengthString() {
        String input = "abcde";
        String actual = Text.removeLastCharacters(input, 5);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLastCharactersFromLongerString() {
        String input = "abcdef";
        String actual = Text.removeLastCharacters(input, 5);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLastCharactersToDeleteZeroChars() {
        String input = "abcdef";
        String actual = Text.removeLastCharacters(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test (expected = StringIndexOutOfBoundsException.class)
    public void removeLastCharactersToDeleteANegativeNumberOfChars() {
        String input = "abcdef";
        String actual = Text.removeLastCharacters(input, -4);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesNormalCase() {
        String input = "abc";
        String actual = Text.fillWithSpaces(input, 5);
        String expected = "abc  ";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesZeroNumber() {
        String input = "abc";
        String actual = Text.fillWithSpaces(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesNegativeNumber() {
        String input = "abc";
        String actual = Text.fillWithSpaces(input, -7);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesNumberEqulasInputLength() {
        String input = "abc";
        String actual = Text.fillWithSpaces(input, input.length());
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesEmptyInput() {
        String input = "";
        String actual = Text.fillWithSpaces(input, 3);
        String expected = "   ";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNormalCase() {
        String input = "abc";
        String actual = Text.fillWithSpacesAtFront(input, 5);
        String expected = "  abc";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontZeroNumber() {
        String input = "abc";
        String actual = Text.fillWithSpacesAtFront(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNegativeNumber() {
        String input = "abc";
        String actual = Text.fillWithSpacesAtFront(input, -7);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontNumberEqulasInputLength() {
        String input = "abc";
        String actual = Text.fillWithSpacesAtFront(input, input.length());
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithSpacesAtFrontEmptyInput() {
        String input = "";
        String actual = Text.fillWithSpacesAtFront(input, 3);
        String expected = "   ";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithLinesNormalCase() {
        String input = "abc";
        String actual = Text.fillWithLines(input, 5);
        String expected = "abc--";
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithLinesZeroNumber() {
        String input = "abc";
        String actual = Text.fillWithLines(input, 0);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithLinesNegativeNumber() {
        String input = "abc";
        String actual = Text.fillWithLines(input, -7);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithLinesNumberEqulasInputLength() {
        String input = "abc";
        String actual = Text.fillWithLines(input, input.length());
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void fillWithLinesEmptyInput() {
        String input = "";
        String actual = Text.fillWithLines(input, 3);
        String expected = "---";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherWithEmptyList() {
        List<String> terms = new ArrayList<>();
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(terms.size(), cleanedTerms.size());
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherWithEmptySet() {
        Set<String> terms = new HashSet<>();
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(terms.size(), cleanedTerms.size());
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherTwoWordsContainingEachOther() {
        List<String> terms = new ArrayList<>();
        terms.add("Blume");
        terms.add("Blumenstrauß");
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(1, cleanedTerms.size());
        assertEquals("Blumenstrauß", cleanedTerms.get(0));
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherTwoSameWords() {
        List<String> terms = new ArrayList<>();
        terms.add("Blumenstrauß");
        terms.add("Blumenstrauß");
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(1, cleanedTerms.size());
        assertEquals("Blumenstrauß", cleanedTerms.get(0));
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherTwoWordsContainingEachOtherInternal() {
        List<String> terms = new ArrayList<>();
        String term1 = "Franz-Josef-Strauß Weg 17";
        String term2 = "Josef";
        terms.add(term1);
        terms.add(term2);
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(1, cleanedTerms.size());
        assertEquals(term1, cleanedTerms.get(0));
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherTwoWordsContainingEachOtherInternal2() {
        List<String> terms = new ArrayList<>();
        String term1 = "Josef";
        String term2 = "Franz-Josef-Strauß Weg 17";
        terms.add(term1);
        terms.add(term2);
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(1, cleanedTerms.size());
        assertEquals(term2, cleanedTerms.get(0));
    }

    @Test
    public void cleanCollectionOfStringsToNotContainingEachOtherThreeWordsContainingEachOther() {
        List<String> terms = new ArrayList<>();
        String term1 = "Franz-Josef-Strauß Weg 17";
        String term2 = "Josef-Strauß";
        String term3 = "Josef";
        terms.add(term1);
        terms.add(term2);
        terms.add(term3);
        List<String> cleanedTerms = Text.cleanCollectionOfStringsToNotContainingEachOther(terms);
        assertEquals(1, cleanedTerms.size());
        assertEquals(term1, cleanedTerms.get(0));
    }

    @Test
    public void getLastPartAfterDividerNotContainingDivider() {
        String input = "foo.bar.baz";
        String actual = Text.getLastPartAfterDivider(input, ";");
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void getLastPartAfterDividerNormalCase() {
        String input = "foo.bar.baz";
        String actual = Text.getLastPartAfterDivider(input, ".");
        String expected = "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void getLastPartAfterDividerNothingBehindDevider() {
        String input = "foo.bar.baz.";
        String actual = Text.getLastPartAfterDivider(input, ".");
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void joinThreeParts() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String delimiter = "#;#";

        String actual = Text.join(delimiter, parts);
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinEmptyList() {
        List<String> parts = new ArrayList<>();

        String delimiter = "#;#";

        String actual = Text.join(delimiter, parts);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithTabulator() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithTabulator(parts);
        String delimiter = "\t";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithLineBreak() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithLineBreak(parts);
        String delimiter = LINE_BREAK;
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithLineBreaks2() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        list.add("bar");
        list.add("baz");
        String actual = Text.joinWithLineBreak(list);
        String expected = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithLineBreaksListWithOnlyOneElemnt() {
        List<String> list = new ArrayList<>();
        list.add("foo");
        String actual = Text.joinWithLineBreak(list);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithPipe() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithPipe(parts);
        String delimiter = "|";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinEptyListWithPipe() {
        List<String> parts = new ArrayList<>();

        String actual = Text.joinWithPipe(parts);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithBlank() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithBlank(parts);
        String delimiter = " ";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithSemicolon() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithSemicolon(parts);
        String delimiter = ";";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithColon() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithColon(parts);
        String delimiter = ":";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithComma() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithComma(parts);
        String delimiter = ",";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithCommaAndBlank() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithCommaAndBlank(parts);
        String delimiter = ", ";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinWithRauteSemicolonRaute() {
        List<String> parts = new ArrayList<>();
        parts.add("foo");
        parts.add("bar");
        parts.add("baz");

        String actual = Text.joinWithRauteSemicolonRaute(parts);
        String delimiter = "#;#";
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinThreePartsViaEllipse() {
        String delimiter = "#;#";

        String actual = Text.join(delimiter, "foo", "bar", "baz");
        String expected = "foo" + delimiter + "bar" + delimiter + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void joinNoPartsViaEllipse() {
        String delimiter = "#;#";

        String actual = Text.join(delimiter);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void splitByWhitespaceSize() {
        String input = "foo bar \t \n\rbaz";
        List<String> parts = Text.splitByWhitespace(input);
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByWhitespace() {
        String input = "foo bar \t baz";
        List<String> parts = Text.splitByWhitespace(input);
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitByMultipleSpaces() {
        String input = "eins zwei  drei   vier    fünf";
        List<String> parts = Text.splitByMultipleSpaces(input, 3);
        assertEquals(3, parts.size());
        assertEquals("eins zwei  drei", parts.get(0));
        assertEquals("vier", parts.get(1));
        assertEquals("fünf", parts.get(2));
    }

    @Test (expected = IllegalArgumentException.class)
    public void splitByMultipleSpacesWithZeroSpaces() {
        String input = "eins zwei  drei   vier    fünf";
        Text.splitByMultipleSpaces(input, 0);
    }

    @Test
    public void splitBySize() {
        String input = "foo | bar";
        List<String> parts = Text.splitBy(input, "\\s*\\|\\s*");
        assertEquals(2, parts.size());
    }

    @Test
    public void splitBySize3() {
        String input = "foo | bar | baz";
        List<String> parts = Text.splitBy(input, "\\s*\\|\\s*");
        assertEquals(3, parts.size());
    }

    @Test
    public void splitBy() {
        String input = "foo | bar | baz";
        List<String> parts = Text.splitBy(input, "\\s*\\|\\s*");
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitByHashSize() {
        String input = "foo # bar";
        List<String> parts = Text.splitBy(input, "\\s*#\\s*");
        assertEquals(2, parts.size());
    }

    @Test
    public void splitByHashSize3() {
        String input = "foo # bar # baz";
        List<String> parts = Text.splitBy(input, "\\s*#\\s*");
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByHash() {
        String input = "foo # bar # baz";
        List<String> parts = Text.splitBy(input, "\\s*#\\s*");
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitByNothingSize() {
        String input = "abc";
        List<String> parts = Text.splitBy(input, "");
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByNothing() {
        String input = "abc";
        List<String> parts = Text.splitBy(input, "");
        assertEquals("a", parts.get(0));
        assertEquals("b", parts.get(1));
        assertEquals("c", parts.get(2));
    }

    @Test
    public void splitEmptyStringByNothingSize() {
        String input = "";
        List<String> parts = Text.splitBy(input, "");
        assertEquals(0, parts.size());
    }

    @Test
    public void splitByPipeSize() {
        String input = "foo | bar";
        List<String> parts = Text.splitByPipe(input);
        assertEquals(2, parts.size());
    }

    @Test
    public void splitByPipe() {
        String input = "foo | bar | baz";
        List<String> parts = Text.splitByPipe(input);
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitStringWithputPipeByPipeSize() {
        String input = "abc";
        List<String> parts = Text.splitByPipe(input);
        assertEquals(1, parts.size());
    }

    @Test
    public void splitStringWithputPipeByPipe() {
        String input = "foo";
        List<String> parts = Text.splitByPipe(input);
        assertEquals("foo", parts.get(0));
    }

    @Test
    public void splitEmptyStringByPipeSize() {
        String input = "";
        List<String> parts = Text.splitByPipe(input);
        assertEquals(0, parts.size());
    }

    @Test
    public void splitByPipeWithoutPipeSize() {
        String input = "foo bar";
        List<String> parts = Text.splitByPipe(input);
        assertEquals(1, parts.size());
    }

    @Test
    public void splitByPipeWithoutPipe() {
        String input = "foo bar";
        List<String> parts = Text.splitByPipe(input);
        assertEquals(input, parts.get(0));
    }

    @Test
    public void splitByPipeNotConsumingWhitespace() {
        String input = "foo | bar | baz";
        List<String> parts = Text.splitByPipeNotConsumingWhitespace(input);
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar ", parts.get(1));
        assertEquals(" baz", parts.get(2));
    }

    @Test
    public void splitByQuotedBlankOrHashSize() {
        String input = "foo '#' bar ' ' baz baz2 # baz3";
        List<String> parts = Text.splitBy(input, "\\s*'(#| )'\\s*");
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByQuotedBlankOrHash() {
        String input = "foo '#' bar ' ' baz baz2 # baz3";
        List<String> parts = Text.splitBy(input, "\\s*'(#| )'\\s*");
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz baz2 # baz3", parts.get(2));
    }

    @Test
    public void splitByQuotedBlankOrHashAsMethodSize() {
        String input = "foo '#' bar ' ' baz baz2 # baz3";
        List<String> parts = Text.splitByQuotedBlankOrHash(input);
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByQuotedBlankOrHashAsMethod() {
        String input = "foo '#' bar ' ' baz baz2 # baz3";
        List<String> parts = Text.splitByQuotedBlankOrHash(input);
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz baz2 # baz3", parts.get(2));
    }

    @Test
    public void splitByAdditionSize() {
        String input = "foo + bar + baz baz2 # baz3";
        List<String> parts = Text.splitByAddition(input);
        assertEquals(3, parts.size());
    }

    @Test
    public void splitByAddition() {
        String input = "foo + bar + baz baz2 # baz3";
        List<String> parts = Text.splitByAddition(input);
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz baz2 # baz3", parts.get(2));
    }

    @Test
    public void splitBySubtractionSize() {
        String input = "foo - bar - baz baz2 # baz3";
        List<String> parts = Text.splitBySubtraction(input);
        assertEquals(3, parts.size());
    }

    @Test
    public void splitBySubtraction() {
        String input = "foo - bar - baz baz2 # baz3";
        List<String> parts = Text.splitBySubtraction(input);
        assertEquals(3, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz baz2 # baz3", parts.get(2));
    }

    @Test
    public void splitByKomma() {
        String input = "foo , bar, baz ,foo,bar";
        List<String> parts = Text.splitByKomma(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByTabulator() {
        String input = "foo \t bar\t baz \tfoo\tbar";
        List<String> parts = Text.splitByTabulator(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByTabulatorMultipleTabulators() {
        String input = "foo\t\tbar";
        List<String> parts = Text.splitByTabulator(input);
        assertEquals(3, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("", parts.get(1));
        assertEquals("bar", parts.get(2));
    }

    @Test
    public void splitByTabulatorWithNoTabulatorInInput() {
        String input = "foo bar baz";
        List<String> parts = Text.splitByTabulator(input);
        assertEquals(1, parts.size());
        assertEquals("foo bar baz", parts.get(0));
    }

    @Test
    public void splitByTabulatorEmptyInput() {
        String input = "";
        List<String> parts = Text.splitByTabulator(input);
        assertEquals(0, parts.size());
    }

    @Test
    public void splitByDot() {
        String input = "foo.bar.baz";
        List<String> parts = Text.splitByDot(input);
        assertEquals(3, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitByDotMultipleDots() {
        String input = "foo..baz";
        List<String> parts = Text.splitByDot(input);
        assertEquals(3, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("", parts.get(1));
        assertEquals("baz", parts.get(2));
    }

    @Test
    public void splitByDot2() {
        String input = "de.duehl.basics.text";
        List<String> parts = Text.splitByDot(input);
        assertEquals(4, parts.size());
        assertEquals("de", parts.get(0));
        assertEquals("duehl", parts.get(1));
        assertEquals("basics", parts.get(2));
        assertEquals("text", parts.get(3));
    }

    @Test
    public void splitByDotNotConsumingWhitespace() {
        String input = "de. duehl . basics .text";
        List<String> parts = Text.splitByDotNotConsumingWhitespace(input);
        assertEquals(4, parts.size());
        assertEquals("de", parts.get(0));
        assertEquals(" duehl ", parts.get(1));
        assertEquals(" basics ", parts.get(2));
        assertEquals("text", parts.get(3));
    }

    @Test
    public void splitByEqualSign() {
        String input = "foo = bar";
        List<String> parts = Text.splitByEqualSign(input);
        assertEquals(2, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
    }

    @Test
    public void splitByTabulatorNotConsumingWhitespace() {
        String input = "foo \t bar\t baz \tfoo\tbar";
        List<String> parts = Text.splitByTabulatorNotConsumingWhitespace(input);
        assertEquals(5, parts.size());
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar", parts.get(1));
        assertEquals(" baz ", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitBySemicolon() {
        String input = "foo ; bar; baz ;foo;bar";
        List<String> parts = Text.splitBySemicolon(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitBySemicolonWithNoSemikolonInInput() {
        String input = "foo bar baz";
        List<String> parts = Text.splitBySemicolon(input);
        assertEquals(1, parts.size());
        assertEquals("foo bar baz", parts.get(0));
    }

    @Test
    public void splitBySemicolonEmptyInput() {
        String input = "";
        List<String> parts = Text.splitBySemicolon(input);
        assertEquals(0, parts.size());
    }

    @Test
    public void splitBySemicolonNotConsumingWhitespace() {
        String input = "foo ; bar; baz ;foo;bar";
        List<String> parts = Text.splitBySemicolonNotConsumingWhitespace(input);
        assertEquals(5, parts.size());
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar", parts.get(1));
        assertEquals(" baz ", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByColon() {
        String input = "foo : bar: baz :foo:bar";
        List<String> parts = Text.splitByColon(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByColonNotConsumingWhitespace() {
        String input = "foo : bar: baz :foo:bar";
        List<String> parts = Text.splitByColonNotConsumingWhitespace(input);
        assertEquals(5, parts.size());
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar", parts.get(1));
        assertEquals(" baz ", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitBySlashOrBackSlash() {
        String input = "de/duehl\\basics/text";
        List<String> parts = Text.splitBySlashOrBackSlash(input);
        assertEquals(4, parts.size());
        assertEquals("de", parts.get(0));
        assertEquals("duehl", parts.get(1));
        assertEquals("basics", parts.get(2));
        assertEquals("text", parts.get(3));
    }

    @Test
    public void splitBySlashOrBackSlashNotConsumingWhitespace() {
        String input = "de/ duehl \\ basics /text";
        List<String> parts = Text.splitBySlashOrBackSlashNotConsumingWhitespace(input);
        assertEquals(4, parts.size());
        assertEquals("de", parts.get(0));
        assertEquals(" duehl ", parts.get(1));
        assertEquals(" basics ", parts.get(2));
        assertEquals("text", parts.get(3));
    }

    @Test
    public void splitBySpecial() {
        String input = "foo #;# bar#;# baz #;#foo#;#bar";
        List<String> parts = Text.splitBySpecial(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByRauteSemicolonRaute() {
        String input = "foo #;# bar#;# baz #;#foo#;#bar";
        List<String> parts = Text.splitByRauteSemicolonRaute(input);
        assertEquals(5, parts.size());
        assertEquals("foo", parts.get(0));
        assertEquals("bar", parts.get(1));
        assertEquals("baz", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitBySpecialNotConsumingWhitespace() {
        String input = "foo #;# bar#;# baz #;#foo#;#bar";
        List<String> parts = Text.splitBySpecialNotConsumingWhitespace(input);
        assertEquals(5, parts.size());
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar", parts.get(1));
        assertEquals(" baz ", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void splitByRauteSemicolonRauteNotConsumingWhitespace() {
        String input = "foo #;# bar#;# baz #;#foo#;#bar";
        List<String> parts = Text.splitByRauteSemicolonRauteNotConsumingWhitespace(input);
        assertEquals(5, parts.size());
        assertEquals("foo ", parts.get(0));
        assertEquals(" bar", parts.get(1));
        assertEquals(" baz ", parts.get(2));
        assertEquals("foo", parts.get(3));
        assertEquals("bar", parts.get(4));
    }

    @Test
    public void removeLineBreaksBlankInput() {
        String in = "";
        String actual = Text.removeLineBreaks(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksSingleSpace() {
        String in = " ";
        String actual = Text.removeLineBreaks(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksOneLineBreak() {
        String in = "foo" + Text.LINE_BREAK + "bar";
        String actual = Text.removeLineBreaks(in);
        String expected = "foo bar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksOneRN() {
        String in = "foo\r\nbar";
        String actual = Text.removeLineBreaks(in);
        String expected = "foo bar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksOneNR() {
        String in = "foo\n\rbar";
        String actual = Text.removeLineBreaks(in);
        String expected = "foo  bar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksOneN() {
        String in = "foo\nbar";
        String actual = Text.removeLineBreaks(in);
        String expected = "foo bar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreaksOneR() {
        String in = "foo\rbar";
        String actual = Text.removeLineBreaks(in);
        String expected = "foo bar";
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksBlankInput() {
        String in = "";
        String actual = Text.repairLineBreaks(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksSingleSpace() {
        String in = " ";
        String actual = Text.repairLineBreaks(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksOneLineBreak() {
        String in = "foo" + Text.LINE_BREAK + "bar";
        String actual = Text.repairLineBreaks(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksOneRN() {
        String in = "foo\r\nbar";
        String actual = Text.repairLineBreaks(in);
        String expected = "foo" + Text.LINE_BREAK + "bar";
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksOneNR() {
        String in = "foo\n\rbar";
        String actual = Text.repairLineBreaks(in);
        String expected = "foo" + Text.LINE_BREAK + Text.LINE_BREAK + "bar";
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksOneN() {
        String in = "foo\nbar";
        String actual = Text.repairLineBreaks(in);
        String expected = "foo" + Text.LINE_BREAK + "bar";
        assertEquals(expected, actual);
    }

    @Test
    public void repairLineBreaksOneR() {
        String in = "foo\rbar";
        String actual = Text.repairLineBreaks(in);
        String expected = "foo" + Text.LINE_BREAK + "bar";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesBlankInput() {
        String in = "";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = in;
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesSingleSpace() {
        String in = " ";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesMultipleSpaces() {
        String in = "                   ";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesMultipleSpacesAroundChar() {
        String in = "             a      ";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesMultipleSpacesFront() {
        String in = "             a";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesMultipleSpacesRear() {
        String in = "a            ";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesMultipleSpacesBetween() {
        String in = "a            b";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "a b";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesTwoSpacesBetween() {
        String in = "a  b";
        String actual = Text.trimAndCompactSpaces(in);
        String expected = "a b";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesEmptyInput() {
        String input = "";
        String actual = Text.trimAndCompactSpaces(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpacesInputWithoutSpaces() {
        String input = "foobarbaz";
        String actual = Text.trimAndCompactSpaces(input);
        String expected = "foobarbaz";
        assertEquals(expected, actual);
    }

    @Test
    public void trimAndCompactSpaces() {
        String input = "            foo   bar     baz     ";
        String actual = Text.trimAndCompactSpaces(input);
        String expected = "foo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void listOfLinesToString() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String actual = Text.listOfLinesToString(lines);
        String expected = "foo" + Text.LINE_BREAK
                + "bar" + Text.LINE_BREAK
                + "baz" + Text.LINE_BREAK;
        assertEquals(expected, actual);
    }

    @Test
    public void stringToListOfLines() {
        String text = "foo" + Text.LINE_BREAK
                + "bar" + Text.LINE_BREAK
                + "baz" + Text.LINE_BREAK;
        List<String> lines = Text.stringToListOfLines(text);
        assertEquals(3, lines.size());
        assertEquals("foo", lines.get(0));
        assertEquals("bar", lines.get(1));
        assertEquals("baz", lines.get(2));
    }

    @Test
    public void stringToListOfLinesWithoutTrailingLineBreak() {
        String text = "foo" + Text.LINE_BREAK
                + "bar" + Text.LINE_BREAK
                + "baz";
        List<String> lines = Text.stringToListOfLines(text);
        assertEquals(3, lines.size());
        assertEquals("foo", lines.get(0));
        assertEquals("bar", lines.get(1));
        assertEquals("baz", lines.get(2));
    }

    @Test
    public void stringToListOfLinesEmptyText() {
        String text = "";
        List<String> lines = Text.stringToListOfLines(text);
        assertEquals(0, lines.size());
    }

    @Test
    public void stringToListOfLinesSingleLineBreak() {
        String text = Text.LINE_BREAK;
        List<String> lines = Text.stringToListOfLines(text);
        assertEquals(1, lines.size());
        assertTrue(lines.get(0).isEmpty());
    }

    @Test
    public void listOfLinesToStringAndViceVersa() {
        List<String> lines1 = new ArrayList<>();
        lines1.add("foo");
        lines1.add("bar");
        lines1.add("baz");
        String text1 = Text.listOfLinesToString(lines1);
        List<String> lines2 = Text.stringToListOfLines(text1);
        String text2 = Text.listOfLinesToString(lines2);
        assertEquals(text1, text2);
        assertEquals(lines1.size(), lines2.size());
        assertEquals(3, lines2.size());
        assertEquals(lines1.get(0), lines2.get(0));
        assertEquals(lines1.get(1), lines2.get(1));
        assertEquals(lines1.get(2), lines2.get(2));
    }

    @Test
    public void searchFirstLineIndexContainig1() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String search = "foo";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainig2() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String search = "bar";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainig3() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String search = "baz";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainigNotContainedSearch() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String search = "hugo";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = Text.NOT_FOUND_INDEX;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainigEmptyList() {
        List<String> lines = new ArrayList<>();
        String search = "hugo";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = Text.NOT_FOUND_INDEX;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainigEmptySearch() {
        List<String> lines = new ArrayList<>();
        lines.add("foo");
        lines.add("bar");
        lines.add("baz");
        String search = "";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void searchFirstLineIndexContainigEmptyListAndEmptySearch() {
        List<String> lines = new ArrayList<>();
        String search = "";
        int actual = Text.searchFirstLineIndexContainig(lines, search);
        int expected = Text.NOT_FOUND_INDEX;
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesEmptyString() {
        String text = "";
        String actual = Text.removeFromTextRespectingSpaces(text, 0, 0);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesInTheMiddleWithoutSpaces() {
        String text = "abc";
        String actual = Text.removeFromTextRespectingSpaces(text, 1, 1);
        String expected = "a c";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesInTheMiddleSpacesBeforeAndBehind() {
        String text = "a b c";
        String actual = Text.removeFromTextRespectingSpaces(text, 2, 1);
        String expected = "a c";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesInTheMiddleSpaceBefore() {
        String text = "a bc";
        String actual = Text.removeFromTextRespectingSpaces(text, 2, 1);
        String expected = "a c";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesInTheMiddleSpaceBehind() {
        String text = "ab c";
        String actual = Text.removeFromTextRespectingSpaces(text, 1, 1);
        String expected = "a c";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesFront() {
        String text = "abc";
        String actual = Text.removeFromTextRespectingSpaces(text, 0, 1);
        String expected = "bc";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRespectingSpacesRear() {
        String text = "abc";
        String actual = Text.removeFromTextRespectingSpaces(text, 2, 1);
        String expected = "ab";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextRespectingSpacesPositionTooSmall() {
        String text = "abc";
        Text.removeFromTextRespectingSpaces(text, -1, 1);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextRespectingSpacesPositionTooBig() {
        String text = "abc";
        Text.removeFromTextRespectingSpaces(text, 3, 1);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextRespectingSpacesLengthTooSmall() {
        String text = "abc";
        Text.removeFromTextRespectingSpaces(text, 0, -2);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextRespectingSpacesLengthAndPositionTooBig() {
        String text = "abc";
        Text.removeFromTextRespectingSpaces(text, 2, 2);
    }

    @Test
    public void removeFromTextEmptyString() {
        String text = "";
        String actual = Text.removeFromText(text, 0, 0);
        String expected = " ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextInTheMiddle() {
        String text = "abc";
        String actual = Text.removeFromText(text, 1, 1);
        String expected = "a c";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextFront() {
        String text = "abc";
        String actual = Text.removeFromText(text, 0, 1);
        String expected = " bc";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFromTextRear() {
        String text = "abc";
        String actual = Text.removeFromText(text, 2, 1);
        String expected = "ab ";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextPositionTooSmall() {
        String text = "abc";
        Text.removeFromText(text, -1, 1);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextPositionTooBig() {
        String text = "abc";
        Text.removeFromText(text, 3, 1);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextLengthTooSmall() {
        String text = "abc";
        Text.removeFromText(text, 0, -2);
    }

    @Test (expected = IllegalArgumentException.class)
    public void removeFromTextLengthAndPositionTooBig() {
        String text = "abc";
        Text.removeFromText(text, 2, 2);
    }

    @Test
    public void removeFirstOccurance() {
        String text = "abc foo def";
        String toRemove = "foo";
        String actual = Text.removeFirstOccurance(text, toRemove);
        String expected = "abc  def";
        assertEquals(expected, actual);
    }

    @Test
    public void removeFirstOccuranceOfEmptyString() {
        String text = "abc foo def";
        String toRemove = "";
        String actual = Text.removeFirstOccurance(text, toRemove);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeFirstOccuranceOfEmptyStringInEmptyString() {
        String text = "";
        String toRemove = "";
        String actual = Text.removeFirstOccurance(text, toRemove);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeFirstOccuranceOfSpace() {
        String text = "abc foo def";
        String toRemove = " ";
        String actual = Text.removeFirstOccurance(text, toRemove);
        String expected = "abcfoo def";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextMiddle() {
        String text = "abc";
        String insertion = "###";
        int index = 1;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "a###bc";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextMiddleZeroLength() {
        String text = "abc";
        String insertion = "";
        int index = 1;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextZeroLengthZeroLength() {
        String text = "";
        String insertion = "";
        int index = 0;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextZeroLength() {
        String text = "";
        String insertion = "abc";
        int index = 0;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextFront() {
        String text = "abc";
        String insertion = "###";
        int index = 0;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "###abc";
        assertEquals(expected, actual);
    }

    @Test
    public void insertIntoTextEnd() {
        String text = "abc";
        String insertion = "###";
        int index = 3;
        String actual = Text.insertIntoText(text, insertion, index);
        String expected = "abc###";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextMiddle() {
        String text = "abc";
        String replace = "###";
        int start = 1;
        int end = 2;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "a###c";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextMiddleZeroLength() {
        String text = "abc";
        String replace = "###";
        int start = 1;
        int end = 1;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "a###bc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextFront() {
        String text = "abc";
        String replace = "###";
        int start = 0;
        int end = 1;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "###bc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextFrontZeroLength() {
        String text = "abc";
        String replace = "###";
        int start = 0;
        int end = 0;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "###abc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextRear() {
        String text = "abc";
        String replace = "###";
        int start = 2;
        int end = 3;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "ab###";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextRearZeroLength() {
        String text = "abc";
        String replace = "###";
        int start = 2;
        int end = 2;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "ab###c";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextRearZeroLengthTotalRear() {
        String text = "abc";
        String replace = "###";
        int start = 3;
        int end = 3;
        String actual = Text.replaceInText(text, replace, start, end);
        String expected = "abc###";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextStartBiggerThanEnd() {
        String text = "abc";
        String replace = "###";
        int start = 2;
        int end = 1;
        Text.replaceInText(text, replace, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextStartTooSmall() {
        String text = "abc";
        String replace = "###";
        int start = -1;
        int end = 1;
        Text.replaceInText(text, replace, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextStartTooBig() {
        String text = "abc";
        String replace = "###";
        int start = 5;
        int end = 5;
        Text.replaceInText(text, replace, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextEndTooSmall() {
        String text = "abc";
        String replace = "###";
        int start = 2;
        int end = -1;
        Text.replaceInText(text, replace, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextEndTooBig() {
        String text = "abc";
        String replace = "###";
        int start = 2;
        int end = 4;
        Text.replaceInText(text, replace, start, end);
    }

    // TODO replaceInTextWithEqualLengthCharacterSequence(String text, char character,
    //      int start, int end)  auch Grenzen, end < start, end == start, end > length, start < 0 ...

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceMiddle() {
        String text = "abc";
        char character = '#';
        int start = 1;
        int end = 2;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "a#c";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceMiddleZeroLength() {
        String text = "abc";
        char character = '#';
        int start = 1;
        int end = 1;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceFront() {
        String text = "abc";
        char character = '#';
        int start = 0;
        int end = 1;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "#bc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceFrontZeroLength() {
        String text = "abc";
        char character = '#';
        int start = 0;
        int end = 0;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceRear() {
        String text = "abc";
        char character = '#';
        int start = 2;
        int end = 3;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "ab#";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceRearZeroLength() {
        String text = "abc";
        char character = '#';
        int start = 2;
        int end = 2;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceRearZeroLengthTotalRear() {
        String text = "abc";
        char character = '#';
        int start = 3;
        int end = 3;
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextWithEqualLengthCharacterSequenceStartBiggerThanEnd() {
        String text = "abc";
        char character = '#';
        int start = 2;
        int end = 1;
        Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextWithEqualLengthCharacterSequenceStartTooSmall() {
        String text = "abc";
        char character = '#';
        int start = -1;
        int end = 1;
        Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextWithEqualLengthCharacterSequenceStartTooBig() {
        String text = "abc";
        char character = '#';
        int start = 5;
        int end = 5;
        Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextWithEqualLengthCharacterSequenceEndTooSmall() {
        String text = "abc";
        char character = '#';
        int start = 2;
        int end = -1;
        Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
    }

    @Test (expected = IllegalArgumentException.class)
    public void replaceInTextWithEqualLengthCharacterSequenceEndTooBig() {
        String text = "abc";
        char character = '#';
        int start = 2;
        int end = 4;
        Text.replaceInTextWithEqualLengthCharacterSequence(text, character, start, end);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceOnce() {
        String text = "foo bar baz foo foo bar baz foo";
        String actual = Text.replaceInTextWithEqualLengthCharacterSequenceOnce(text, "bar");
        String expected = "foo ### baz foo foo bar baz foo";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithList1() {
        String text = "foo bar baz foo foo bar baz foo";
        List<String> replaceWords = CollectionsHelper.buildListFrom("bar");
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = "foo ### baz foo foo ### baz foo";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithList2() {
        String text = "foo bar baz foo foo bar baz foo";
        List<String> replaceWords = CollectionsHelper.buildListFrom("bar", "baz");
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = "foo ### ### foo foo ### ### foo";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithList3() {
        String text = "foo bar baz foo foo bar baz foo";
        List<String> replaceWords = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = "### ### ### ### ### ### ### ###";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithEmptyList() {
        String text = "foo bar baz foo foo bar baz foo";
        List<String> replaceWords = CollectionsHelper.buildListFrom();
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithListEmptyText() {
        String text = "";
        List<String> replaceWords = CollectionsHelper.buildListFrom("foo", "bar", "baz");
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void replaceInTextWithEqualLengthCharacterSequenceWithEmptyListAndEmptyText() {
        String text = "";
        List<String> replaceWords = CollectionsHelper.buildListFrom();
        String actual = Text.replaceInTextWithEqualLengthCharacterSequence(text, replaceWords);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void findInText01() {
        String found = Text.findInText("<id>15</id>", "<id>", "</id>");
        assertEquals("15", found);
    }

    @Test (expected = RuntimeException.class)
    public void findInText02() {
        Text.findInText("<id>15</od>", "<id>", "</id>");
    }

    @Test (expected = RuntimeException.class)
    public void findInText03() {
        Text.findInText("<id>15<id>", "<id>", "</id>");
    }

    @Test
    public void findInText04() {
        String found = Text.findInText("<foo><bar>BAR</bar><id>15</id></foo>", "<id>", "</id>");
        assertEquals("15", found);
    }

    @Test (expected = RuntimeException.class)
    public void findInTextMultipleStart() {
        Text.findInText("<id>15</id><id>", "<id>", "</id>");
    }

    @Test (expected = RuntimeException.class)
    public void findInTextMultipleEnd() {
        Text.findInText("<id>15</id></id>", "<id>", "</id>");
    }

    @Test (expected = RuntimeException.class)
    public void findInTextMissingStart() {
        Text.findInText("</id>15</id>", "<id>", "</id>");
    }

    @Test (expected = RuntimeException.class)
    public void findInTextMissingEnd() {
        Text.findInText("<id>15<id>", "<id>", "</id>");
    }

    @Test
    public void hideLineBreaks() {
        String input = "ein " + Text.LINE_BREAK + " Test";
        String actual = Text.hideLineBreaks(input);
        String expected = "ein " + Text.LINE_BREAK_PLACE_HOLDER + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void hideLineBreaksN() {
        String input = "ein \n Test";
        String actual = Text.hideLineBreaks(input);
        String expected = "ein " + Text.LINE_BREAK_PLACE_HOLDER + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void hideLineBreaksRN() {
        String input = "ein \r\n Test";
        String actual = Text.hideLineBreaks(input);
        String expected = "ein " + Text.LINE_BREAK_PLACE_HOLDER + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void hideLineBreaksWithLiteralBackslashN() {
        String input = "ein " + Text.LINE_BREAK + " Test";
        String actual = Text.hideLineBreaksWithLiteralBackslashN(input);
        String expected = "ein " + "\\n" + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void hideLineBreaksWithLiteralBackslashNN() {
        String input = "ein \n Test";
        String actual = Text.hideLineBreaksWithLiteralBackslashN(input);
        String expected = "ein " + "\\n" + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void hideLineBreaksWithLiteralBackslashNRN() {
        String input = "ein \r\n Test";
        String actual = Text.hideLineBreaksWithLiteralBackslashN(input);
        String expected = "ein " + "\\n" + " Test";
        assertEquals(expected, actual);
    }

    @Test
    public void doubleTwoDecimalPlacesWithTwoDecimalPlaces() {
        double input = 1.02d;
        String actual = Text.doubleTwoDecimalPlaces(input);
        String expected = "1,02";
        assertEquals(expected, actual);
    }

    @Test
    public void doubleTwoDecimalPlacesWithOneDecimalPlaces() {
        double input = 1.2d;
        String actual = Text.doubleTwoDecimalPlaces(input);
        String expected = "1,20";
        assertEquals(expected, actual);
    }

    @Test
    public void doubleTwoDecimalPlacesNotRounded() {
        double input = 1.234d;
        String actual = Text.doubleTwoDecimalPlaces(input);
        String expected = "1,23";
        assertEquals(expected, actual);
    }

    @Test
    public void doubleTwoDecimalPlacesRounded() {
        double input = 1.235d;
        String actual = Text.doubleTwoDecimalPlaces(input);
        String expected = "1,24";
        assertEquals(expected, actual);
    }

    @Test
    public void splitByLineBreaks() {
        String input = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        List<String> list = Text.splitByLineBreaks(input);
        assertEquals(3, list.size());
        assertEquals("foo", list.get(0));
        assertEquals("bar", list.get(1));
        assertEquals("baz", list.get(2));
    }

    @Test
    public void buildCompareText0() {
        String input = "Dies ist ein Probetext mit äöüß Umlauten ÄÖÜ.";
        String actual = Text.buildCompareText(input);
        String expected = "dies ist ein probetext mit aeoeuess umlauten aeoeue.";
        assertEquals(expected, actual);
    }

    @Test
    public void buildCompareText1() {
        String input = "Blaubären schießen über Vögel.";
        String actual = Text.buildCompareText(input);
        assertEquals("blaubaeren schiessen ueber voegel.", actual);
    }

    @Test
    public void buildCompareText2() {
        String input = "ÄäÖöÜüSSßfooäÖäÖäü.,-#|";
        String actual = Text.buildCompareText(input);
        assertEquals("aeaeoeoeueuessssfooaeoeaeoeaeue.,-#|", actual);
    }

    @Test
    public void isWordChar() {
        assertTrue(Text.isWordChar('a'));
        assertTrue(Text.isWordChar('B'));
        assertTrue(Text.isWordChar('ä'));
        assertTrue(Text.isWordChar('#'));

        assertFalse(Text.isWordChar(' '));
        assertFalse(Text.isWordChar('\n'));
        assertFalse(Text.isWordChar('\t'));

        assertFalse(Text.isWordChar('-'));
        assertFalse(Text.isWordChar(','));

        assertFalse(Text.isWordChar('('));
        assertFalse(Text.isWordChar(')'));
        assertFalse(Text.isWordChar('['));
        assertFalse(Text.isWordChar(']'));
        assertFalse(Text.isWordChar('{'));
        assertFalse(Text.isWordChar('}'));
    }

    @Test
    public void isWhitespace() {
        assertTrue(Text.isWhitespace(' '));
        assertTrue(Text.isWhitespace('\n'));
        assertTrue(Text.isWhitespace('\t'));

        assertFalse(Text.isWhitespace('a'));
        assertFalse(Text.isWhitespace('B'));

        assertFalse(Text.isWhitespace('-'));
        assertFalse(Text.isWhitespace(','));

        assertFalse(Text.isWhitespace('('));
        assertFalse(Text.isWhitespace(')'));
        assertFalse(Text.isWhitespace('['));
        assertFalse(Text.isWhitespace(']'));
        assertFalse(Text.isWhitespace('{'));
        assertFalse(Text.isWhitespace('}'));
    }

    @Test
    public void isBrace() {
        assertTrue(Text.isBrace('('));
        assertTrue(Text.isBrace(')'));
        assertTrue(Text.isBrace('['));
        assertTrue(Text.isBrace(']'));
        assertTrue(Text.isBrace('{'));
        assertTrue(Text.isBrace('}'));

        assertFalse(Text.isBrace(' '));
        assertFalse(Text.isBrace('\n'));
        assertFalse(Text.isBrace('\t'));

        assertFalse(Text.isBrace('a'));
        assertFalse(Text.isBrace('B'));

        assertFalse(Text.isBrace('-'));
        assertFalse(Text.isBrace(','));
    }

    @Test
    public void isPunctuation() {
        assertTrue(Text.isPunctuation('-'));
        assertTrue(Text.isPunctuation('_'));
        assertTrue(Text.isPunctuation(','));
        assertTrue(Text.isPunctuation('.'));
        assertTrue(Text.isPunctuation(':'));
        assertTrue(Text.isPunctuation(';'));
        assertTrue(Text.isPunctuation('!'));
        assertTrue(Text.isPunctuation('?'));

        assertFalse(Text.isPunctuation('('));
        assertFalse(Text.isPunctuation(')'));
        assertFalse(Text.isPunctuation('['));
        assertFalse(Text.isPunctuation(']'));
        assertFalse(Text.isPunctuation('{'));
        assertFalse(Text.isPunctuation('}'));

        assertFalse(Text.isPunctuation(' '));
        assertFalse(Text.isPunctuation('\n'));
        assertFalse(Text.isPunctuation('\t'));

        assertFalse(Text.isPunctuation('a'));
        assertFalse(Text.isPunctuation('B'));
    }

    @Test
    public void isCapitalLetter() {
        assertTrue(Text.isCapitalLetter('A'));
        assertTrue(Text.isCapitalLetter('B'));
        assertTrue(Text.isCapitalLetter('Ä'));
        assertTrue(Text.isCapitalLetter('Ö'));
        assertTrue(Text.isCapitalLetter('Ü'));

        assertFalse(Text.isCapitalLetter('a'));
        assertFalse(Text.isCapitalLetter('ä'));
        assertFalse(Text.isCapitalLetter('ö'));
        assertFalse(Text.isCapitalLetter('ü'));
        assertFalse(Text.isCapitalLetter('ß'));

        assertFalse(Text.isCapitalLetter('('));
        assertFalse(Text.isCapitalLetter(')'));
        assertFalse(Text.isCapitalLetter('['));
        assertFalse(Text.isCapitalLetter(']'));
        assertFalse(Text.isCapitalLetter('{'));
        assertFalse(Text.isCapitalLetter('}'));

        assertFalse(Text.isCapitalLetter(' '));
        assertFalse(Text.isCapitalLetter('\n'));
        assertFalse(Text.isCapitalLetter('\t'));

        assertFalse(Text.isCapitalLetter('-'));
        assertFalse(Text.isCapitalLetter('_'));
        assertFalse(Text.isCapitalLetter(','));
        assertFalse(Text.isCapitalLetter('.'));
        assertFalse(Text.isCapitalLetter(':'));
        assertFalse(Text.isCapitalLetter(';'));
        assertFalse(Text.isCapitalLetter('!'));
        assertFalse(Text.isCapitalLetter('?'));
    }

    @Test
    public void firstLetterIsCapitalLetter() {
        assertTrue(Text.firstLetterIsCapitalLetter("Aber"));
        assertTrue(Text.firstLetterIsCapitalLetter("Bärbel"));
        assertTrue(Text.firstLetterIsCapitalLetter("Cäsar"));
        assertTrue(Text.firstLetterIsCapitalLetter("David"));
        assertTrue(Text.firstLetterIsCapitalLetter("Edih"));
        assertTrue(Text.firstLetterIsCapitalLetter("Fahrrad"));
        assertTrue(Text.firstLetterIsCapitalLetter("Gerd"));
        assertTrue(Text.firstLetterIsCapitalLetter("Haus"));
        assertTrue(Text.firstLetterIsCapitalLetter("Igel"));
        assertTrue(Text.firstLetterIsCapitalLetter("Jörg"));
        assertTrue(Text.firstLetterIsCapitalLetter("Karl"));
        assertTrue(Text.firstLetterIsCapitalLetter("Libelle"));
        assertTrue(Text.firstLetterIsCapitalLetter("Marmelade"));
        assertTrue(Text.firstLetterIsCapitalLetter("Notfallverordnung"));
        assertTrue(Text.firstLetterIsCapitalLetter("Oregano"));
        assertTrue(Text.firstLetterIsCapitalLetter("Pausenbrot"));
        assertTrue(Text.firstLetterIsCapitalLetter("Quelle"));
        assertTrue(Text.firstLetterIsCapitalLetter("Rand"));
        assertTrue(Text.firstLetterIsCapitalLetter("Sand"));
        assertTrue(Text.firstLetterIsCapitalLetter("Tand"));
        assertTrue(Text.firstLetterIsCapitalLetter("Urmel"));
        assertTrue(Text.firstLetterIsCapitalLetter("Ventil"));
        assertTrue(Text.firstLetterIsCapitalLetter("Wolke"));
        assertTrue(Text.firstLetterIsCapitalLetter("Xantippe"));
        assertTrue(Text.firstLetterIsCapitalLetter("Ypsilon"));
        assertTrue(Text.firstLetterIsCapitalLetter("Zahl"));
        assertTrue(Text.firstLetterIsCapitalLetter("Ärmel"));
        assertTrue(Text.firstLetterIsCapitalLetter("Öl"));
        assertTrue(Text.firstLetterIsCapitalLetter("Übertrag"));

        assertFalse(Text.firstLetterIsCapitalLetter("aber"));
        assertFalse(Text.firstLetterIsCapitalLetter("ärmel"));
        assertFalse(Text.firstLetterIsCapitalLetter("öl"));
        assertFalse(Text.firstLetterIsCapitalLetter("übertrag"));

        assertFalse(Text.firstLetterIsCapitalLetter(""));
    }

    @Test
    public void multipleString() {
        String text = "foo";
        int length = 3;
        String actual = Text.multipleString(text, length);
        String expected = "foofoofoo";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringSingleCharAsString() {
        String text = "A";
        int length = 7;
        String actual = Text.multipleString(text, length);
        String expected = "AAAAAAA";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringZeroLength() {
        String text = "foo bar baz";
        int length = 0;
        String actual = Text.multipleString(text, length);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringNegativeLength() {
        String text = "foo bar baz";
        int length = -15;
        String actual = Text.multipleString(text, length);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringZeroTextLength() {
        String text = "";
        int length = 250;
        String actual = Text.multipleString(text, length);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringCharacter() {
        char character = '#';
        int length = 3;
        String actual = Text.multipleString(character, length);
        String expected = "###";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringCharacterZeroLength() {
        char character = '#';
        int length = 0;
        String actual = Text.multipleString(character, length);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void multipleStringCharacterNegativeLength() {
        char character = '#';
        int length = -44;
        String actual = Text.multipleString(character, length);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionOfEmptyTexts() {
        String text1 = "";
        String text2 = "";
        String actual = Text.findIntersection(text1, text2);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionOfOneCharacterEqualTexts() {
        String text1 = "a";
        String text2 = "a";
        String actual = Text.findIntersection(text1, text2);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionOneCharacter() {
        String text1 = "abc";
        String text2 = "cde";
        String actual = Text.findIntersection(text1, text2);
        String expected = "c";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionMetalle() {
        String text1 = "NE-Metalle";
        String text2 = "Metallen";
        String actual = Text.findIntersection(text1, text2);
        String expected = "Metalle";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionFails1() {
        String text1 = "NE-MetalleX";
        String text2 = "Metallen";
        String actual = Text.findIntersection(text1, text2);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void findIntersectionFails2() {
        String text1 = "NE-Metalle";
        String text2 = "XMetallen";
        String actual = Text.findIntersection(text1, text2);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void containsUpperLetter1() {
        String input = "foo bar baz";
        boolean actual = Text.containsUpperLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsUpperLetter2() {
        String input = "foo Bar baz";
        boolean actual = Text.containsUpperLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void containsUpperLetter3() {
        String input = "foo bar Übel baz";
        boolean actual = Text.containsUpperLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void contains() {
        String input = "Der Gegenstand wird erweitert: Bla bla";
        List<String> searches = CollectionsHelper.buildListFrom("änder", "Änder", "aender",
                "Aender", "bericht", "Bericht", "erweiter", "Erweiter");
        boolean actual = Text.contains(input, searches);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void containsFailing() {
        String input = "Gegenstand: Bla bla";
        List<String> searches = CollectionsHelper.buildListFrom("änder", "Änder", "aender",
                "Aender", "bericht", "Bericht", "erweiter", "Erweiter");
        boolean actual = Text.contains(input, searches);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWith() {
        String input = "Der Gegenstand wird erweitert: Bla bla";
        List<String> searches = CollectionsHelper.buildListFrom("foo", "bar", "baz",
                "blubber", "bl", "bla");
        boolean actual = Text.endsWith(input, searches);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithFailing() {
        String input = "Gegenstand: Bla bla";
        List<String> searches = CollectionsHelper.buildListFrom("foo", "bar", "baz",
                "blubber", "bl");
        boolean actual = Text.endsWith(input, searches);
        boolean expected = false;
        assertEquals(expected, actual);
    }


    @Test
    public void toFirstUpperCaseEmptyString() {
        String input = "";
        String actual = Text.toFirstUpperCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCaseOneUpperLetter() {
        String input = "A";
        String actual = Text.toFirstUpperCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCaseOneLowerLetter() {
        String input = "a";
        String actual = Text.toFirstUpperCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase01() {
        String input = "Hammer";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase02() {
        String input = "hammer";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase03() {
        String input = "hammer nagel Schere Licht!";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Hammer nagel schere licht!";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase04() {
        String input = "straße";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Straße";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase05() {
        String input = "ärger";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Ärger";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase06() {
        String input = "Übel";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Übel";
        assertEquals(expected, actual);
    }

    @Test
    public void toFirstUpperCase07() {
        String input = "BÖSE";
        String actual = Text.toFirstUpperCase(input);
        String expected = "Böse";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCaseEmptyString() {
        String input = "";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCaseOneUpperLetter() {
        String input = "A";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCaseOneLowerLetter() {
        String input = "a";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase01() {
        String input = "Hammer";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase02() {
        String input = "hammer";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase03() {
        String input = "hammer nagel Schere Licht!";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Hammer nagel Schere Licht!";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase04() {
        String input = "straße";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Straße";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase05() {
        String input = "ärger";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Ärger";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase06() {
        String input = "Übel";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "Übel";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToUpperCase07() {
        String input = "BÖSE";
        String actual = Text.firstCharToUpperCase(input);
        String expected = "BÖSE";
        assertEquals(expected, actual);
    }

    @Test
    public void javaToLowerCaseGerman() {
        String input = "BÖSE";
        String actual = Text.toLowerCase(input);
        String expected = "böse";
        assertEquals(expected, actual);
    }

    @Test
    public void javaToLowerCaseGermanStrasse() {
        String input = "Straße";
        String actual = Text.toLowerCase(input);
        String expected = "straße";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCaseEmptyString() {
        String input = "";
        String actual = Text.toCamelCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCaseOneUpperLetter() {
        String input = "A";
        String actual = Text.toCamelCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCaseOneLowerLetter() {
        String input = "a";
        String actual = Text.toCamelCase(input);
        String expected = "A";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCase01() {
        String input = "Hammer";
        String actual = Text.toCamelCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCase02() {
        String input = "hammer";
        String actual = Text.toCamelCase(input);
        String expected = "Hammer";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCase03() {
        String input = "hammer nagel Schere Licht!";
        String actual = Text.toCamelCase(input);
        String expected = "HammerNagelSchereLicht!";
        assertEquals(expected, actual);
    }

    @Test
    public void toCamelCase04() {
        String input = "wort1 dieseWortHatBereitsCamelCase wort2";
        String actual = Text.toCamelCase(input);
        String expected = "Wort1DieseWortHatBereitsCamelCaseWort2";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameEmtyInput() {
        String input = "";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameUntrimmedInput() {
        String input = "    foo      \t  ";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceOneSpaceToOneUnderscore() {
        String input = "foo bar";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "foo_bar";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceMultipleSpaceToOneUnderscore() {
        String input = "foo       bar";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "foo_bar";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceMultipleUnderscoreSpaceMixToOneUnderscore() {
        String input = "foo  __   ___ _ _      _bar";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "foo_bar";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceUmlaute1() {
        String input = "Ärger Övelgönne Übel";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "Aerger_Oevelgoenne_Uebel";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceUmlaute2() {
        String input = "Krähe Kübel Straße";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "Kraehe_Kuebel_Strasse";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanForSasVariableNameReplaceRubbish() {
        String input = "foo -!,;.:-/\\0123456789 bar";
        String actual = Text.cleanForSasVariableName(input);
        String expected = "foo_bar";
        assertEquals(expected, actual);
    }

    @Test
    public void nicePath01() {
        String input = "     ";
        String actual = Text.nicePath(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void nicePath02() {
        String input = "c:\\temp";
        String actual = Text.nicePath(input);
        String expected = "c:\\temp";
        assertEquals(expected, actual);
    }

    @Test
    public void nicePath03() {
        String input = "c:/temp";
        String actual = Text.nicePath(input);
        String expected = "c:\\temp";
        assertEquals(expected, actual);
    }

    @Test
    public void nicePathes01() {
        List<String> list = CollectionsHelper.buildListFrom("     ", "c:\\temp", "c:/temp");
        List<String> actualList = Text.nicePathes(list);
        List<String> expectedList = CollectionsHelper.buildListFrom("", "c:\\temp", "c:\\temp");
        assertEquals(3, actualList.size());
        for (int index = 0; index < 3; ++index) {
            String actual = actualList.get(index);
            String expected = expectedList.get(index);
            assertEquals(expected, actual);
        }
    }

    @Test
    public void skipSublinkPartWithoutHashPart() {
        String input = "hilfe.html";
        String actual = Text.skipSublinkPart(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void skipSublinkPartWithHashPart() {
        String input = "hilfe.html#why";
        String actual = Text.skipSublinkPart(input);
        String expected = "hilfe.html";
        assertEquals(expected, actual);
    }

    @Test
    public void skipSublinkPartEmptyInput() {
        String input = "";
        String actual = Text.skipSublinkPart(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void skipSublinkPartEmptyInputWithHash() {
        String input = "#why";
        String actual = Text.skipSublinkPart(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlyNormalLettersOnlyLowercaseLetters() {
        String input = "why";
        assertTrue(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersOnlyUppercaseLetters() {
        String input = "WHY";
        assertTrue(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersMixedLetters() {
        String input = "WHYareYOUsoSERIOUS";
        assertTrue(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersFailesUmlaut() {
        String input = "hähä";
        assertFalse(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersFailesSpace() {
        String input = "foo bar baz";
        assertFalse(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersFailesComma() {
        String input = "foo,bar,baz";
        assertFalse(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLetters7z() {
        String input = "7z";
        assertFalse(Text.containsOnlyNormalLetters(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsOnlyLowercaseLetters() {
        String input = "why";
        assertTrue(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsOnlyUppercaseLetters() {
        String input = "WHY";
        assertTrue(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsMixedLetters() {
        String input = "WHYareYOUsoSERIOUS";
        assertTrue(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsFailesUmlaut() {
        String input = "hähä";
        assertFalse(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsFailesSpace() {
        String input = "foo bar baz";
        assertFalse(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigitsFailesComma() {
        String input = "foo,bar,baz";
        assertFalse(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyNormalLettersOrDigits7z() {
        String input = "7z";
        assertTrue(Text.containsOnlyNormalLettersOrDigits(input));
    }

    @Test
    public void containsOnlyWordCharsOnlyLowercaseLetters() {
        String input = "äpfel";
        assertTrue(Text.containsOnlyWordChars(input));
    }

    @Test
    public void containsOnlyWordCharsMixedLetters() {
        String input = "Äpfel";
        assertTrue(Text.containsOnlyWordChars(input));
    }

    @Test
    public void containsOnlyWordCharsOnlyUppercaseLetters() {
        String input = "ÄPFEL";
        assertTrue(Text.containsOnlyWordChars(input));
    }

    @Test
    public void containsOnlyWordCharsNoSpace() {
        String input = "Äpfel und Birnen";
        assertFalse(Text.containsOnlyWordChars(input));
    }

    @Test
    public void containsOnlyWordCharsNotEmpty() {
        String input = "";
        assertFalse(Text.containsOnlyWordChars(input));
    }

    @Test
    public void tabToSpace() {
        String input = "foo\tbar\tbaz";
        String actual = Text.tabToSpace(input);
        String expected = "foo    bar    baz";
        assertEquals(expected, actual);
    }

    @Test
    public void tabToSpaceWithoutTab() {
        String input = "foo bar baz";
        String actual = Text.tabToSpace(input);
        String expected = "foo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void tabToSpaceEmptyInput() {
        String input = "";
        String actual = Text.tabToSpace(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractQuotetBraceParameters() {
        String input = "MM_openBrWindow('impressum.html','PopUp','scrollbars=yes,width=560,height=565')";

        List<String> list = Text.extractQuotetBraceParameters(input);
        assertEquals(3, list.size());
        assertEquals("impressum.html", list.get(0));
        assertEquals("PopUp", list.get(1));
        assertEquals("scrollbars=yes,width=560,height=565", list.get(2));
    }

    @Test
    public void extractQuotetBraceParametersWithSpace() {
        String input = "  MM_openBrWindow('impressum.html','PopUp','scrollbars=yes,width=560,height=565')  ";

        List<String> list = Text.extractQuotetBraceParameters(input);
        assertEquals(3, list.size());
        assertEquals("impressum.html", list.get(0));
        assertEquals("PopUp", list.get(1));
        assertEquals("scrollbars=yes,width=560,height=565", list.get(2));
    }

    @Test
    public void extractQuotetBraceParametersWithMoreSpace() {
        String input = "  MM_openBrWindow   ('impressum.html','PopUp','scrollbars=yes,width=560,height=565')  ";

        List<String> list = Text.extractQuotetBraceParameters(input);
        assertEquals(3, list.size());
        assertEquals("impressum.html", list.get(0));
        assertEquals("PopUp", list.get(1));
        assertEquals("scrollbars=yes,width=560,height=565", list.get(2));
    }

    @Test (expected = IllegalArgumentException.class)
    public void extractQuotetBraceParametersWithTwoOpeningBraces() {
        String input = "MM_openBrWindow('impressum.html,('PopUp','scrollbars=yes,width=560,height=565')";

        Text.extractQuotetBraceParameters(input);
    }

    @Test (expected = IllegalArgumentException.class)
    public void extractQuotetBraceParametersWithNoOpeningBraces() {
        String input = "MM_openBrWindow['impressum.html,'PopUp','scrollbars=yes,width=560,height=565')";

        Text.extractQuotetBraceParameters(input);
    }

    @Test (expected = IllegalArgumentException.class)
    public void extractQuotetBraceParametersWithTwoClosingBraces() {
        String input = "MM_openBrWindow('impressum.html),'PopUp','scrollbars=yes,width=560,height=565')";

        Text.extractQuotetBraceParameters(input);
    }

    @Test (expected = IllegalArgumentException.class)
    public void extractQuotetBraceParametersWithNoClosingBraces() {
        String input = "MM_openBrWindow('impressum.html,'PopUp','scrollbars=yes,width=560,height=565']";

        Text.extractQuotetBraceParameters(input);
    }

    @Test
    public void removeTextAtFrontIfStartsWith() {
        String frontToRemove = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\";
        String textToChange = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\"
                + "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontToRemove);
        String expected = "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtFrontIfStartsWithRemoveEmptyString() {
        String frontToRemove = "";
        String textToChange = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\"
                + "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtFrontIfStartsWithRemoveWrongStart() {
        String frontToRemove = "\\\\Cluster1\\Projekte1\\";
        String textToChange = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\"
                + "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtFrontIfStartsWithRemoveFromEmptyString() {
        String frontToRemove = "\\\\Cluster1\\Projekte1\\";
        String textToChange = "";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtFrontIfStartsWithRemoveEmptyFromEmptyString() {
        String frontToRemove = "";
        String textToChange = "";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtFrontIfStartsWithList() {
        List<String> frontsToRemove = CollectionsHelper.buildListFrom(",", ".", "/", "|");
        String textToChange = ", |/ foo";
        String actual = Text.removeTextAtFrontIfStartsWith(textToChange, frontsToRemove);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtEndIfEndsWith() {
        String endToRemove = "bar.baz";
        String textToChange = "bla.foo.bar.baz";
        String actual = Text.removeTextAtEndIfEndsWith(textToChange, endToRemove);
        String expected = "bla.foo.";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtEndIfEndsWithRemoveEmptyString() {
        String endToRemove = "";
        String textToChange = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\"
                + "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        String actual = Text.removeTextAtEndIfEndsWith(textToChange, endToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtEndIfEndsWithRemoveWrongStart() {
        String endToRemove = "bar.baz1789";
        String textToChange = "bla.foo.bar.baz";
        String actual = Text.removeTextAtEndIfEndsWith(textToChange, endToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtEndIfEndsWithRemoveFromEmptyString() {
        String endToRemove = "bar.baz";
        String textToChange = "";
        String actual = Text.removeTextAtEndIfEndsWith(textToChange, endToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtEndIfEndsWithRemoveEmptyFromEmptyString() {
        String endToRemove = "";
        String textToChange = "";
        String actual = Text.removeTextAtEndIfEndsWith(textToChange, endToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtStartIfStartsWith() {
        String frontToRemove = "bla.foo";
        String textToChange = "bla.foo.bar.baz";
        String actual = Text.removeTextAtStartIfStartsWith(textToChange, frontToRemove);
        String expected = ".bar.baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtStartIfStartsWithRemoveEmptyString() {
        String frontToRemove = "";
        String textToChange = "\\\\Cluster2\\Projekte2\\Schufa\\Impressum\\data\\"
                + "splitted_batches\\20170506_mtest\\20170506_mtest_part25";
        String actual = Text.removeTextAtStartIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtStartIfStartsWithRemoveWrongStart() {
        String frontToRemove = "bla.foo1789";
        String textToChange = "bla.foo.bar.baz";
        String actual = Text.removeTextAtStartIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtStartIfStartsWithRemoveFromEmptyString() {
        String frontToRemove = "bar.baz";
        String textToChange = "";
        String actual = Text.removeTextAtStartIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeTextAtStartIfStartsWithRemoveEmptyFromEmptyString() {
        String frontToRemove = "";
        String textToChange = "";
        String actual = Text.removeTextAtStartIfStartsWith(textToChange, frontToRemove);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromStringWithLineBreak() {
        String textToChange = "abc" + Text.LINE_BREAK;
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromStringWithN() {
        String textToChange = "abc" + "\n";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromStringWithMultipleLineBreaks() {
        String textToChange = "abc" + Text.LINE_BREAK
                + "def" + Text.LINE_BREAK
                + "ghi" + Text.LINE_BREAK;
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "abc" + Text.LINE_BREAK
                + "def" + Text.LINE_BREAK
                + "ghi";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromStringWithMultipleNs() {
        String textToChange = "abc" + "\n"
                + "def" + "\n"
                + "ghi" + "\n";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "abc" + "\n"
                + "def" + "\n"
                + "ghi";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromStringWithoutLineBreaks() {
        String textToChange = "abc";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromEmptyString() {
        String textToChange = "";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = textToChange;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithFromEmptyStringWithLineBreak() {
        String textToChange = "" + Text.LINE_BREAK;
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithRemoveEmptyFromStringWithN() {
        String textToChange = "" + "\n";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithRemoveEmptyFromStringWithLineBreakAndN() {
        String textToChange = "" + Text.LINE_BREAK + "\n";
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "" + Text.LINE_BREAK;
        assertEquals(expected, actual);
    }

    @Test
    public void removeLineBreakAtEndIfEndsWithRemoveEmptyFromStringWithNAndLineBreak() {
        String textToChange = ""  + "\n"+ Text.LINE_BREAK;
        String actual = Text.removeLineBreakAtEndIfEndsWith(textToChange);
        String expected = "" + "\n";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchText() {
        String text = "foo bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = "bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextSearchTextEmpty() {
        String text = "foo bar baz";
        String searchText = "";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextTextEmpty() {
        String text = "";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextBothEmpty() {
        String text = "";
        String searchText = "";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextNotContainedSearchText() {
        String text = "foo bar baz";
        String searchText = "huhu";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextMultipleContainedSearchText() {
        String text = "foo bar bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = "bar bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextSearchTextAtStart() {
        String text = "foo bar baz";
        String searchText = "foo";
        String actual = Text.removePartBeforeSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchText() {
        String text = "foo bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = " baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextSearchTextEmpty() {
        String text = "foo bar baz";
        String searchText = "";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextTextEmpty() {
        String text = "";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextBothEmpty() {
        String text = "";
        String searchText = "";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextNotContainedSearchText() {
        String text = "foo bar baz";
        String searchText = "huhu";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextMultipleContainedSearchText() {
        String text = "foo bar bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = " bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeSearchTextIncludingSearchTextSearchTextAtStart() {
        String text = "foo bar baz";
        String searchText = "foo";
        String actual = Text.removePartBeforeSearchTextIncludingSearchText(text, searchText);
        String expected = " bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchText() {
        String text = "foo bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = "bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextSearchTextEmpty() {
        String text = "foo bar baz";
        String searchText = "";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextTextEmpty() {
        String text = "";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextBothEmpty() {
        String text = "";
        String searchText = "";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextNotContainedSearchText() {
        String text = "foo bar baz";
        String searchText = "huhu";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextMultipleContainedSearchText() {
        String text = "foo bar bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = "bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextMultipleContainedSearchTextDots() {
        String text = "de.duehl.basics.text.TextTest";
        String searchText = ".";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = ".TextTest";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextSearchTextAtStart() {
        String text = "foo bar baz";
        String searchText = "foo";
        String actual = Text.removePartBeforeLastSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchText() {
        String text = "foo bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = " baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextSearchTextEmpty() {
        String text = "foo bar baz";
        String searchText = "";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextTextEmpty() {
        String text = "";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextBothEmpty() {
        String text = "";
        String searchText = "";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextNotContainedSearchText() {
        String text = "foo bar baz";
        String searchText = "huhu";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextMultipleContainedSearchText() {
        String text = "foo bar bar baz";
        String searchText = "bar";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = " baz";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextMultipleContainedSearchTextDots() {
        String text = "de.duehl.basics.text.TextTest";
        String searchText = ".";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = "TextTest";
        assertEquals(expected, actual);
    }

    @Test
    public void removePartBeforeLastSearchTextIncludingSearchTextSearchTextAtStart() {
        String text = "foo bar baz";
        String searchText = "foo";
        String actual = Text.removePartBeforeLastSearchTextIncludingSearchText(text, searchText);
        String expected = " bar baz";
        assertEquals(expected, actual);
    }


    @Test
    public void unifyLineBreaksEmptyInput() {
        String text = "";
        String actual = Text.unifyLineBreaks(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void unifyLineBreaksWithoutBreaks() {
        String text = "Dies ist ein schöner Text ohne Umbrüche!";
        String actual = Text.unifyLineBreaks(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void unifyLineBreaksWithLineBreakBreaks() {
        String text = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        String actual = Text.unifyLineBreaks(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void unifyLineBreaksWithNBreaks() {
        String text = "Dies ist ein\nText mit\nUmbrüchen!";
        String actual = Text.unifyLineBreaks(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void unifyLineBreaksWithRNBreaks() {
        String text = "Dies ist ein\r\nText mit\r\nUmbrüchen!";
        String actual = Text.unifyLineBreaks(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void unifyLineBreaksWithMultipleLineBreakBreaks() {
        String text = "Dies ist ein" + LINE_BREAK + LINE_BREAK + "Text mit" + LINE_BREAK
                + LINE_BREAK + LINE_BREAK + LINE_BREAK + LINE_BREAK + "Umbrüchen!";
        String actual = Text.unifyLineBreaks(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesEmptyInput() {
        String text = "";
        String actual = Text.removeEmptyLines(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithoutBreaks() {
        String text = "Dies ist ein schöner Text ohne Umbrüche!";
        String actual = Text.removeEmptyLines(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithLineBreakBreaks() {
        String text = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithNBreaks() {
        String text = "Dies ist ein\nText mit\nUmbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithRNBreaks() {
        String text = "Dies ist ein\r\nText mit\r\nUmbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithRNBreaks2() {
        String text = "Dies ist ein\r\nText mit\r\nUmbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = text; // "\r\n" == LINE_BREAK !
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithMultipleBreaks() {
        String text = "Dies ist ein" + LINE_BREAK + LINE_BREAK + "Text mit" + LINE_BREAK
                + LINE_BREAK + LINE_BREAK + LINE_BREAK + LINE_BREAK + "Umbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesWithMultipleBreaksWithSpaces() {
        String text = "Dies ist ein" + LINE_BREAK + "   " + LINE_BREAK + "Text mit" + LINE_BREAK
                + LINE_BREAK + "     " + LINE_BREAK + LINE_BREAK + " " + LINE_BREAK + "Umbrüchen!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein" + LINE_BREAK + "Text mit" + LINE_BREAK + "Umbrüchen!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFront() {
        String text = LINE_BREAK + "Dies ist ein Text mit Umbruch vorn!!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein Text mit Umbruch vorn!!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontWithSpace() {
        String text = "    " + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn!!";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein Text mit Umbruch vorn!!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtRear() {
        String text = "Dies ist ein Text mit Umbruch hinten!!" + LINE_BREAK;
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein Text mit Umbruch hinten!!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtRearWithSpace() {
        String text = "Dies ist ein Text mit Umbruch hinten!!" + LINE_BREAK + "    ";
        String actual = Text.removeEmptyLines(text);
        String expected = "Dies ist ein Text mit Umbruch hinten!!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndRemoveFrontAndRearWithNormalLineBreak() {
        String text = "   " + LINE_BREAK + "   \t" + LINE_BREAK + "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK;
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndRemoveFrontAndRearWithNormalLineBreakEmptyLinesInside() {
        String text = "   " + LINE_BREAK + "   \t" + LINE_BREAK + "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK + "    def    " + LINE_BREAK
                + "   " + LINE_BREAK;
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK + "    def    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndRemoveFrontAndRearWithBackslashN() {
        String text = "   " + "\n" + "   \t" + "\n" + "    abc    " + "\n"
                + "   " + "\n";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndRemoveFrontAndRearWithBackslashRN() {
        String text = "   " + "\r\n" + "   \t" + "\r\n" + "    abc    " + "\r\n"
                + "   " + "\r\n";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndRemoveFrontAndRearWithMixedLineBreakEmptyLinesInside() {
        String text = "   " + LINE_BREAK + "   \t" + "\n" + "    abc    " + "\r\n"
                + "   " + LINE_BREAK + "    def    " + "\n"
                + "   " + "\r\n";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK + "    def    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndNoRemoveWithNormalLineBreakEmptyLinesInside() {
        String text = "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK + "    def    ";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndNoRemoveWithBackslashNEmptyLinesInside() {
        String text = "    abc    " + "\n"
                + "   " + "\n" + "    def    ";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "    abc    " + LINE_BREAK
                + "   " + LINE_BREAK + "    def    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndEmptyString() {
        String text = "";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeEmptyLinesAtFrontAndEndWhitespaces() {
        String text = "     \t    \t    ";
        String actual = Text.removeEmptyLinesAtFrontAndEnd(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeShortLinesAtFront() {
        String text = "abc" + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn!!";
        String actual = Text.removeShortLines(text, 3);
        String expected = "Dies ist ein Text mit Umbruch vorn!!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeShortLinesAtMiddle() {
        String text = "Dies ist ein Text mit Umbruch vorn und hinten!"
                + LINE_BREAK + "abc"
                + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn und hinten!";
        String actual = Text.removeShortLines(text, 3);
        String expected = "Dies ist ein Text mit Umbruch vorn und hinten!"
                + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn und hinten!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeTwoShortLinesAtMiddle() {
        String text = "Dies ist ein Text mit Umbruch vorn und hinten!"
                + LINE_BREAK + "abc"
                + LINE_BREAK + "abc"
                + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn und hinten!";
        String actual = Text.removeShortLines(text, 3);
        String expected = "Dies ist ein Text mit Umbruch vorn und hinten!"
                + LINE_BREAK + "Dies ist ein Text mit Umbruch vorn und hinten!";
        assertEquals(expected, actual);
    }

    @Test
    public void removeShortLinesAtRear() {
        String text = "Dies ist ein Text mit Umbruch hinten!" + LINE_BREAK + "abc";
        String actual = Text.removeShortLines(text, 3);
        String expected = "Dies ist ein Text mit Umbruch hinten!";
        assertEquals(expected, actual);
    }

    @Test
    public void eachCharFromStringEmptyInput() {
        String text = "";
        List<String> characters = Text.eachCharFromString(text);
        assertEquals(0, characters.size());
    }

    @Test
    public void eachCharFromStringSingleCharacter() {
        String text = "A";
        List<String> characters = Text.eachCharFromString(text);
        assertEquals(1, characters.size());
        assertEquals("A", characters.get(0));
    }

    @Test
    public void eachCharFromStringAB() {
        String text = "AB";
        List<String> characters = Text.eachCharFromString(text);
        assertEquals(2, characters.size());
        assertEquals("A", characters.get(0));
        assertEquals("B", characters.get(1));
    }

    @Test
    public void eachCharFromStringABC() {
        String text = "ABC";
        List<String> characters = Text.eachCharFromString(text);
        assertEquals(3, characters.size());
        assertEquals("A", characters.get(0));
        assertEquals("B", characters.get(1));
        assertEquals("C", characters.get(2));
    }

    @Test
    public void eachCharFromStringAToZ() {
        String text = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        assertEquals(26, text.length());
        List<String> characters = Text.eachCharFromString(text);
        assertEquals(text.length(), characters.size());
        assertEquals("A", characters.get(0));
        assertEquals("B", characters.get(1));
        assertEquals("C", characters.get(2));
        assertEquals("D", characters.get(3));
        assertEquals("E", characters.get(4));
        assertEquals("F", characters.get(5));
        assertEquals("G", characters.get(6));
        assertEquals("H", characters.get(7));
        assertEquals("I", characters.get(8));
        assertEquals("J", characters.get(9));
        assertEquals("K", characters.get(10));
        assertEquals("L", characters.get(11));
        assertEquals("M", characters.get(12));
        assertEquals("N", characters.get(13));
        assertEquals("O", characters.get(14));
        assertEquals("P", characters.get(15));
        assertEquals("Q", characters.get(16));
        assertEquals("R", characters.get(17));
        assertEquals("S", characters.get(18));
        assertEquals("T", characters.get(19));
        assertEquals("U", characters.get(20));
        assertEquals("V", characters.get(21));
        assertEquals("W", characters.get(22));
        assertEquals("X", characters.get(23));
        assertEquals("Y", characters.get(24));
        assertEquals("Z", characters.get(25));
    }

    @Test
    public void startsWithIgnoreCase() {
        String text = "Jahresbilanz 2018";
        String start = "JAHRESBIlanz";
        assertTrue(Text.startsWithIgnoreCase(text, start));
    }

    @Test
    public void startsWithIgnoreCaseUmlaut() {
        String text = "ÄäÜüÖöß Haha!";
        String start = "ääÜÜööß";
        assertTrue(Text.startsWithIgnoreCase(text, start));
    }

    @Test
    public void startsWithIgnoreCaseFailing() {
        String text = "foo bar baz";
        String start = "bar baz";
        assertFalse(Text.startsWithIgnoreCase(text, start));
    }

    @Test
    public void removeSlashStarComments() {
        String text = "Ein Text /* mit einem schönen */Kommentar";
        String actual = Text.removeSlashStarComments(text );
        String expected = "Ein Text Kommentar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSlashStarCommentsAtStart() {
        String text = "/* mit einem schönen */Kommentar";
        String actual = Text.removeSlashStarComments(text );
        String expected = "Kommentar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSlashStarCommentsAtEnd() {
        String text = "Ein Text /* mit einem schönen */";
        String actual = Text.removeSlashStarComments(text );
        String expected = "Ein Text ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSlashStarCommentsMultiple() {
        String text = "/* bla /* sfawawd /**/Ein Text /* mit einem schönen */Kommentar/*     awdaw */";
        String actual = Text.removeSlashStarComments(text );
        String expected = "Ein Text Kommentar";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormEmptyString() {
        String input = "";
        String actual = Text.removeSpaces(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormStringWithOnlySpaces() {
        String input = "               ";
        String actual = Text.removeSpaces(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormStringWithoutSpaces() {
        String input = "foo";
        String actual = Text.removeSpaces(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormStringWithSpacesInFront() {
        String input = "   foo";
        String actual = Text.removeSpaces(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormStringWithSpacesBehind() {
        String input = "foo   ";
        String actual = Text.removeSpaces(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesFormStringWithSpacesInFrontAndBehind() {
        String input = "            foo   ";
        String actual = Text.removeSpaces(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpaces() {
        String input = "            foo   bar     baz     ";
        String actual = Text.removeSpaces(input);
        String expected = "foobarbaz";
        assertEquals(expected, actual);
    }

    @Test
    public void isStartOfWordBeginOfText() {
        String text = "foo bar baz";
        int index = 0;
        assertTrue(Text.isStartOfWord(index, text));
    }

    @Test
    public void isStartOfWordBeforeBeginOfText() {
        String text = "foo bar baz";
        int index = -3;
        assertFalse(Text.isStartOfWord(index, text));
    }

    @Test
    public void isStartOfWordAtEndOfText() {
        String text = "foo bar baz";
        int index = text.length();
        assertFalse(Text.isStartOfWord(index, text));
    }

    @Test
    public void isStartOfWordBeginOfWord() {
        String text = "foo bar baz";
        int index = 4;
        assertTrue(Text.isStartOfWord(index, text));
    }

    @Test
    public void isStartOfWordMiddleOfWord() {
        String text = "foo bar baz";
        int index = 5;
        assertFalse(Text.isStartOfWord(index, text));
    }

    @Test
    public void isStartOfWordBeginOfWordAfterSpecialChar() {
        String text = "foo 'bar' baz";
        int index = 5;
        assertTrue(Text.isStartOfWord(index, text));
    }

    @Test
    public void isEndOfWordEndOfText() {
        String text = "foo bar baz";
        int index = text.length();
        assertTrue(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordEndOfWord() {
        String text = "foo bar baz";
        int index = 3;
        assertTrue(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordMiddleOfWord() {
        String text = "foo bar baz";
        int index = 2;
        assertFalse(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordEndOfWordBeforeSpecialChar() {
        String text = "foo 'bar' baz";
        int index = 9;
        assertTrue(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordStartOfText() {
        String text = "foo bar baz";
        int index = 0;
        assertFalse(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordBeforeStartOfText() {
        String text = "foo bar baz";
        int index = -13;
        assertFalse(Text.isEndOfWord(index, text));
    }

    @Test
    public void isEndOfWordAfterEndOfText() {
        String text = "foo bar baz";
        int index = 66;
        assertFalse(Text.isEndOfWord(index, text));
    }

    @Test
    public void isStartAndEndOfWordOfOneWord() {
        String text = "foo bar baz";
        int startIndex = 4;
        int endIndex = 7;
        assertTrue(Text.isStartAndEndOfWord(startIndex, endIndex, text));
    }

    @Test
    public void isStartAndEndOfWordOfTwoWord() {
        String text = "foo bar baz";
        int startIndex = 4;
        int endIndex = 11;
        assertTrue(Text.isStartAndEndOfWord(startIndex, endIndex, text));
    }

    @Test
    public void isNotStartAndEndOfWordOfOneWord() {
        String text = "foo bar baz";
        int startIndex = 4;
        int endIndex = 8;
        assertFalse(Text.isStartAndEndOfWord(startIndex, endIndex, text));
    }

    @Test
    public void escapeDoubleQuotes() {
        String input = "Text \"mit\" Anführungszeichen.";
        String actual = Text.escapeDoubleQuotes(input);
        String expected = "Text \\\"mit\\\" Anführungszeichen.";
        assertEquals(expected, actual);
    }

    @Test
    public void isUpperCase() {
        String text = "ABCDEFGHIJKLMNOPQRSTUVWXYZÄÖÜÁÉÍÓÚÀÈÌÒÙÂÊÎÔÛ";
        assertTrue(Text.isUpperCase(text));
    }

    @Test
    public void isUpperCase2() {
        String text = "ABCDEF, - 'GHIJKLMNOPQRSTUVWX' / YZÄÖÜÁÉÍÓÚÀ ÈÌÒÙÂÊÎÔÛ 2134*@$&%/!\"?´`^";
        assertTrue(Text.isUpperCase(text));
    }

    @Test
    public void isNoUpperCase() {
        String text = "abcdefghijklmnopqrstuvwxyzäöüßáéíóúàèìòùâêîôû";
        assertFalse(Text.isUpperCase(text));
    }

    @Test
    public void isNoUpperCase2() {
        String text = "ò";
        assertFalse(Text.isUpperCase(text));
    }

    @Test
    public void isCamelCaseEmptyString() {
        String text = "";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseNormalText() {
        String text = "Foo";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseNormalTextWithBlanks() {
        String text = "Foo Bar";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseTwoCamels() {
        String text = "FooBar";
        assertTrue(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseThreeCamels() {
        String text = "FooBarBaz";
        assertTrue(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseFourCamels() {
        String text = "FooBarBazFour";
        assertTrue(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseFourCamelsWithOnlyOneSmallLetterBetween() {
        String text = "FoBaBaFo";
        assertTrue(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseFourCamelsWithoutOnlyOneSmallLetterBetween() {
        String text = "FooBarBazCFour";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseFullBig() {
        String text = "ABIGTEXT";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseFullSmall() {
        String text = "asmalltext";
        assertFalse(Text.isCamelCase(text));
    }

    @Test
    public void isCamelCaseWithDotsWithDotAtEnd() {
        String text = "Grund.Wert.Vermögen.";
        assertTrue(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void isCamelCaseWithDotWithoutDotAtEnd() {
        String text = "Grund.Wert.Vermögen";
        assertTrue(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void isCamelCaseWithDotWithCamelCase() {
        String text = "GrundWertVermögen";
        assertFalse(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void isCamelCaseWithDotEmpty() {
        String text = "";
        assertFalse(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void isCamelCaseWithDotFullBig() {
        String text = "ABCDEFGHI";
        assertFalse(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void isCamelCaseWithDotFullSmall() {
        String text = "abcdefhgi";
        assertFalse(Text.isCamelCaseWithDots(text));
    }

    @Test
    public void findAllPositionsOfCompleteWordsWordNotContained() {
        String searchText = "foo";
        String text = "foofoofoo";
        List<Integer> list = Text.findAllPositionsOfCompleteWords(searchText, text);
        assertEquals(0, list.size());
    }

    @Test
    public void findAllPositionsOfCompleteWords() {
        String searchText = "foo";
        String text = "foo foo foo";
        List<Integer> list = Text.findAllPositionsOfCompleteWords(searchText, text);
        assertEquals(3, list.size());
        assertEquals(Integer.valueOf(0), list.get(0));
        assertEquals(Integer.valueOf(4), list.get(1));
        assertEquals(Integer.valueOf(8), list.get(2));
    }

    @Test
    public void findAllPositionsOfCompleteWords2() {
        String searchText = "foo";
        String text = "foo fooo foo";
        List<Integer> list = Text.findAllPositionsOfCompleteWords(searchText, text);
        assertEquals(2, list.size());
        assertEquals(Integer.valueOf(0), list.get(0));
        assertEquals(Integer.valueOf(9), list.get(1));
    }

    @Test
    public void firstLoweredChar() {
        String input = "Übel";
        String actual = Text.firstLoweredChar(input);
        String expected = "ü";
        assertEquals(expected, actual);
    }

    @Test
    public void firstLoweredCharEmptyInput() {
        String input = "";
        String actual = Text.firstLoweredChar(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void firstLoweredCharAlreadyLowered() {
        String input = "äpfel";
        String actual = Text.firstLoweredChar(input);
        String expected = "ä";
        assertEquals(expected, actual);
    }

    @Test
    public void firstLoweredCharEmpty() {
        String input = "   Äpfel   ";
        String actual = Text.firstLoweredChar(input);
        String expected = " ";
        assertEquals(expected, actual);
    }

    @Test
    public void firstLoweredCharNumber() {
        String input = "1234567";
        String actual = Text.firstLoweredChar(input);
        String expected = "1";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCase() {
        String input = "Übel";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "übel";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseEmptyInput() {
        String input = "";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseAlreadyLowered() {
        String input = "äpfel";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "äpfel";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseEmpty() {
        String input = "   Äpfel   ";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "   Äpfel   ";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseNumber() {
        String input = "1234567";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "1234567";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseSingleSmallLetter() {
        String input = "a";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharToLoweredCaseSingleBigLetter() {
        String input = "A";
        String actual = Text.firstCharToLowerCase(input);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test
    public void wordToLettersEmptyInput() {
        String input = "";
        List<String> letters = Text.wordToLetters(input);
        assertEquals(0, letters.size());
    }

    @Test
    public void wordToLetters() {
        String input = "abc";
        List<String> letters = Text.wordToLetters(input);
        assertEquals(3, letters.size());
        assertEquals("a", letters.get(0));
        assertEquals("b", letters.get(1));
        assertEquals("c", letters.get(2));
    }

    @Test
    public void isQuoted() {
        assertFalse(Text.isQuoted(""));
        assertFalse(Text.isQuoted("foo"));
        assertFalse(Text.isQuoted("\"foo"));
        assertFalse(Text.isQuoted("foo\""));
        assertTrue(Text.isQuoted("\"foo\""));
        assertTrue(Text.isQuoted("\"\""));
    }

    @Test
    public void deQuoteQuotedString() {
        String input = "\"foo\"";
        String actual = Text.deQuoteQuotedString(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void deQuoteQuotedStringOnNotQuotedString() {
        Text.deQuoteQuotedString("foo");
    }

    @Test
    public void deQuoteQuotedStringIfQuotedQuotedString() {
        String input = "\"foo\"";
        String actual = Text.deQuoteQuotedStringIfQuoted(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void deQuoteQuotedStringIfQuotedNotQuotedString() {
        String input = "foo";
        String actual = Text.deQuoteQuotedStringIfQuoted(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLetters() {
        String input = "foo";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersSingleSmallLetter() {
        String input = "a";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersEmptyInput() {
        String input = "";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersSingleSpace() {
        String input = " ";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersSmallLettersWithSpace() {
        String input = "abc def";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersFirstLetterIsBig() {
        String input = "Foo";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersMiddleLetterIsBig() {
        String input = "fOo";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersLastLetterIsBig() {
        String input = "foO";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void containsOnlySmallLettersSingleBigLetter() {
        String input = "O";
        boolean actual = Text.containsOnlySmallLetters(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterLowerCase() {
        String input = "foo";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterUpperCase() {
        String input = "Foo";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterEmpty() {
        String input = "";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterBlank() {
        String input = " ";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterOneLowerCase() {
        String input = "a";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterOneUpperCase() {
        String input = "Z";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithLowercaseLetterOneNumber() {
        String input = "1";
        boolean actual = Text.startsWithLowercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterLowerCase() {
        String input = "foo";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterUpperCase() {
        String input = "Foo";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterEmpty() {
        String input = "";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterBlank() {
        String input = " ";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterOneLowerCase() {
        String input = "a";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterOneUpperCase() {
        String input = "Z";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithUppercaseLetterOneNumber() {
        String input = "1";
        boolean actual = Text.startsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterOneNumber() {
        String input = "1";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterEmpty() {
        String input = "";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterOneSmallChar() {
        String input = "a";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterOneBigChar() {
        String input = "F";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterThreeBigChar() {
        String input = "ABC";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterSmallCharsAndOneBigChar() {
        String input = "abC";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterCamelCaseEndingWithBigChar() {
        String input = "ApfelBaumG";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void endsWithUppercaseLetterCamelCaseEndingWithSmallChar() {
        String input = "ApfelBaum";
        boolean actual = Text.endsWithUppercaseLetter(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitLowerCase() {
        String input = "foo";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitUpperCase() {
        String input = "Foo";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitEmpty() {
        String input = "";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitBlank() {
        String input = " ";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitOneLowerCase() {
        String input = "a";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitOneUpperCase() {
        String input = "Z";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitOneNumber() {
        String input = "1";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithDigitLongNumber() {
        String input = "1234";
        boolean actual = Text.startsWithDigit(input);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalse() {
        String input = "Produktionsgesellschaft";
        String part = "gesellschaft";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyTrue() {
        String input = "Produktions- Gesellschaft";
        String part = "Gesellschaft";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalseAtWordStart() {
        String input = "Produktionsgesellschaft";
        String part = "Produktion";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalseIfPartNotContained() {
        String input = "Produktionsgesellschaft";
        String part = "Foo";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalseIfInputIsEmpty() {
        String input = "";
        String part = "Foo";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalseIfPartIsEmpty() {
        String input = "Produktionsgesellschaft";
        String part = "";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void positionBeforePartIsEmptyFalseIfInputAndPartIsEmpty() {
        String input = "";
        String part = "";
        boolean actual = Text.positionBeforePartIsEmpty(input, part);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void toUpperCaseEmptyInput() {
        String input = "";
        String actual = Text.toUpperCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void toUpperCaseUmlautInput() {
        String input = "ÄÖÜäöüß";
        String actual = Text.toUpperCase(input);
        String expected = "ÄÖÜÄÖÜSS";
        assertEquals(expected, actual);
    }

    @Test
    public void toLowerCaseEmptyInput() {
        String input = "";
        String actual = Text.toLowerCase(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void toLowerCaseUmlautInput() {
        String input = "ÄÖÜäöüß";
        String actual = Text.toLowerCase(input);
        String expected = "äöüäöüß";
        assertEquals(expected, actual);
    }

    @Test
    public void listToLowerCase() {
        List<String> inputs = CollectionsHelper.buildListFrom("Aber", "ÄÖÜäöüß", "ZOO");
        List<String> actual = Text.toLowerCase(inputs);
        List<String> expected = CollectionsHelper.buildListFrom("aber", "äöüäöüß", "zoo");
        assertEquals(expected, actual);
    }

    @Test
    public void startsWithEmptyList1() {
        String text = "";
        List<String> possibleStarts = CollectionsHelper.buildListFrom();
        assertFalse(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithEmptyList2() {
        String text = "Huhu";
        List<String> possibleStarts = CollectionsHelper.buildListFrom();
        assertFalse(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithBothEmptyListContainsEmptyStart() {
        String text = "";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithStartEmptyTextWithLetters() {
        String text = "abc";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithStartEmptyTextWithNoLetters() {
        String text = "123";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithLettersBehind() {
        String text = "Sumpfoase";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("Sumpf");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithHyphenBehind() {
        String text = "Sumpf-Oase";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("Sumpf");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithNumberBehind() {
        String text = "Sumpf1";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("Sumpf");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithultiple() {
        String text = "foo bar baz";
        List<String> possibleStarts = CollectionsHelper.buildListFrom("Apfel", "fo", "Baum");
        assertTrue(Text.startsWith(text, possibleStarts));
    }

    @Test
    public void startsWithCompleteWordBothEmpty() {
        String text = "";
        String possibleStartOfText = "";
        assertTrue(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void startsWithCompleteWordStartEmptyTextWithLetters() {
        String text = "abc";
        String possibleStartOfText = "";
        assertFalse(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void startsWithCompleteWordStartEmptyTextWithNoLetters() {
        String text = "123";
        String possibleStartOfText = "";
        assertTrue(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void startsWithCompleteWordLettersBehind() {
        String text = "Sumpfoase";
        String possibleStartOfText = "Sumpf";
        assertFalse(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void startsWithCompleteWordHyphenBehind() {
        String text = "Sumpf-Oase";
        String possibleStartOfText = "Sumpf";
        assertTrue(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void startsWithCompleteWordNumberBehind() {
        String text = "Sumpf1";
        String possibleStartOfText = "Sumpf";
        assertTrue(Text.startsWithCompleteWord(text, possibleStartOfText));
    }

    @Test
    public void indexAsCompleteWordBothEmpty() {
        String text = "";
        String perhapsContained = "";
        assertEquals(0, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordSearchedEmptyTextNumbers() {
        String text = "1234";
        String perhapsContained = "";
        assertEquals(0, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test @Ignore // da hängt es sich neuerdings auf, obwohl es im Debugger funktioniert...
    public void indexAsCompleteWordSearchedEmptyTextLetters() {
        String text = "abcd";
        String perhapsContained = "";
        assertEquals(-1, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWord1() {
        String text = "Sumpf";
        String perhapsContained = "Sumpf";
        assertEquals(0, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindInMiddle() {
        String text = "Sumpfland Sumpfoase Sumpfmorast";
        String perhapsContained = "Sumpfoase";
        assertEquals(10, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordDoNotStopIfFirstFoundIsNotSuccessfull() {
        String text = "Sumpfland Sumpf Sumpfmorast";
        String perhapsContained = "Sumpf";
        assertEquals(10, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordDoNotStopIfFirstFoundIsNotSuccessfull2() {
        String text = "Sumpfland Sumpfmorast Sumpf";
        String perhapsContained = "Sumpf";
        assertEquals(22, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindInFront() {
        String text = "Sumpf Baum Ast Zweig Sonnentau";
        String perhapsContained = "Sumpf";
        assertEquals(0, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindInMiddle2() {
        String text = "Baum Ast Sumpf Zweig Sonnentau";
        String perhapsContained = "Sumpf";
        assertEquals(9, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindAtEnd() {
        String text = "Baum Ast Zweig Sonnentau Sumpf";
        String perhapsContained = "Sumpf";
        assertEquals(25, Text.indexAsCompleteWord(text, perhapsContained));
    }

    @Test
    public void containsBeforeEndTextAtEnd() {
        String text = "Baumhaus";
        String perhapsContained = "haus";
        assertFalse(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void containsBeforeEndTextEmptyPart() {
        String text = "Baumhaus";
        String perhapsContained = "";
        assertTrue(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void containsBeforeEndTextEmptyTextAndEnptyPart() {
        String text = "";
        String perhapsContained = "";
        assertFalse(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindBeforeEnd() {
        String text = "Baumhausstrickleiter";
        String perhapsContained = "haus";
        assertTrue(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindAtStartNotAtAnd() {
        String text = "Baumhausstrickleiter";
        String perhapsContained = "Baumhaus";
        assertTrue(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void indexAsCompleteWordFindAtStartAndAnd() {
        String text = "Baumhaus";
        String perhapsContained = "Baumhaus";
        assertFalse(Text.containsBeforeEnd(text, perhapsContained));
    }

    @Test
    public void findIndexOfFirstDigitEmptyInput() {
        String text = "";
        assertEquals(-1, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputWithoutDigit() {
        String text = "abcdefg";
        assertEquals(-1, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInFront() {
        String text = "9abcdefg";
        assertEquals(0, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween1() {
        String text = "a8bcdefg";
        assertEquals(1, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween2() {
        String text = "ab7cdefg";
        assertEquals(2, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween3() {
        String text = "abc6defg";
        assertEquals(3, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween4() {
        String text = "abcd5efg";
        assertEquals(4, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween5() {
        String text = "abcde4fg";
        assertEquals(5, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween6() {
        String text = "abcdef3g";
        assertEquals(6, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween7() {
        String text = "abcdefg2hij";
        assertEquals(7, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween8() {
        String text = "abcdefgh1ij";
        assertEquals(8, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitInBetween9() {
        String text = "abcdefghi0j";
        assertEquals(9, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputDigitBehind() {
        String text = "abc1";
        assertEquals(3, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfFirstDigitInputMultipleDigits() {
        String text = "a 5 b 1 c 16.80731";
        assertEquals(2, Text.findIndexOfFirstDigit(text));
    }

    @Test
    public void findIndexOfLastDigitEmptyInput() {
        String text = "";
        assertEquals(-1, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputWithoutDigit() {
        String text = "abcdefg";
        assertEquals(-1, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInFront() {
        String text = "9abcdefg";
        assertEquals(0, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween1() {
        String text = "a8bcdefg";
        assertEquals(1, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween2() {
        String text = "ab7cdefg";
        assertEquals(2, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween3() {
        String text = "abc6defg";
        assertEquals(3, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween4() {
        String text = "abcd5efg";
        assertEquals(4, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween5() {
        String text = "abcde4fg";
        assertEquals(5, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween6() {
        String text = "abcdef3g";
        assertEquals(6, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween7() {
        String text = "abcdefg2hij";
        assertEquals(7, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween8() {
        String text = "abcdefgh1ij";
        assertEquals(8, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitInBetween9() {
        String text = "abcdefghi0j";
        assertEquals(9, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputDigitBehind() {
        String text = "abc1";
        assertEquals(3, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputMultipleDigits1() {
        String text = "807";
        assertEquals(2, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputMultipleDigits2() {
        String text = "80731";
        assertEquals(4, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputMultipleDigits3() {
        String text = "16.80731";
        assertEquals(7, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void findIndexOfLastDigitInputMultipleDigits() {
        String text = "a 5 b 1 c 16.80731";
        assertEquals(17, Text.findIndexOfLastDigit(text));
    }

    @Test
    public void endsWithDigitEmpty() {
        String text = "";
        assertFalse(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEnd() {
        String text = "12345a";
        assertFalse(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEndSpace() {
        String text = "12345 ";
        assertFalse(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEndDot() {
        String text = "12345.";
        assertFalse(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEndOneDigit() {
        String text = "1";
        assertTrue(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEndDigitBehindAlpha() {
        String text = "a1";
        assertTrue(Text.endsWithDigit(text));
    }

    @Test
    public void endsWithDigitNoDigitAtEndDigitBehindSpace() {
        String text = "a 1";
        assertTrue(Text.endsWithDigit(text));
    }

    @Test
    public void isDigit1() {
        String text = "1";
        assertTrue(Text.isDigit(text));
    }

    @Test
    public void isDigitEmpty() {
        String text = "";
        assertFalse(Text.isDigit(text));
    }

    @Test
    public void isDigitSpace() {
        String text = " ";
        assertFalse(Text.isDigit(text));
    }

    @Test
    public void isDigitA() {
        String text = "A";
        assertFalse(Text.isDigit(text));
    }

    @Test
    public void isDigitTwoDigits() {
        String text = "12";
        assertFalse(Text.isDigit(text));
    }

    @Test
    public void indexOfFirstDigitInContinousDigitsAtEndEmpty() {
        String text = "";
        assertEquals(-1, Text.indexOfFirstDigitInContinousDigitsAtEnd(text));
    }

    @Test
    public void indexOfFirstDigitInContinousDigitsAtEnd1() {
        String text = "1";
        assertEquals(0, Text.indexOfFirstDigitInContinousDigitsAtEnd(text));
    }

    @Test
    public void indexOfFirstDigitInContinousDigitsAtEnd123() {
        String text = "123";
        assertEquals(0, Text.indexOfFirstDigitInContinousDigitsAtEnd(text));
    }

    @Test
    public void indexOfFirstDigitInContinousDigitsAtEnd123a123() {
        String text = "123a123";
        assertEquals(4, Text.indexOfFirstDigitInContinousDigitsAtEnd(text));
    }

    @Test
    public void isMixedWordEmpty() {
        String text = "";
        assertFalse(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordNumbers() {
        String text = "12343";
        assertFalse(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordLowerSpelledWord() {
        String text = "und";
        assertTrue(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordSubstantiveWithBlankBefore() {
        String text = " Mund";
        assertFalse(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordSubstantiveWithBlankBehind() {
        String text = "Mund ";
        assertFalse(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordMultipleSubstantives() {
        String text = "Augen Mund und Nase";
        assertFalse(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordSubstantive() {
        String text = "Mund";
        assertTrue(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordBig() {
        String text = "EDV";
        assertTrue(Text.isMixedWord(text));
    }

    @Test
    public void isMixedWordMixed() {
        String text = "GmbH";
        assertTrue(Text.isMixedWord(text));
    }

    @Test
    public void isSubstantiveEmpty() {
        String text = "";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveNumbers() {
        String text = "12343";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveLowerSpelledWord() {
        String text = "und";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveSubstantiveWithBlankBefore() {
        String text = " Mund";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveSubstantiveWithBlankBehind() {
        String text = "Mund ";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveMultipleSubstantives() {
        String text = "Augen Mund und Nase";
        assertFalse(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveSubstantive() {
        String text = "Mund";
        assertTrue(Text.isSubstantive(text));
    }

    @Test
    public void isSubstantiveHyphenSubstantive() {
        String text = "Baum-Haus";
        assertTrue(Text.isSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void isSubstantiveHyphenSubstantiveSingleSubstantive() {
        String text = "Baum";
        assertFalse(Text.isSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void isSubstantiveHyphenSubstantiveThreeSubstantives() {
        String text = "Baum-Haus-Tür";
        assertFalse(Text.isSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void isSubstantiveHyphenSubstantiveWrongHyphen() {
        String text = "Baum=Haus";
        assertFalse(Text.isSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void isSubstantiveSpaceSubstantive() {
        String text = "Baum Haus";
        assertTrue(Text.isSubstantiveSpaceSubstantive(text));
    }

    @Test
    public void isSubstantiveSpaceSubstantiveSingleSubstantive() {
        String text = "Baum";
        assertFalse(Text.isSubstantiveSpaceSubstantive(text));
    }

    @Test
    public void isSubstantiveSpaceSubstantiveThreeSubstantives() {
        String text = "Baum Haus Tür";
        assertFalse(Text.isSubstantiveSpaceSubstantive(text));
    }

    @Test
    public void isSubstantiveSpaceSubstantiveWrongUnderscore() {
        String text = "Baum_Haus";
        assertFalse(Text.isSubstantiveSpaceSubstantive(text));
    }

    @Test
    public void isSubstantiveAndSubstantiveWithSubstantive() {
        String text = "Mund";
        assertEquals(Text.NO_SUBSTANTIVE_PAIR_FOUND, Text.isSubstantiveAndSubstantive(text));
    }

    @Test
    public void isSubstantiveAndSubstantiveWithSubstantiveAndSubstantive() {
        String text = "Mund und Nase";
        assertEquals(new Pair<>("Mund", "Nase"), Text.isSubstantiveAndSubstantive(text));
    }

//    @Test
//    public void isSubstantiveLowerSpelledWord() {
//        String text = "und";
//        assertFalse(Text.isSubstantive(text));
//    }
//
//    @Test
//    public void isSubstantiveSubstantiveWithBlankBefore() {
//        String text = " Mund";
//        assertFalse(Text.isSubstantive(text));
//    }
//
//    @Test
//    public void isSubstantiveSubstantiveWithBlankBehind() {
//        String text = "Mund ";
//        assertFalse(Text.isSubstantive(text));
//    }
//
//    @Test
//    public void isSubstantiveMultipleSubstantives() {
//        String text = "Augen Mund und Nase";
//        assertFalse(Text.isSubstantive(text));
//    }
//
//    @Test
//    public void isSubstantiveSubstantive() {
//        String text = "Mund";
//        assertTrue(Text.isSubstantive(text));
//    }

    @Test
    public void containesSubstantiveEmpty() {
        String text = "";
        assertFalse(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSmalLetters() {
        String text = "abc";
        assertFalse(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSmalLettersWithSpaces() {
        String text = " abc ";
        assertFalse(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSubstantiveWithoutSpaces() {
        String text = "Apfel";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSubstantiveWithSpaceBefore() {
        String text = " Apfel";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSubstantiveWithSpaceBehind() {
        String text = "Apfel ";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSubstantiveWithSpaces() {
        String text = " Apfel ";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveSubstantiveInOtherThings() {
        String text = "123 abc Apfel und Birne oder nicht?";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveNoSubstantive() {
        String text = "a 5 b 1 c 16.80731";
        assertFalse(Text.containesSubstantive(text));
    }

    @Test
    public void containesSubstantiveNanu() {
        //String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        String text = "3 Zwiebeln, Butter: 500g";
        assertTrue(Text.containesSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveEmpty() {
        String text = "";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSmalLetters() {
        String text = "abc";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSmalLettersWithSpaces() {
        String text = " abc ";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveWithoutSpaces() {
        String text = "Apfel";
        assertTrue(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveWithSpaceBefore() {
        String text = " Apfel";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveWithSpaceBehind() {
        String text = "Apfel ";
        assertTrue(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveWithSlashBehind() {
        String text = "Apfel/Birne";
        assertTrue(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveWithSpaces() {
        String text = " Apfel ";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveSubstantiveInOtherThings() {
        String text = "123 abc Apfel und Birne oder nicht?";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveNoSubstantive() {
        String text = "a 5 b 1 c 16.80731";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveNanu() {
        //String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        String text = "3 Zwiebeln, Butter: 500g";
        assertFalse(Text.startsWithSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveHyphenSubstantiveGood() {
        String text = "Moor-Blume, 3 Zwiebeln, Butter: 500g";
        assertTrue(Text.startsWithSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveHyphenSubstantiveOnlySubstantive() {
        String text = "Moorblume, 3 Zwiebeln, Butter: 500g";
        assertFalse(Text.startsWithSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void startsWithSubstantiveWithUmbruchSubstantiveWithoutSpaces() {
        String text = "Ap- fel";
        assertTrue(Text.startsWithSubstantiveWithUmbruch(text));
    }

    @Test
    public void startsWithSubstantiveWithUmbruchSubstantiveWithSpaceBefore() {
        String text = " Ap- fel";
        assertFalse(Text.startsWithSubstantiveWithUmbruch(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmpty() {
        String text = "";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySmalLetters() {
        String text = "abc";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySmalLettersWithSpaces() {
        String text = " abc ";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySubstantiveWithoutSpaces() {
        String text = "Apfel-";
        assertTrue(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySmallSubstantiveWithoutSpaces() {
        String text = "apfel-";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySubstantiveWithSpaceBefore() {
        String text = " Apfel-";
        assertTrue(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenSubstantiveWithSpaceBehind() {
        String text = "Apfel- ";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySubstantiveWithSpaces() {
        String text = " Apfel- ";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenEmptySubstantiveInOtherThings() {
        String text = "123 abc Apfel- und Birne- oder nicht?";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenNoSubstantive() {
        String text = "a 5 b 1 c 16.80731";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithSubstantiveHyphenNanu() {
        //String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        String text = "3 Zwiebeln, Butter: 500g";
        assertFalse(Text.endsWithSubstantiveHyphen(text));
    }

    @Test
    public void endsWithCommaSubstantiveCommaSubstantive() {
        String text = ", Apfel";
        assertTrue(Text.endsWithCommaSubstantive(text));
    }

    @Test
    public void endsWithCommaSubstantiveTextCommaSubstantive() {
        String text = "bla, Apfel";
        assertTrue(Text.endsWithCommaSubstantive(text));
    }

    @Test
    public void endsWithCommaSubstantiveTextCommaSubstantiveSpace() {
        String text = "bla, Apfel ";
        assertFalse(Text.endsWithCommaSubstantive(text));
    }

    @Test
    public void endsWithCommaSubstantiveTextCommaNoSpaceSubstantive() {
        String text = "bla,Apfel";
        assertFalse(Text.endsWithCommaSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveEmpty() {
        String text = "";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSmalLetters() {
        String text = "abc";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSmalLettersWithSpaces() {
        String text = " abc ";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveWithoutSpaces() {
        String text = "Apfel";
        assertEquals("Apfel", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveWithoutSpacesAndOthersBehind() {
        String text = "Äpfel und Birnen";
        assertEquals("Äpfel", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveWithSpaceBefore() {
        String text = " Apfel";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveWithSpaceBehind() {
        String text = "Apfel ";
        assertEquals("Apfel", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveWithSpaces() {
        String text = " Apfel ";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveSubstantiveInOtherThings() {
        String text = "123 abc Apfel und Birne oder nicht?";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveNoSubstantive() {
        String text = "a 5 b 1 c 16.80731";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveNanu() {
        //String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        String text = "3 Zwiebeln, Butter: 500g";
        assertEquals("", Text.determineStartingSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveHyphenSubstantiveGood() {
        String text = "Moor-Blume, 3 Zwiebeln, Butter: 500g";
        assertEquals("Moor-Blume", Text.determineStartingSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveHyphenSubstantiveOnlySubstantive() {
        String text = "Moorblume, 3 Zwiebeln, Butter: 500g";
        assertEquals("", Text.determineStartingSubstantiveHyphenSubstantive(text));
    }

    @Test
    public void determineStartingSubstantiveWithUmbruchSubstantiveWithoutSpaces() {
        String text = "Ap- fel";
        assertEquals("Ap- fel", Text.determineStartingSubstantiveWithUmbruch(text));
    }

    @Test
    public void determineEndingSubstantiveOnlySubstantive() {
        String text = "Moorblume";
        assertEquals("Moorblume", Text.determineEndingSubstantive(text));
    }

    @Test
    public void determineEndingSubstantiveTwoSubstantive() {
        String text = "Moorblume, Butterblume";
        assertEquals("Butterblume", Text.determineEndingSubstantive(text));
    }

    @Test
    public void determineEndingSubstantiveNotASubstantive() {
        String text = "MoorblumeButterblume";
        assertEquals("", Text.determineEndingSubstantive(text));
    }

    @Test
    public void findIndexOfFirstDigitAfterLastSubstantiveWantedCase() {
        String text = "Dargetzow 3. Wendung 11a";
        assertEquals(21, Text.findIndexOfFirstDigitAfterLastSubstantive(text));
    }

    @Test
    public void findIndexOfFirstDigitAfterLastSubstantiveEmpty() {
        String text = "";
        assertEquals(-1, Text.findIndexOfFirstDigitAfterLastSubstantive(text));
    }

    @Test
    public void findIndexOfFirstDigitAfterLastSubstantiveNoSubstantive() {
        String text = "a 5 b 1 c 16.80731";
        assertEquals(2, Text.findIndexOfFirstDigitAfterLastSubstantive(text));
    }

    @Test
    public void findIndexOfFirstDigitAfterLastSubstantiveWithSubstantives() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        assertEquals(29, Text.findIndexOfFirstDigitAfterLastSubstantive(text));
    }

    @Test
    public void findIndexOfFirstWordLetterEmptyText() {
        String text = "";
        assertEquals(-1, Text.findIndexOfFirstWordLetter(text));
    }

    @Test
    public void findIndexOfFirstWordLetterText() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        assertEquals(2, Text.findIndexOfFirstWordLetter(text));
    }

    @Test
    public void findIndexOfFirstWordLetterText2() {
        String text = "12345 und";
        assertEquals(6, Text.findIndexOfFirstWordLetter(text));
    }

    @Test
    public void findIndexOfFirstWordLetterText3() {
        String text = "     ß";
        assertEquals(5, Text.findIndexOfFirstWordLetter(text));
    }

    @Test
    public void findIndexOfFirstNonWordLetterEmptyText() {
        String text = "";
        assertEquals(-1, Text.findIndexOfFirstNonWordLetter(text));
    }

    @Test
    public void findIndexOfFirstNonWordLetterText() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        assertEquals(0, Text.findIndexOfFirstNonWordLetter(text));
    }

    @Test
    public void findIndexOfFirstNonWordLetterText2() {
        String text = "und,";
        assertEquals(3, Text.findIndexOfFirstNonWordLetter(text));
    }

    @Test
    public void findIndexOfFirstNonWordLetterText3() {
        String text = "äöüÄÖÜß!";
        assertEquals(7, Text.findIndexOfFirstNonWordLetter(text));
    }

    @Test
    public void findIndexOfFirstNonSpaceEmptyText() {
        String text = "";
        assertEquals(-1, Text.findIndexOfFirstNonSpace(text));
    }

    @Test
    public void findIndexOfFirstNonSpaceTextOnlySpace() {
        String text = "      ";
        assertEquals(-1, Text.findIndexOfFirstNonSpace(text));
    }

    @Test
    public void findIndexOfFirstNonSpaceTextNoSpaceAtFront() {
        String text = "Äpfel und Birnen";
        assertEquals(0, Text.findIndexOfFirstNonSpace(text));
    }

    @Test
    public void findIndexOfFirstNonSpaceTextSpacesAtFront() {
        String text = "    Äpfel und Birnen";
        assertEquals(4, Text.findIndexOfFirstNonSpace(text));
    }

    @Test
    public void findIndexOfLastNonWordLetterEmptyText() {
        String text = "";
        int startIndex = 0;
        assertEquals(-1, Text.findIndexOfLastNonWordLetter(text, startIndex));
    }

    @Test
    public void findIndexOfLastNonWordLetterTextAtStart() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        int startIndex = 0;
        assertEquals(-1, Text.findIndexOfLastNonWordLetter(text, startIndex));
    }

    @Test
    public void findIndexOfLastNonWordLetterTextBehindStart() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        int startIndex = 1;
        assertEquals(0, Text.findIndexOfLastNonWordLetter(text, startIndex));
    }

    @Test
    public void findIndexOfLastNonWordLetterTextInBetween() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        int startIndex = 12;
        assertEquals(10, Text.findIndexOfLastNonWordLetter(text, startIndex));
    }

    @Test
    public void findIndexOfLastNonWordLetterTextAtEnd() {
        String text = "5 Äpfel, 3 Zwiebeln, Butter: 500g";
        int startIndex = 32;
        assertEquals(31, Text.findIndexOfLastNonWordLetter(text, startIndex));
    }

    @Test
    public void barLine() {
        String text = Text.barLine();
        assertEquals("--------------------"
                + "--------------------"
                + "--------------------"
                + "--------------------"
                + "\n",
                text);
    }

    @Test
    public void barLineWithoutLinebreak() {
        String text = Text.barLineWithoutLinebreak();
        assertEquals("--------------------"
                + "--------------------"
                + "--------------------"
                + "--------------------"
                + "",
                text);
    }

    @Test
    public void barEqualLine() {
        String text = Text.barEqualLine();
        assertEquals("===================="
                + "===================="
                + "===================="
                + "===================="
                + "\n",
                text);
    }

    @Test
    public void indentSingleLine() {
        String text = "foo";
        String actual = Text.indent(text);
        String expected = "    foo";
        assertEquals(expected, actual);
    }

    @Test
    public void indentIndentedSingleLine() {
        String text = "  foo";
        String actual = Text.indent(text);
        String expected = "      foo";
        assertEquals(expected, actual);
    }

    @Test
    public void indentTwoLinesWithN() {
        String text = "foo\nbar";
        String actual = Text.indent(text);
        String expected = "    foo\n    bar";
        assertEquals(expected, actual);
    }

    @Test
    public void indentTwoLinesWithLineBreak() {
        String text = "foo" + LINE_BREAK + "bar";
        String actual = Text.indent(text);
        String expected = "    foo" + LINE_BREAK + "    bar";
        assertEquals(expected, actual);
    }

    @Test
    public void indentThreeLinesWithN() {
        String text = "foo\nbar\nbaz";
        String actual = Text.indent(text);
        String expected = "    foo\n    bar\n    baz";
        assertEquals(expected, actual);
    }

    @Test
    public void indentThreeLinesWithLineBreak() {
        String text = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        String actual = Text.indent(text);
        String expected = "    foo" + LINE_BREAK + "    bar" + LINE_BREAK + "    baz";
        assertEquals(expected, actual);
    }

    @Test
    public void indentThreeLinesWithLineBreakTwoTimes() {
        String text = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        String actual = Text.indent(text, 2);
        String expected = "        foo" + LINE_BREAK + "        bar" + LINE_BREAK + "        baz";
        assertEquals(expected, actual);
    }

    @Test
    public void indentAfterFirstLineThreeLinesWithLineBreak() {
        String text = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        String actual = Text.indentAfterFirstLine(text);
        String expected = "foo" + LINE_BREAK + "    bar" + LINE_BREAK + "    baz";
        assertEquals(expected, actual);
    }

    @Test
    public void indentAfterFirstLineThreeLinesWithLineBreakTwoTimes() {
        String text = "foo" + LINE_BREAK + "bar" + LINE_BREAK + "baz";
        String actual = Text.indentAfterFirstLine(text, 2);
        String expected = "foo" + LINE_BREAK + "        bar" + LINE_BREAK + "        baz";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentSingleLineWith4Spaces() {
        String text = "    foo";
        String actual = Text.unindent(text);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentSingleLineWith3Spaces() {
        String text = "   foo";
        String actual = Text.unindent(text);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentSingleLineWith2Spaces() {
        String text = "  foo";
        String actual = Text.unindent(text);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentSingleLineWith1Spaces() {
        String text = " foo";
        String actual = Text.unindent(text);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentSingleLineWith0Spaces() {
        String text = "foo";
        String actual = Text.unindent(text);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentThreeLinesWithN() {
        String text = ""
                + "    foo\n"
                + "        bar\n"
                + "    baz";
        String actual = Text.unindent(text);
        String expected = ""
                + "foo\n"
                + "    bar\n"
                + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void unindentThreeLinesWithLineBreak() {
        String text = ""
                + "    foo" + LINE_BREAK
                + "        bar" + LINE_BREAK
                + "    baz";
        String actual = Text.unindent(text);
        String expected = ""
                + "foo" + LINE_BREAK
                + "    bar" + LINE_BREAK
                + "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void insertLetterSpacing() {
        String text = "foo";
        String actual = Text.insertLetterSpacing(text);
        String expected = "f o o";
        assertEquals(expected, actual);
    }

    @Test
    public void insertLetterSpacingEmptyInput() {
        String text = "";
        String actual = Text.insertLetterSpacing(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void insertLetterSpacingMultipleWords() {
        String text = "foo bar baz";
        String actual = Text.insertLetterSpacing(text);
        String expected = "f o o   b a r   b a z";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanFromPunctuationEmptyInput() {
        String text = "";
        String actual = Text.cleanFromPunctuation(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanFromPunctuationWhitespace() {
        String text = "   \n   \r   \t    ";
        String actual = Text.cleanFromPunctuation(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void cleanFromPunctuationGegenstand() {
        String text = "(Gegenstand des xxxx, yyyy-zzzz aller Art.) .";
        String actual = Text.cleanFromPunctuation(text);
        String expected = "Gegenstand des xxxx yyyy zzzz aller Art";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexSingleCharInput() {
        String text = "a";
        int index = 0;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "a";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void getWordAtIndexEmptyInputAndWrongIndex() {
        String text = "";
        int index = 3;
        Text.getWordAtIndex(text, index);
    }

    @Test (expected = IllegalArgumentException.class)
    public void getWordAtIndexEmptyInputAndWrongIndex2() {
        String text = "";
        int index = -1;
        Text.getWordAtIndex(text, index);
    }

    @Test (expected = IllegalArgumentException.class)
    public void getWordAtIndexSingleCharInputAndWrongIndex() {
        String text = "a";
        int index = 3;
        Text.getWordAtIndex(text, index);
    }

    @Test (expected = IllegalArgumentException.class)
    public void getWordAtIndexSingleCharInputAndWrongIndex2() {
        String text = "a";
        int index = -1;
        Text.getWordAtIndex(text, index);
    }

    @Test
    public void getWordAtIndexSingleWordInput() {
        String text = "Apfelbaum";
        int index = 3;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "Apfelbaum";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexSingleWordInputIndexAtFront() {
        String text = "Apfelbaum";
        int index = 0;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "Apfelbaum";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexSingleWordInputIndexAtEnd() {
        String text = "Apfelbaum";
        int index = 8;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "Apfelbaum";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void getWordAtIndexSingleWordInputIndexBehindEnd() {
        String text = "Apfelbaum";
        int index = 9;
        Text.getWordAtIndex(text, index);
    }

    @Test
    public void getWordAtIndexThreeWordInputBehindWords() {
        String text = "foo bar baz";
        int index = 3;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexThreeWordInputFirstWord() {
        String text = "foo bar baz";
        int index = 2;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexThreeWordInputSecondWord() {
        String text = "foo bar baz";
        int index = 4;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "bar";
        assertEquals(expected, actual);
    }

    @Test
    public void getWordAtIndexThreeWordInputThirdWord() {
        String text = "foo bar baz";
        int index = 8;
        String actual = Text.getWordAtIndex(text, index);
        String expected = "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void getDistinctWordsContainingEmptySearch() {
        String text = "foo bar baz";
        String search = "";
        List<String> actual = Text.getDistinctWordsContaining(text, search);
        List<String> expected = CollectionsHelper.buildListFrom("foo", "", "bar", "baz");
        assertEquals(expected, actual);
    }

    @Test
    public void getDistinctWordsContaining() {
        String text = "foo bar baz";
        String search = "ba";
        List<String> actual = Text.getDistinctWordsContaining(text, search);
        List<String> expected = CollectionsHelper.buildListFrom("bar", "baz");
        assertEquals(expected, actual);
    }

    @Test
    public void getDistinctWordsContainingVertrag() {
        String text = "Der Gesellschaftsvertrag vom ... Übernahmevertrag ... Spaltungsvertrag ...";
        String search = "vertrag";
        List<String> actual = Text.getDistinctWordsContaining(text, search);
        List<String> expected = CollectionsHelper.buildListFrom("Gesellschaftsvertrag",
                "Übernahmevertrag", "Spaltungsvertrag");
        assertEquals(expected, actual);
    }

    @Test
    public void getDistinctWordsContainingVertrag2() {
        String text = "Beherrschungs- und Gewinnabführungsvertrag besteht nach Verschmelzung des "
                + "herrschenden Unternehmens nunmehr";
        String search = "vertrag";
        List<String> actual = Text.getDistinctWordsContaining(text, search);
        List<String> expected = CollectionsHelper.buildListFrom("Gewinnabführungsvertrag");
        assertEquals(expected, actual);
    }

    @Test
    public void lastChars() {
        String text = "foo bar baz";
        int numberOfChars = 3;
        String actual = Text.lastChars(text, numberOfChars);
        String expected = "... baz";
        assertEquals(expected, actual);
    }

    @Test
    public void lastCharsShorterText() {
        String text = "foo bar baz";
        int numberOfChars = 453;
        String actual = Text.lastChars(text, numberOfChars);
        String expected = text;
        assertEquals(expected, actual);
    }

    @Test
    public void lastCharsZeroChars() {
        String text = "foo bar baz";
        int numberOfChars = 0;
        String actual = Text.lastChars(text, numberOfChars);
        String expected = "... ";
        assertEquals(expected, actual);
    }

    @Test
    public void lastCharsOneChar() {
        String text = "foo bar baz";
        int numberOfChars = 1;
        String actual = Text.lastChars(text, numberOfChars);
        String expected = "... z";
        assertEquals(expected, actual);
    }

    @Test
    public void lastCharsAllButOneChar() {
        String text = "foo bar baz";
        int numberOfChars = text.length() - 1;
        String actual = Text.lastChars(text, numberOfChars);
        String expected = "... oo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void lastThreeChars() {
        String text = "foo bar baz";
        String actual = Text.lastThreeChars(text);
        String expected = "baz";
        assertEquals(expected, actual);
    }

    @Test
    public void lastThreeCharsLengthThree() {
        String text = "baz";
        String actual = Text.lastThreeChars(text);
        String expected = "baz";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void lastThreeCharsLengthTwo() {
        String text = "az";
        Text.lastThreeChars(text);
    }

    @Test (expected = IllegalArgumentException.class)
    public void lastThreeCharsLengthOne() {
        String text = "z";
        Text.lastThreeChars(text);
    }

    @Test (expected = IllegalArgumentException.class)
    public void lastThreeCharsLengthZero() {
        String text = "";
        Text.lastThreeChars(text);
    }

    @Test
    public void findIndexOfLastCharContainedSingleChar() {
        String text = "a";
        int actual = Text.findIndexOfLastCharContained(text, "abc");
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void findIndexOfLastCharContained() {
        String text = "und";
        int actual = Text.findIndexOfLastCharContained(text, Text.MIXED_CHARS);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void findIndexOfLastCharContainedSingleCharBetweenOthers() {
        String text = "Diamant";
        int actual = Text.findIndexOfLastCharContained(text, "abc");
        int expected = 4;
        assertEquals(expected, actual);
    }

    @Test
    public void findIndexOfLastCharContainedNoContainedChar() {
        String text = "Diamant";
        int actual = Text.findIndexOfLastCharContained(text, "bc");
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findIndexOfLastCharContainedLonge() {
        String text = "1. Diamanten und Zirkone.";
        int actual = Text.findIndexOfLastCharContained(text, Text.MIXED_CHARS);
        int expected = 23;
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordSingleSmallWord() {
        String text = "und";
        String actual = Text.extractLastWord(text);
        String expected = "und";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordSingleSubstantiv() {
        String text = "Apfel";
        String actual = Text.extractLastWord(text);
        String expected = "Apfel";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordMultipleSmallWord() {
        String text = "eins und zwei";
        String actual = Text.extractLastWord(text);
        String expected = "zwei";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordMultipleSubstantiv() {
        String text = "Birne, Zitrone und Apfel";
        String actual = Text.extractLastWord(text);
        String expected = "Apfel";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordEmptyInput() {
        String text = "";
        String actual = Text.extractLastWord(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordNoWordChar() {
        String text = "123242412";
        String actual = Text.extractLastWord(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractLastWordDotBehind() {
        String text = "Das ist ein Baum.";
        String actual = Text.extractLastWord(text);
        String expected = "Baum";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterSingleSmallWord() {
        String text = "und";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterSingleSubstantiv() {
        String text = "Apfel";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "Apfel";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterdMultipleSmallWord() {
        String text = "eins und zwei";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterMultipleSubstantiv() {
        String text = "Birne, Zitrone und Apfel";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "Apfel";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterMultipleSubstantiv2() {
        String text = "Birnen, Zitronen und Äpfel.";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "Äpfel";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterEmptyInput() {
        String text = "";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void extractFromLastUpperLetterNoWordChar() {
        String text = "123242412";
        String actual = Text.extractFromLastUpperLetter(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    // extractFromLastUpperLetter TODO mehr

    @Test
    public void lineDifferenceOnEqualLines() {
        String line1 = "123242412";
        String line2 = "123242412";
        String actual = Text.lineDifference(line1, line2);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void lineDifferenceOnEqualEmptyLines() {
        String line1 = "";
        String line2 = "";
        String actual = Text.lineDifference(line1, line2);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void lineDifferenceFirstLineIsEmmtpy() {
        String line1 = "abc";
        String line2 = "";
        String actual = Text.lineDifference(line1, line2);
        String expected = "Die erste Zeile ist länger: 3 < 0.";
        assertEquals(expected, actual);
    }

    @Test
    public void lineDifferenceSecondLineIsEmmtpy() {
        String line1 = "";
        String line2 = "abc";
        String actual = Text.lineDifference(line1, line2);
        String expected = "Die erste Zeile ist kürzer: 0 < 3.";
        assertEquals(expected, actual);
    }

    @Test
    public void lineDifferenceOnEqualLengthLines() {
        String line1 = "123242412-A-123242412";
        String line2 = "123242412-B-624141243";
        String actual = Text.lineDifference(line1, line2);
        String expected = "Am Index 10 unterscheiden sich die beiden Zeilen:\n"
                + "\t" + "Zeichen der ersten Zeile : 'A'\n"
                + "\t" + "Zeichen der zweiten Zeile: 'B'";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsEmpty() {
        String input = "";
        String actual = Text.firstChars(input, 8);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsShorter() {
        String input = "foo";
        String actual = Text.firstChars(input, 8);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsExactLength() {
        String input = "foo";
        String actual = Text.firstChars(input, 3);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsExactLonger() {
        String input = "foo";
        String actual = Text.firstChars(input, 2);
        String expected = "fo";
        assertEquals(expected, actual);
    }

    @Test
    public void firstCharsExactToZeroLength() {
        String input = "foo";
        String actual = Text.firstChars(input, 0);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void firstCharsExactToNegativeLength() {
        String input = "foo";
        Text.firstChars(input, -50);
    }

    @Test
    public void removePunctuationAtFrontAndEndNothinmgToRemove() {
        String input = "foo";
        String actual = Text.removePunctuationAtFrontAndEnd(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removePunctuationAtFrontAndEndEmptyInput() {
        String input = "";
        String actual = Text.removePunctuationAtFrontAndEnd(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void removePunctuationAtFrontAndEnd() {
        String input = ", foo.:. .";
        String actual = Text.removePunctuationAtFrontAndEnd(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginEmptyInput() {
        String input = "";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginNoSpacesAtFront() {
        String input = "foo";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 0;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginOnlyOneSpace() {
        String input = " ";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginOneSpaceBeforeText() {
        String input = " foo";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginOnlyTwoSpaces() {
        String input = "  ";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginTwoSpacesBeforeText() {
        String input = "  foo";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 2;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBegin() {
        String input = "        int count = 17;";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 8;
        assertEquals(expected, actual);
    }

    @Test
    public void countNumberOfSpacesAtBeginOneSpaceBeforeTextAndASpacesWithin() {
        String input = " f o o ";
        int actual = Text.countNumberOfSpacesAtBegin(input);
        int expected = 1;
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksEmptyInput() {
        String input = "";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksInputWithoutPipes() {
        String input = "   d  awd wa   foo ;::Wdajhwaf ";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksPipeOnly() {
        String input = "|";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = " | ";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksTwoPipeOnly() {
        String input = "||";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = " | | ";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksPipeWithOnlyBehind() {
        String input = "|a";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = " | a";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksPipeWithOnlyBefore() {
        String input = "a|";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = "a | ";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundPipesWithBlanksPipeNormalCase() {
        String input = "a|b";
        String actual = Text.surroundPipesWithBlanks(input);
        String expected = "a | b";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksEmptyInput() {
        String input = "";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksInputWithoutPipes() {
        String input = "   d  awd wa   foo ;::Wdajhwaf ";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksPipeOnly() {
        String input = "|";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "|";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksTwoPipeOnly() {
        String input = "||";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "||";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksPipeWithOnlyBehind() {
        String input = "|a";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "|a";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksPipeWithOnlyBefore() {
        String input = "a|";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "a|";
        assertEquals(expected, actual);
    }

    @Test
    public void surroundSeparatingPipesWithBlanksPipeNormalCase() {
        String input = "a|b";
        String actual = Text.surroundSeparatingPipesWithBlanks(input);
        String expected = "a | b";
        assertEquals(expected, actual);
    }

    @Test
    public void backslashToCodeFormEmptyInput() {
        String input = "";
        String actual = Text.backslashToCodeForm(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void backslashToCodeFormNoBackslashes() {
        String input = "c:/foo/bar/baz.txt";
        String actual = Text.backslashToCodeForm(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void backslashToCodeFormOneBackslashe() {
        String input = "\\";
        String actual = Text.backslashToCodeForm(input);
        String expected = "\\\\";
        assertEquals(expected, actual);
    }

    @Test
    public void backslashToCodeFormNormalCase() {
        String input = "c:\\foo\\bar\\baz.txt";
        String actual = Text.backslashToCodeForm(input);
        String expected = "c:\\\\foo\\\\bar\\\\baz.txt";
        assertEquals(expected, actual);
    }

    @Test
    public void hideQuotesForSourceCodeEmptyInput() {
        String input = "";
        String actual = Text.hideQuotesForSourceCode(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void hideQuotesForSourceCodeInputWithoutQuotes() {
        String input = "'abc' + äöü - ÄÖÜß = Foo bar Baz!";
        String actual = Text.hideQuotesForSourceCode(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void hideQuotesForSourceCodeInputWithQuotes() {
        String input = "\"abc\"";
        String actual = Text.hideQuotesForSourceCode(input);
        String expected = "\\\"abc\\\"";
        assertEquals(expected, actual);
    }

    @Test
    public void determineNumberWithAllDifferentSpacesEmptyNumber() {
        String number = "";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        assertEquals(0, numberVariations.size());
    }

    @Test
    public void determineNumberWithAllDifferentSpacesSingleLetter() {
        String number = "a";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        assertEquals(1, numberVariations.size());
        assertEquals("a", numberVariations.get(0));
    }

    @Test
    public void determineNumberWithAllDifferentSpacesTwoLetters() {
        String number = "ab";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(2, numberVariations.size());
        assertEquals("ab", numberVariations.get(0));
        assertEquals("a b", numberVariations.get(1));
    }

    @Test
    public void determineNumberWithAllDifferentSpacesThreeLetters() {
        String number = "abc";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(4, numberVariations.size());
        assertEquals("abc", numberVariations.get(0));
        assertEquals("ab c", numberVariations.get(1));
        assertEquals("a bc", numberVariations.get(2));
        assertEquals("a b c", numberVariations.get(3));
    }

    @Test
    public void determineNumberWithAllDifferentSpacesFourLetters() {
        String number = "abcd";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(8, numberVariations.size());
        assertEquals("abcd", numberVariations.get(0));
        assertEquals("abc d", numberVariations.get(1));
        assertEquals("ab cd", numberVariations.get(2));
        assertEquals("ab c d", numberVariations.get(3));
        assertEquals("a bcd", numberVariations.get(4));
        assertEquals("a bc d", numberVariations.get(5));
        assertEquals("a b cd", numberVariations.get(6));
        assertEquals("a b c d", numberVariations.get(7));
    }

    @Test
    public void determineNumberWithAllDifferentSpacesFiveLetters() {
        String number = "abcde";
        List<String> numberVariations = Text.determineNumberWithAllDifferentSpaces(number);
        //CollectionsHelper.printAssertEquals(numberVariations, "numberVariations");
        assertEquals(16, numberVariations.size());
        assertEquals("abcde", numberVariations.get( 0));
        assertEquals("abcd e", numberVariations.get( 1));
        assertEquals("abc de", numberVariations.get( 2));
        assertEquals("abc d e", numberVariations.get( 3));
        assertEquals("ab cde", numberVariations.get( 4));
        assertEquals("ab cd e", numberVariations.get( 5));
        assertEquals("ab c de", numberVariations.get( 6));
        assertEquals("ab c d e", numberVariations.get( 7));
        assertEquals("a bcde", numberVariations.get( 8));
        assertEquals("a bcd e", numberVariations.get( 9));
        assertEquals("a bc de", numberVariations.get(10));
        assertEquals("a bc d e", numberVariations.get(11));
        assertEquals("a b cde", numberVariations.get(12));
        assertEquals("a b cd e", numberVariations.get(13));
        assertEquals("a b c de", numberVariations.get(14));
        assertEquals("a b c d e", numberVariations.get(15));
    }

    @Test
    public void splitAtDigitBordersEmptyInput() {
        String number = "";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(0, numberParts.size());
    }

    @Test
    public void splitAtDigitBordersOneDigit() {
        String number = "1";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("1", numberParts.get(0));
    }

    @Test
    public void splitAtDigitBordersOneNonDigit() {
        String number = "/";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("/", numberParts.get(0));
    }

    @Test
    public void splitAtDigitBordersTwoDigits() {
        String number = "12";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("12", numberParts.get(0));
    }

    @Test
    public void splitAtDigitBordersTwoNonDigits() {
        String number = "AB";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A", numberParts.get(0));
        assertEquals("B", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersDigitAndNonDigits() {
        String number = "A1";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A", numberParts.get(0));
        assertEquals("1", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersNonDigitAndDigits() {
        String number = "1A";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("1", numberParts.get(0));
        assertEquals("A", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersDigitSpaceAndNonDigits() {
        String number = "1 A";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("1", numberParts.get(0));
        assertEquals("A", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersNonDigitSpaceAndDigits() {
        String number = "A 1";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A", numberParts.get(0));
        assertEquals("1", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersTwoDigitsSpaceAndNonDigits() {
        String number = "12 A";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("12", numberParts.get(0));
        assertEquals("A", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersNonDigitSpaceAndTwoDigits() {
        String number = "A 12";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A", numberParts.get(0));
        assertEquals("12", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersTwoDigitsAndNonDigits() {
        String number = "12A";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("12", numberParts.get(0));
        assertEquals("A", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersNonDigitAndTwoDigits() {
        String number = "A12";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A", numberParts.get(0));
        assertEquals("12", numberParts.get(1));
    }

    @Test
    public void splitAtDigitBordersExample() {
        String number = "4 542 IN 1031/15";
        List<String> numberParts = Text.splitAtDigitBorders(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(7, numberParts.size());
        assertEquals("4", numberParts.get( 0));
        assertEquals("542", numberParts.get( 1));
        assertEquals("I", numberParts.get( 2));
        assertEquals("N", numberParts.get( 3));
        assertEquals("1031", numberParts.get( 4));
        assertEquals("/", numberParts.get( 5));
        assertEquals("15", numberParts.get( 6));
    }

    @Test
    public void determineNumbersWithDifferentSpacesEmptyNumber() {
        String number = "";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        assertEquals(0, numberVariations.size());
    }

    @Test
    public void determineNumbersWithDifferentSpacesSingleLetter() {
        String number = "a";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        assertEquals(1, numberVariations.size());
        assertEquals("a", numberVariations.get(0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesTwoLetters() {
        String number = "ab";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(2, numberVariations.size());
        assertEquals("ab", numberVariations.get(0));
        assertEquals("a b", numberVariations.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesThreeLetters() {
        String number = "abc";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(4, numberVariations.size());
        assertEquals("abc", numberVariations.get(0));
        assertEquals("ab c", numberVariations.get(1));
        assertEquals("a bc", numberVariations.get(2));
        assertEquals("a b c", numberVariations.get(3));
    }

    @Test
    public void determineNumbersWithDifferentSpacesFourLetters() {
        String number = "abcd";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printListNice(numberVariations);
        assertEquals(8, numberVariations.size());
        assertEquals("abcd", numberVariations.get(0));
        assertEquals("abc d", numberVariations.get(1));
        assertEquals("ab cd", numberVariations.get(2));
        assertEquals("ab c d", numberVariations.get(3));
        assertEquals("a bcd", numberVariations.get(4));
        assertEquals("a bc d", numberVariations.get(5));
        assertEquals("a b cd", numberVariations.get(6));
        assertEquals("a b c d", numberVariations.get(7));
    }

    @Test
    public void determineNumbersWithDifferentSpacesFiveLetters() {
        String number = "abcde";
        List<String> numberVariations = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printAssertEquals(numberVariations, "numberVariations");
        assertEquals(16, numberVariations.size());
        assertEquals("abcde", numberVariations.get( 0));
        assertEquals("abcd e", numberVariations.get( 1));
        assertEquals("abc de", numberVariations.get( 2));
        assertEquals("abc d e", numberVariations.get( 3));
        assertEquals("ab cde", numberVariations.get( 4));
        assertEquals("ab cd e", numberVariations.get( 5));
        assertEquals("ab c de", numberVariations.get( 6));
        assertEquals("ab c d e", numberVariations.get( 7));
        assertEquals("a bcde", numberVariations.get( 8));
        assertEquals("a bcd e", numberVariations.get( 9));
        assertEquals("a bc de", numberVariations.get(10));
        assertEquals("a bc d e", numberVariations.get(11));
        assertEquals("a b cde", numberVariations.get(12));
        assertEquals("a b cd e", numberVariations.get(13));
        assertEquals("a b c de", numberVariations.get(14));
        assertEquals("a b c d e", numberVariations.get(15));
    }

    @Test
    public void determineNumbersWithDifferentSpacesTooMuchPartsForAllButNotForBorder() {
        String number = "1234 76251 45141";
        List<String> numberParts = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(4, numberParts.size());
        assertEquals("12347625145141", numberParts.get( 0));
        assertEquals("123476251 45141", numberParts.get( 1));
        assertEquals("1234 7625145141", numberParts.get( 2));
        assertEquals("1234 76251 45141", numberParts.get( 3));
    }

    @Test
    public void determineNumbersWithDifferentSpaces12Parts() {
        String number = "1a2b3d4e5f6";
        List<String> numberParts = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(1, numberParts.size());
        assertEquals("1a2b3d4e5f6", numberParts.get( 0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesMuchTooMuchParts() {
        String number = "1a2b3d4e5f6g7h8i9j";
        List<String> numberParts = Text.determineNumbersWithDifferentSpaces(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(1, numberParts.size());
        assertEquals("1a2b3d4e5f6g7h8i9j", numberParts.get( 0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersEmptyInput() {
        String number = "";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(0, numberParts.size());
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersOneDigit() {
        String number = "1";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("1", numberParts.get(0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersOneNonDigit() {
        String number = "/";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("/", numberParts.get(0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersTwoDigits() {
        String number = "12";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(1, numberParts.size());
        assertEquals("12", numberParts.get(0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersTwoNonDigits() {
        String number = "AB";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("AB", numberParts.get(0));
        assertEquals("A B", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersDigitAndNonDigits() {
        String number = "A1";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A1", numberParts.get(0));
        assertEquals("A 1", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersNonDigitAndDigits() {
        String number = "1A";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("1A", numberParts.get(0));
        assertEquals("1 A", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersDigitSpaceAndNonDigits() {
        String number = "1 A";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("1A", numberParts.get(0));
        assertEquals("1 A", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersNonDigitSpaceAndDigits() {
        String number = "A 1";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A1", numberParts.get(0));
        assertEquals("A 1", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersTwoDigitsSpaceAndNonDigits() {
        String number = "12 A";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("12A", numberParts.get(0));
        assertEquals("12 A", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersNonDigitSpaceAndTwoDigits() {
        String number = "A 12";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A12", numberParts.get(0));
        assertEquals("A 12", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersTwoDigitsAndNonDigits() {
        String number = "12A";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("12A", numberParts.get(0));
        assertEquals("12 A", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersNonDigitAndTwoDigits() {
        String number = "A12";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        assertEquals(2, numberParts.size());
        assertEquals("A12", numberParts.get(0));
        assertEquals("A 12", numberParts.get(1));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersExample() {
        String number = "4 542 IN 1031/15";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(64, numberParts.size());
        assertEquals("4542IN1031/15", numberParts.get( 0));
        assertEquals("4542IN1031/ 15", numberParts.get( 1));
        assertEquals("4542IN1031 /15", numberParts.get( 2));
        assertEquals("4542IN1031 / 15", numberParts.get( 3));
        assertEquals("4542IN 1031/15", numberParts.get( 4));
        assertEquals("4542IN 1031/ 15", numberParts.get( 5));
        assertEquals("4542IN 1031 /15", numberParts.get( 6));
        assertEquals("4542IN 1031 / 15", numberParts.get( 7));
        assertEquals("4542I N1031/15", numberParts.get( 8));
        assertEquals("4542I N1031/ 15", numberParts.get( 9));
        assertEquals("4542I N1031 /15", numberParts.get(10));
        assertEquals("4542I N1031 / 15", numberParts.get(11));
        assertEquals("4542I N 1031/15", numberParts.get(12));
        assertEquals("4542I N 1031/ 15", numberParts.get(13));
        assertEquals("4542I N 1031 /15", numberParts.get(14));
        assertEquals("4542I N 1031 / 15", numberParts.get(15));
        assertEquals("4542 IN1031/15", numberParts.get(16));
        assertEquals("4542 IN1031/ 15", numberParts.get(17));
        assertEquals("4542 IN1031 /15", numberParts.get(18));
        assertEquals("4542 IN1031 / 15", numberParts.get(19));
        assertEquals("4542 IN 1031/15", numberParts.get(20));
        assertEquals("4542 IN 1031/ 15", numberParts.get(21));
        assertEquals("4542 IN 1031 /15", numberParts.get(22));
        assertEquals("4542 IN 1031 / 15", numberParts.get(23));
        assertEquals("4542 I N1031/15", numberParts.get(24));
        assertEquals("4542 I N1031/ 15", numberParts.get(25));
        assertEquals("4542 I N1031 /15", numberParts.get(26));
        assertEquals("4542 I N1031 / 15", numberParts.get(27));
        assertEquals("4542 I N 1031/15", numberParts.get(28));
        assertEquals("4542 I N 1031/ 15", numberParts.get(29));
        assertEquals("4542 I N 1031 /15", numberParts.get(30));
        assertEquals("4542 I N 1031 / 15", numberParts.get(31));
        assertEquals("4 542IN1031/15", numberParts.get(32));
        assertEquals("4 542IN1031/ 15", numberParts.get(33));
        assertEquals("4 542IN1031 /15", numberParts.get(34));
        assertEquals("4 542IN1031 / 15", numberParts.get(35));
        assertEquals("4 542IN 1031/15", numberParts.get(36));
        assertEquals("4 542IN 1031/ 15", numberParts.get(37));
        assertEquals("4 542IN 1031 /15", numberParts.get(38));
        assertEquals("4 542IN 1031 / 15", numberParts.get(39));
        assertEquals("4 542I N1031/15", numberParts.get(40));
        assertEquals("4 542I N1031/ 15", numberParts.get(41));
        assertEquals("4 542I N1031 /15", numberParts.get(42));
        assertEquals("4 542I N1031 / 15", numberParts.get(43));
        assertEquals("4 542I N 1031/15", numberParts.get(44));
        assertEquals("4 542I N 1031/ 15", numberParts.get(45));
        assertEquals("4 542I N 1031 /15", numberParts.get(46));
        assertEquals("4 542I N 1031 / 15", numberParts.get(47));
        assertEquals("4 542 IN1031/15", numberParts.get(48));
        assertEquals("4 542 IN1031/ 15", numberParts.get(49));
        assertEquals("4 542 IN1031 /15", numberParts.get(50));
        assertEquals("4 542 IN1031 / 15", numberParts.get(51));
        assertEquals("4 542 IN 1031/15", numberParts.get(52));
        assertEquals("4 542 IN 1031/ 15", numberParts.get(53));
        assertEquals("4 542 IN 1031 /15", numberParts.get(54));
        assertEquals("4 542 IN 1031 / 15", numberParts.get(55));
        assertEquals("4 542 I N1031/15", numberParts.get(56));
        assertEquals("4 542 I N1031/ 15", numberParts.get(57));
        assertEquals("4 542 I N1031 /15", numberParts.get(58));
        assertEquals("4 542 I N1031 / 15", numberParts.get(59));
        assertEquals("4 542 I N 1031/15", numberParts.get(60));
        assertEquals("4 542 I N 1031/ 15", numberParts.get(61));
        assertEquals("4 542 I N 1031 /15", numberParts.get(62));
        assertEquals("4 542 I N 1031 / 15", numberParts.get(63));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBorders10Parts() {
        String number = "1a2b3d4e5f6";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(1, numberParts.size());
        assertEquals("1a2b3d4e5f6", numberParts.get( 0));
    }

    @Test
    public void determineNumbersWithDifferentSpacesBetweenDigitBordersMuchTooMuchParts() {
        String number = "1a2b3d4e5f6g7h8i9j";
        List<String> numberParts =
                Text.determineNumbersWithDifferentSpacesBetweenDigitBorders(number);
        //CollectionsHelper.printAssertEquals(numberParts, "numberParts");
        assertEquals(1, numberParts.size());
        assertEquals("1a2b3d4e5f6g7h8i9j", numberParts.get( 0));
    }

    @Test
    public void searchNumberWithDifferentSpacesInText() {
        String text = "Aktenzeichen: 542 IN 1031/15 - Insolvenzverfahren über das Vermögen der "
                + "Nadya Nedeva, geb. 31.05.1985, ul. Zhaltusha 23, 6600 KARDJALI, BULGARIEN "
                + "Inhaber der Firma Nadya Nedeva, \"\"Reinigung nach Hausfrauenart\"\", "
                + "Weinböhlaer Straße 9, 01127 Dresden";
        String number = "542 IN 1031/15";
        List<String> numbersInText = Text.searchNumberWithDifferentSpacesInText(number, text);
        assertEquals(1, numbersInText.size());
        assertEquals("542 IN 1031/15", numbersInText.get( 0));
    }

    @Test
    public void searchNumberWithDifferentSpacesInTextDifferentNumber() {
        String text = "Aktenzeichen: 542 IN 1031/15 - Insolvenzverfahren über das Vermögen der "
                + "Nadya Nedeva, geb. 31.05.1985, ul. Zhaltusha 23, 6600 KARDJALI, BULGARIEN "
                + "Inhaber der Firma Nadya Nedeva, \"\"Reinigung nach Hausfrauenart\"\", "
                + "Weinböhlaer Straße 9, 01127 Dresden";
        String number = "542IN1031/15";
        List<String> numbersInText = Text.searchNumberWithDifferentSpacesInText(number, text);
        assertEquals(1, numbersInText.size());
        assertEquals("542 IN 1031/15", numbersInText.get( 0));
    }

    @Test
    public void searchNumberWithDifferentSpacesInTextManyMatches() {
        String text = "foo 1 23 45 bar baz 12345 foo 1 2 3 45";
        String number = "12345";
        List<String> numbersInText = Text.searchNumberWithDifferentSpacesInText(number, text);
        assertEquals(3, numbersInText.size());
        assertEquals("12345", numbersInText.get(0));
        assertEquals("1 23 45", numbersInText.get(1));
        assertEquals("1 2 3 45", numbersInText.get(2));
    }

    @Test
    public void findLast() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("er", "so", "Mal");
        FoundSearch found = Text.findLast(text, searchPhrases);
        assertTrue(found.wasSuccessfull());
        assertEquals("so", found.getSearch());
        assertEquals(34, found.getIndex());
    }

    @Test
    public void findLastNotFound() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("sie", "somit", "Malbuch");
        FoundSearch found = Text.findLast(text, searchPhrases);
        assertFalse(found.wasSuccessfull());
    }

    @Test
    public void findLastNotStartingWith() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("er", "so", "Mal");
        FoundSearch found = Text.findLastNotStartingWith(text, searchPhrases, " findet");
        assertTrue(found.wasSuccessfull());
        assertEquals("er", found.getSearch());
        assertEquals(16, found.getIndex());
    }

    @Test
    public void findLastNotStartingWithSameWordInText() {
        String text = "Mal schauen was er so in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("er", "so", "Mal");
        FoundSearch found = Text.findLastNotStartingWith(text, searchPhrases, " findet");
        assertTrue(found.wasSuccessfull());
        assertEquals("so", found.getSearch());
        assertEquals(19, found.getIndex());
    }

    @Test
    public void findLastNotStartingWithList1() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("er", "so", "Mal");
        List<String> textAfterSearchNotStartingWithList = CollectionsHelper.buildListFrom(" findet");
        FoundSearch found = Text.findLastNotStartingWith(text, searchPhrases,
                textAfterSearchNotStartingWithList);
        assertTrue(found.wasSuccessfull());
        assertEquals("er", found.getSearch());
        assertEquals(16, found.getIndex());
    }

    @Test
    public void findLastNotStartingWithList2() {
        String text = "Der Apfel hängt am Apfelbaum.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("Apfel", "Birne");
        List<String> textAfterSearchNotStartingWithList = CollectionsHelper.buildListFrom("baum");
        FoundSearch found = Text.findLastNotStartingWith(text, searchPhrases,
                textAfterSearchNotStartingWithList);
        assertTrue(found.wasSuccessfull());
        assertEquals("Apfel", found.getSearch());
        assertEquals(4, found.getIndex());
    }

    @Test
    public void findLastNotStartingWithList() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom("er", "so", "Mal");
        FoundSearch found = Text.findLastNotStartingWith(text, searchPhrases,
                CollectionsHelper.buildListFrom(" findet", " in"));
        assertTrue(found.wasSuccessfull());
        assertEquals("Mal", found.getSearch());
        assertEquals(0, found.getIndex());
    }

    @Test
    public void findLongest() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "Mal", "in", "Baum");
        FoundSearch found = Text.findLongest(text, searchPhrases);
        assertTrue(found.wasSuccessfull());
        assertEquals("Mal", found.getSearch());
        assertEquals(0, found.getIndex());
    }

    @Test
    public void findLongestNotFound() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> searchPhrases = CollectionsHelper.buildListFrom(
                "sie", "somit", "Malbuch", "innnen", "Baum");
        FoundSearch found = Text.findLongest(text, searchPhrases);
        assertFalse(found.wasSuccessfull());
    }


    @Test
    public void findLongestAtStart() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "M", "Mal", "in", "Baum", "Ma");
        String longestStartPhrase = Text.findLongestAtStart(text, startPhrases);
        assertEquals("Mal", longestStartPhrase);
    }

    @Test
    public void findLongestAtStartNotStartingWith() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "in", "Baum");
        String longestStartPhrase = Text.findLongestAtStart(text, startPhrases);
        assertEquals("", longestStartPhrase);
    }

    @Test
    public void findFirstPosition() {
        List<String> starts = CollectionsHelper.buildListFrom("a", "A");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = 8;
        assertEquals(expected, actual);
        assertEquals("A", found.getSearch());
    }

    @Test
    public void findFirstPositionAtFirstChar() {
        List<String> starts = CollectionsHelper.buildListFrom("b", "B");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = 0;
        assertEquals(expected, actual);
        assertEquals("B", found.getSearch());
    }

    @Test
    public void findFirstPositionNotFound() {
        List<String> starts = CollectionsHelper.buildListFrom("z", "Z");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionNotFoundWithEmptyList() {
        List<String> starts = CollectionsHelper.buildListFrom();
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionNotFoundInEmptyString() {
        List<String> starts = CollectionsHelper.buildListFrom("z", "Z");
        String text = "";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionNotFoundInEmptyStringAndEmptyString() {
        List<String> starts = CollectionsHelper.buildListFrom();
        String text = "";
        FoundSearch found = Text.findFirstPosition(text, starts);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionNotStartingWith() {
        List<String> starts = CollectionsHelper.buildListFrom("a", "A");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        FoundSearch found = Text.findFirstPositionNotStartingWith(text, starts, "shd");
        int actual = found.getIndex();
        int expected = 15;
        assertEquals(expected, actual);
        assertEquals("a", found.getSearch());
    }

    @Test
    public void findFirstPositionIndex() {
        List<String> starts = CollectionsHelper.buildListFrom("a", "A");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        int fromIndex = 9;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = 15;
        assertEquals(expected, actual);
        assertEquals("a", found.getSearch());
    }

    @Test
    public void findFirstPositionIndexNotFound() {
        List<String> starts = CollectionsHelper.buildListFrom("z", "Z");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        int fromIndex = 9;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionIndexNotFoundBecauseOfHighIndex() {
        List<String> starts = CollectionsHelper.buildListFrom("a", "A");
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        int fromIndex = 100;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionIndexNotFoundWithEmptyList() {
        List<String> starts = CollectionsHelper.buildListFrom();
        String text = "BbbdghshAshdhsjajjsdjjajjsdkadjAsdjjaa";
        int fromIndex = 5;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionIndexNotFoundInEmptyString() {
        List<String> starts = CollectionsHelper.buildListFrom("z", "Z");
        String text = "";
        int fromIndex = 5;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPositionIndexNotFoundInEmptyStringAndEmptyString() {
        List<String> starts = CollectionsHelper.buildListFrom();
        String text = "";
        int fromIndex = 0;
        FoundSearch found = Text.findFirstPosition(text, starts, fromIndex);
        int actual = found.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void findFirstPosition1a() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "in", "Baum");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(16, foundSearch.getIndex());
    }

    @Test
    public void findFirstPositionAndFirstIndex() {
        String text = "Mal schauen was er in diesem Text so eher findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "in", "Baum");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(16, foundSearch.getIndex());
    }

    @Test
    public void findFirstPosition2() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz", "er", "so", "in", "Baum");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(16, foundSearch.getIndex());
    }

    @Test
    public void findFirstPosition3a() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz", "er", "so", "in", "Baum", "was");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("was", foundSearch.getSearch());
        assertEquals(12, foundSearch.getIndex());
    }

    @Test
    public void findFirstPosition3() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "Text", "baz", "er", "so", "in");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(16, foundSearch.getIndex());
    }

    @Test
    public void findFirstPosition3b() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "foo", "bar", "in", "Text", "baz", "so");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(16, foundSearch.getIndex());
    }

    @Test
    public void findFirstPositionEmptyText() {
        String text = "";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "Text", "baz", "er", "so", "in");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findFirstPositionEmptyTextEmptyList() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom();
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findFirstPositionOnlyNotContainedPhrases() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz");
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findFirstPositionEmptyList() {
        String text = "";
        List<String> startPhrases = CollectionsHelper.buildListFrom();
        FoundSearch foundSearch = Text.findFirstPosition(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findLastListElementInText() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "in", "Baum");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("in", foundSearch.getSearch());
        assertEquals(38, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInTextAndLastIndex() {
        String text = "Mal schauen was er in diesem Text so eher findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "Baum");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("er", foundSearch.getSearch());
        assertEquals(39, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInText2() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz", "er", "so", "Baum");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("so", foundSearch.getSearch());
        assertEquals(34, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInText3a() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz", "er", "so", "in", "Baum", "was");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("in", foundSearch.getSearch());
        assertEquals(38, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInText3() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "Text", "baz", "er", "so", "in");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("in", foundSearch.getSearch());
        assertEquals(38, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInText3b() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "foo", "bar", "in", "Text", "baz", "so");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals("in", foundSearch.getSearch());
        assertEquals(38, foundSearch.getIndex());
    }

    @Test
    public void findLastListElementInTextEmptyText() {
        String text = "";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "Text", "baz", "er", "so", "in");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findLastListElementInTextEmptyTextEmptyList() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom();
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findLastListElementInTextOnlyNotContainedPhrases() {
        String text = "Mal schauen was er in diesem Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "foo", "bar", "baz");
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findLastListElementInTextEmptyList() {
        String text = "";
        List<String> startPhrases = CollectionsHelper.buildListFrom();
        FoundSearch foundSearch = Text.findLastListElementInText(text, startPhrases);
        assertEquals(FoundSearch.NOT_FOUND, foundSearch);
    }

    @Test
    public void findAllSearchPhrasesInText() {
        String text = "Mal schauen was er in diesem sonderbaren und erbaulichen Text so findet.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "er", "so", "in", "Baum");
        List<FoundSearch> foundSearches = Text.findAllSearchPhrasesInText(text, startPhrases);
        assertEquals(7, foundSearches.size());

        FoundSearch foundSearch1 = foundSearches.get(0);
        FoundSearch foundSearch2 = foundSearches.get(1);
        FoundSearch foundSearch3 = foundSearches.get(2);
        FoundSearch foundSearch4 = foundSearches.get(3);
        FoundSearch foundSearch5 = foundSearches.get(4);
        FoundSearch foundSearch6 = foundSearches.get(5);
        FoundSearch foundSearch7 = foundSearches.get(6);

        assertEquals("er", foundSearch1.getSearch());
        assertEquals(16, foundSearch1.getIndex());
        assertEquals("er", foundSearch2.getSearch());
        assertEquals(33, foundSearch2.getIndex());
        assertEquals("er", foundSearch3.getSearch());
        assertEquals(45, foundSearch3.getIndex());
        assertEquals("so", foundSearch4.getSearch());
        assertEquals(29, foundSearch4.getIndex());
        assertEquals("so", foundSearch5.getSearch());
        assertEquals(62, foundSearch5.getIndex());
        assertEquals("in", foundSearch6.getSearch());
        assertEquals(19, foundSearch6.getIndex());
        assertEquals("in", foundSearch7.getSearch());
        assertEquals(66, foundSearch7.getIndex());
    }

    @Test
    public void findAllSearchPhrasesInTextOverlappingSearches() {
        String text = "Hier steht nur fooBarBaz.";
        List<String> startPhrases = CollectionsHelper.buildListFrom(
                "fooBarBaz", "Bar");
        List<FoundSearch> foundSearches = Text.findAllSearchPhrasesInText(text, startPhrases);
        assertEquals(2, foundSearches.size());

        FoundSearch foundSearch1 = foundSearches.get(0);
        FoundSearch foundSearch2 = foundSearches.get(1);

        assertEquals("fooBarBaz", foundSearch1.getSearch());
        assertEquals(15, foundSearch1.getIndex());
        assertEquals("Bar", foundSearch2.getSearch());
        assertEquals(18, foundSearch2.getIndex());
    }

    @Test
    public void removeCommaAtStart() {
        String input = ", Hallo";
        String actual = Text.removeCommaAtStart(input);
        String expected = "Hallo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeCommaAtStartNoComma() {
        String input = " Hallo";
        String actual = Text.removeCommaAtStart(input);
        String expected = "Hallo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeCommaAtEnd() {
        String input = "Hallo,";
        String actual = Text.removeCommaAtEnd(input);
        String expected = "Hallo";
        assertEquals(expected, actual);
    }

    @Test
    public void removeCommaAtEndWithBlank() {
        String input = "Hallo, ";
        String actual = Text.removeCommaAtEnd(input);
        String expected = "Hallo";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaMethodCommentShort() {
        String input = "Hallo. ";
        List<String> expected = CollectionsHelper.buildListFrom("    /** Hallo. */");

        List<String> actual = Text.createJavaMethodComment(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentEmpty() {
        String input = "";
        List<String> expected = CollectionsHelper.buildListFrom("    /**  */");

        List<String> actual = Text.createJavaMethodComment(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentBlank() {
        String input = "   ";
        List<String> expected = CollectionsHelper.buildListFrom("    /**  */");

        List<String> actual = Text.createJavaMethodComment(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentShort87Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 8) + "1234 67";
        assertEquals(87, input.length());
        List<String> expected = CollectionsHelper.buildListFrom("    /** " + input + " */");

        List<String> actual = Text.createJavaMethodComment(input);
        assertEquals(98, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLong88Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 8) + "1234 678";
        assertEquals(88, input.length());
        List<String> expected = CollectionsHelper.buildListFrom("    /** " + input + " */");

        List<String> actual = Text.createJavaMethodComment(input);
        assertEquals(99, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLong89Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 8) + "1234 6789";
        assertEquals(89, input.length());

        List<String> expected = CollectionsHelper.buildListFrom(
                "    /**",
                "     * " + input,
                "     */");

        List<String> actual = Text.createJavaMethodComment(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaMethodCommentLong120Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 11) + "1234 6789X";
        assertEquals(120, input.length());

        String expected1 = Text.multipleString(tenChars, 8) + "1234 6789";
        String expected2 = "1234 6789 1234 6789 1234 6789X";
        List<String> expected = CollectionsHelper.buildListFrom(
                "    /**",
                "     * " + expected1,
                "     * " + expected2,
                "     */");

        List<String> actual = Text.createJavaMethodComment(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaHeadCommentLinesShort() {
        String input = "Hallo. ";
        List<String> expected = CollectionsHelper.buildListFrom(" * Hallo.");

        List<String> actual = Text.createJavaHeadCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaHeadCommentLinesEmpty() {
        String input = "";
        List<String> expected = CollectionsHelper.buildListFrom(" * ");

        List<String> actual = Text.createJavaHeadCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaHeadCommentLinesBlank() {
        String input = "   ";
        List<String> expected = CollectionsHelper.buildListFrom(" * ");

        List<String> actual = Text.createJavaHeadCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaHeadCommentLinesShort96Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 9) + "1234 6";
        assertEquals(96, input.length());
        List<String> expected = CollectionsHelper.buildListFrom(" * " + input);

        List<String> actual = Text.createJavaHeadCommentLines(input);
        assertEquals(99, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaHeadCommentLinesLong97Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 9) + "1234 67";
        assertEquals(97, input.length());

        String expected1 = Text.multipleString(tenChars, 9) + "1234";
        String expected2 = "67";
        List<String> expected = CollectionsHelper.buildListFrom(
                " * " + expected1,
                " * " + expected2);

        List<String> actual = Text.createJavaHeadCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaHeadCommentLinesLong120Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 11) + "1234 6789X";
        assertEquals(120, input.length());

        String expected1 = Text.multipleString(tenChars, 9) + "1234";
        String expected2 = "6789 1234 6789 1234 6789X";
        List<String> expected = CollectionsHelper.buildListFrom(
                " * " + expected1,
                " * " + expected2);

        List<String> actual = Text.createJavaHeadCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaMethodCommentLinesShort() {
        String input = "Hallo. ";
        List<String> expected = CollectionsHelper.buildListFrom("     * Hallo.");

        List<String> actual = Text.createJavaMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLinesEmpty() {
        String input = "";
        List<String> expected = CollectionsHelper.buildListFrom("     * ");

        List<String> actual = Text.createJavaMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLinesBlank() {
        String input = "   ";
        List<String> expected = CollectionsHelper.buildListFrom("     * ");

        List<String> actual = Text.createJavaMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLinesShort92Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 9) + "12";
        assertEquals(92, input.length());
        List<String> expected = CollectionsHelper.buildListFrom("     * " + input);

        List<String> actual = Text.createJavaMethodCommentLines(input);
        assertEquals(99, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaMethodCommentLinesLong93Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 9) + "123";
        assertEquals(93, input.length());

        String expected1 = Text.multipleString(tenChars, 8) + "1234 6789";
        String expected2 = "123";
        List<String> expected = CollectionsHelper.buildListFrom(
                "     * " + expected1,
                "     * " + expected2);

        List<String> actual = Text.createJavaMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaMethodCommentLinesLong116Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 11) + "1234 6";
        assertEquals(116, input.length());

        String expected1 = Text.multipleString(tenChars, 8) + "1234 6789";
        String expected2 = "1234 6789 1234 6789 1234 6";
        List<String> expected = CollectionsHelper.buildListFrom(
                "     * " + expected1,
                "     * " + expected2);

        List<String> actual = Text.createJavaMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesShort() {
        String input = "Hallo. ";
        List<String> expected = CollectionsHelper.buildListFrom("         * Hallo.");

        List<String> actual = Text.createJavaInMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesEmpty() {
        String input = "";
        List<String> expected = CollectionsHelper.buildListFrom("         * ");

        List<String> actual = Text.createJavaInMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesBlank() {
        String input = "   ";
        List<String> expected = CollectionsHelper.buildListFrom("         * ");

        List<String> actual = Text.createJavaInMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesShort88Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 8) + "1234 678";
        assertEquals(88, input.length());
        List<String> expected = CollectionsHelper.buildListFrom("         * " + input);

        List<String> actual = Text.createJavaInMethodCommentLines(input);
        assertEquals(99, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesLong89Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 8) + "1234 6789";
        assertEquals(89, input.length());

        String expected1 = Text.multipleString(tenChars, 8) + "1234";
        String expected2 = "6789";
        List<String> expected = CollectionsHelper.buildListFrom(
                "         * " + expected1,
                "         * " + expected2);

        List<String> actual = Text.createJavaInMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaInMethodCommentLinesLong112Chars() {
        String tenChars = "1234 6789 ";
        String input = Text.multipleString(tenChars, 11) + "12";
        assertEquals(112, input.length());

        String expected1 = Text.multipleString(tenChars, 8) + "1234";
        String expected2 = "6789 1234 6789 1234 6789 12";
        List<String> expected = CollectionsHelper.buildListFrom(
                "         * " + expected1,
                "         * " + expected2);

        List<String> actual = Text.createJavaInMethodCommentLines(input);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaCommandEmpty() {
        int initialIdentationDepth = 4;
        String input = "";
        List<String> expected = CollectionsHelper.buildListFrom("    ");

        List<String> actual = Text.createJavaCommand(input, initialIdentationDepth);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaCommandBlank() {
        int initialIdentationDepth = 4;
        String input = "   ";
        List<String> expected = CollectionsHelper.buildListFrom("    ");

        List<String> actual = Text.createJavaCommand(input, initialIdentationDepth);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaCommandShort100Chars() {
        int initialIdentationDepth = 4;
        String tenChars = "1234 6789 ";
        String input = "    " + Text.multipleString(tenChars, 9) + "567890";
        assertEquals(100, input.length());
        List<String> expected = CollectionsHelper.buildListFrom(input);

        List<String> actual = Text.createJavaCommand(input, initialIdentationDepth);
        assertEquals(100, actual.get(0).length());

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
        }
    }

    @Test
    public void createJavaCommandLong97Chars() {
        int initialIdentationDepth = 4;
        String tenChars = "1234 6789 ";
        String input = "    " + Text.multipleString(tenChars, 9) + "56789 1";
        assertEquals(101, input.length());

        String expected1 = "    " + Text.multipleString(tenChars, 9) + "56789";
        String expected2 = "            1";
        List<String> expected = CollectionsHelper.buildListFrom(
                expected1,
                expected2);

        List<String> actual = Text.createJavaCommand(input, initialIdentationDepth);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void createJavaCommandLong120Chars() {
        int initialIdentationDepth = 4;
        String tenChars = "1234 6789 ";
        String input = "    " + Text.multipleString(tenChars, 11) + "1234 6";
        assertEquals(120, input.length());

        String expected1 = "    " + Text.multipleString(tenChars, 9) + "1234";
        String expected2 = "            6789 1234 6789 1234 6";
        List<String> expected = CollectionsHelper.buildListFrom(
                expected1,
                expected2);

        List<String> actual = Text.createJavaCommand(input, initialIdentationDepth);

        assertEquals(expected.size(), actual.size());
        for (int index = 0; index < actual.size(); ++index) {
            String expectedAtIndex = expected.get(index);
            String actualAtIndex = actual.get(index);
            assertEquals(expectedAtIndex, actualAtIndex);
            assertTrue(actualAtIndex.length() < 100);
        }
    }

    @Test
    public void changeBeginningToGenitivHeulendeEule() {
        String input = "die heulende Eule";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "der heulenden Eule";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivEule() {
        String input = "die Eule";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "der Eule";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivName() {
        String input = "der Name";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des Namens";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivMann() {
        String input = "der Mann";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des Mannes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivGehenderMann() {
        String input = "der gehende Mann";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des gehenden Mannes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivKind() {
        String input = "das Kind";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des Kindes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivSpielendesKind() {
        String input = "das spielende Kind";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des spielenden Kindes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivOrt() {
        String input = "der Ort";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des Ortes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivDunklerOrt() {
        String input = "der dunkle Ort";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des dunklen Ortes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivHrOrt() {
        String input = "der HR-Ort";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des HR-Ortes";
        assertEquals(expected, actual);
    }

    @Test
    public void changeBeginningToGenitivDunklerHrOrt() {
        String input = "der dunkle HR-Ort";
        String actual = Text.changeBeginningToGenitiv(input);
        String expected = "des dunklen HR-Ortes";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceUmlauts() {
        String input = "Ärger Övelgönne Übel Bären Müll süß";
        String actual = Text.replaceUmlauts(input);
        String expected = "Aerger Oevelgoenne Uebel Baeren Muell suess";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceUmlautsEmptyInput() {
        String input = "";
        String actual = Text.replaceUmlauts(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceUmlautsInputWithoutUmlauts() {
        String input = "Apfel Baum Mitte interessant";
        String actual = Text.replaceUmlauts(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaClassName() {
        String input = "Ärger Övelgönne Übel Bären Müll süß";
        String actual = Text.createJavaClassName(input);
        String expected = "AergerOevelgoenneUebelBaerenMuellSuess";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaClassNameInputWithoutUmlauts() {
        String input = "Apfel Baum Mitte interessant";
        String actual = Text.createJavaClassName(input);
        String expected = "ApfelBaumMitteInteressant";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaClassNameEmptyInput() {
        String input = "";
        String actual = Text.createJavaClassName(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaClassNameAuslaendischeMuetter() {
        String input = "Ausländische Mütter";
        String actual = Text.createJavaClassName(input);
        String expected = "AuslaendischeMuetter";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaVariableName() {
        String input = "Ärger Övelgönne Übel Bären Müll süß";
        String actual = Text.createJavaVariableName(input);
        String expected = "aergerOevelgoenneUebelBaerenMuellSuess";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaVariableNameInputWithoutUmlauts() {
        String input = "Apfel Baum Mitte interessant";
        String actual = Text.createJavaVariableName(input);
        String expected = "apfelBaumMitteInteressant";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaVariableNameEmptyInput() {
        String input = "";
        String actual = Text.createJavaVariableName(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaVariableNameAuslaendischeMuetter() {
        String input = "Ausländische Mütter";
        String actual = Text.createJavaVariableName(input);
        String expected = "auslaendischeMuetter";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaVariableNamePerson() {
        String input = "1. Behrens, Johanna, *01.11.2001, Eutin";
        String actual = Text.createJavaVariableName(input);
        String expected = "behrensJohanna01112001Eutin";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaConstantName() {
        String input = "Ärger Övelgönne Übel Bären Müll süß";
        String actual = Text.createJavaConstantName(input);
        String expected = "AERGER_OEVELGOENNE_UEBEL_BAEREN_MUELL_SUESS";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaConstantNameInputWithoutUmlauts() {
        String input = "Apfel Baum Mitte interessant";
        String actual = Text.createJavaConstantName(input);
        String expected = "APFEL_BAUM_MITTE_INTERESSANT";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaConstantCamelCase() {
        String input = "apfelBaumMitteInteressant";
        String actual = Text.createJavaConstantName(input);
        String expected = "APFEL_BAUM_MITTE_INTERESSANT";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaConstantNameEmptyInput() {
        String input = "";
        String actual = Text.createJavaConstantName(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void createJavaConstantNameAuslaendischeMuetter() {
        String input = "Ausländische Mütter";
        String actual = Text.createJavaConstantName(input);
        String expected = "AUSLAENDISCHE_MUETTER";
        assertEquals(expected, actual);
    }

    @Test
    public void splitCamelCaseHrNummer() {
        String input = "hrNummer";
        List<String> parts = Text.splitCamelCase(input);
        assertEquals(2, parts.size());
        assertEquals("hr", parts.get(0));
        assertEquals("Nummer", parts.get(1));
    }

    @Test
    public void splitCamelCaseSessionManager() {
        String input = "SessionManager";
        List<String> parts = Text.splitCamelCase(input);
        assertEquals(2, parts.size());
        assertEquals("Session", parts.get(0));
        assertEquals("Manager", parts.get(1));
    }

    @Test
    public void splitCamelCaseEmptyInput() {
        String input = "";
        List<String> parts = Text.splitCamelCase(input);
        assertEquals(0, parts.size());
    }

    @Test
    public void splitCamelCaseNoCamelCase() {
        String input = "hr nummer";
        assertThrows(IllegalArgumentException.class, () ->  Text.splitCamelCase(input));
    }

    @Test
    public void separateCamelCaseHrNummer() {
        String camelCase = "hrNummer";
        String fill = "-";
        String actual = Text.separateCamelCase(camelCase, fill);
        String expected = "hr-nummer";
        assertEquals(expected, actual);
    }

    @Test
    public void separateCamelCaseSessionManager() {
        String camelCase = "SessionManager";
        String fill = "-";
        String actual = Text.separateCamelCase(camelCase, fill);
        String expected = "session-manager";
        assertEquals(expected, actual);
    }

    @Test
    public void separateCamelCaseNoCamelCase() {
        String camelCase = "hr nummer";
        String fill = "-";
        assertThrows(IllegalArgumentException.class, () ->  Text.separateCamelCase(camelCase, fill));
    }

    @Test
    public void breakLongJavaMethodStart1() {
        String code = "    public String generateTextHtml(String text, String name, String town, "
                + "String hrTown, String hrNumber) {";
        List<String> lines = Text.breakLongJavaMethodStart(code);
        assertEquals(2, lines.size());
        assertEquals("    public String generateTextHtml(String text, String name, String town, "
                + "String hrTown,",
                lines.get(0));
        assertEquals("            String hrNumber) {", lines.get(1));
    }

    @Test
    public void breakLongJavaMethodStart1b() {
        String code = "    public String generateTextHtml(int textxxx, int namexxx, int townxxx, "
                + "int hrTownxxx, int hrNumberxxx) {";
        List<String> lines = Text.breakLongJavaMethodStart(code);
        assertEquals(2, lines.size());
        assertEquals("    public String generateTextHtml(int textxxx, int namexxx, int townxxx, "
                + "int hrTownxxx,",
                lines.get(0));
        assertEquals("            int hrNumberxxx) {", lines.get(1));
    }

    @Test
    public void breakLongJavaMethodStart2() {
        String code = "    public String colorText(String text, String name, String town, "
                + "String hrTown, String hrNumber, String search, int htmlFontSize) {";;
        List<String> lines = Text.breakLongJavaMethodStart(code);
        assertEquals(2, lines.size());
        assertEquals("    public String colorText(String text, String name, String town, "
                + "String hrTown, String hrNumber,",
                lines.get(0));
        assertEquals("            String search, int htmlFontSize) {", lines.get(1));
    }

    @Test
    public void endsWithBigLetterEmptyInput() {
        String input = "";
        assertFalse(Text.endsWithBigLetter(input));
    }

    @Test
    public void endsWithBigLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.endsWithBigLetter(input));
    }

    @Test
    public void endsWithBigLetterCharDigit() {
        String input = "A";
        assertTrue(Text.endsWithBigLetter(input));
    }

    @Test
    public void endsWithBigLetterSingleLetterWithSpaceBehind() {
        String input = "A ";
        assertFalse(Text.endsWithBigLetter(input));
    }

    @Test
    public void endsWithSmallLetterEmptyInput() {
        String input = "";
        assertFalse(Text.endsWithSmallLetter(input));
    }

    @Test
    public void endsWithSmallLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.endsWithSmallLetter(input));
    }

    @Test
    public void endsWithSmallLetterChar() {
        String input = "a";
        assertTrue(Text.endsWithSmallLetter(input));
    }

    @Test
    public void endsWithSmallLetterSingleLetterWithSpaceBehind() {
        String input = "a ";
        assertFalse(Text.endsWithSmallLetter(input));
    }

    @Test
    public void endsWithLetterEmptyInput() {
        String input = "";
        assertFalse(Text.endsWithLetter(input));
    }

    @Test
    public void endsWithLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.endsWithLetter(input));
    }

    @Test
    public void endsWithLetterChar() {
        String input = "a";
        assertTrue(Text.endsWithLetter(input));
    }

    @Test
    public void endsWithLetterBigChar() {
        String input = "A";
        assertTrue(Text.endsWithLetter(input));
    }

    @Test
    public void endsWithLetterSingleLetterWithSpaceBehind() {
        String input = "a ";
        assertFalse(Text.endsWithLetter(input));
    }

    @Test
    public void startsWithBigLetterEmptyInput() {
        String input = "";
        assertFalse(Text.startsWithBigLetter(input));
    }

    @Test
    public void startsWithBigLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.startsWithBigLetter(input));
    }

    @Test
    public void startsWithBigLetterCharDigit() {
        String input = "A";
        assertTrue(Text.startsWithBigLetter(input));
    }

    @Test
    public void startsWithBigLetterSingleLetterWithSpaceBefore() {
        String input = " A";
        assertFalse(Text.startsWithBigLetter(input));
    }

    @Test
    public void startsWithSmallLetterEmptyInput() {
        String input = "";
        assertFalse(Text.startsWithSmallLetter(input));
    }

    @Test
    public void startsWithSmallLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.startsWithSmallLetter(input));
    }

    @Test
    public void startsWithSmallLetterChar() {
        String input = "a";
        assertTrue(Text.startsWithSmallLetter(input));
    }

    @Test
    public void startsWithSmallLetterSingleLetterWithSpaceBefore() {
        String input = " a";
        assertFalse(Text.startsWithSmallLetter(input));
    }

    @Test
    public void startsWithLetterEmptyInput() {
        String input = "";
        assertFalse(Text.startsWithLetter(input));
    }

    @Test
    public void startsWithLetterSingleDigit() {
        String input = "7";
        assertFalse(Text.startsWithLetter(input));
    }

    @Test
    public void startsWithLetterChar() {
        String input = "a";
        assertTrue(Text.startsWithLetter(input));
    }

    @Test
    public void startsWithLetterBigChar() {
        String input = "A";
        assertTrue(Text.startsWithLetter(input));
    }

    @Test
    public void startsWithLetterSingleLetterWithSpaceBefore() {
        String input = " a";
        assertFalse(Text.startsWithLetter(input));
    }

    @Test
    public void endsWithRegex() {
        String text = "Apfelbaum";
        String regex = "Banane|[Bb]aum";
        assertTrue(Text.endsWithRegex(text, regex));
    }

    @Test
    public void endsWithRegexNotEnding() {
        String text = "Apfelbäume";
        String regex = "Banane|[Bb]aum";
        assertFalse(Text.endsWithRegex(text, regex));
    }

    @Test
    public void endsWithRegexColorFF() {
        String text = "<style=\"color:FF";
        String regex = "color:.{0,2}";
        assertTrue(Text.endsWithRegex(text, regex));
    }

    @Test
    public void endsWithRegexColorF() {
        String text = "<style=\"color:F";
        String regex = "color:.{0,2}";
        assertTrue(Text.endsWithRegex(text, regex));
    }

    @Test
    public void endsWithRegexColor() {
        String text = "<style=\"color:";
        String regex = "color:.{0,2}";
        assertTrue(Text.endsWithRegex(text, regex));
    }

    @Test
    public void endsWithRegexColorFFF() {
        String text = "<style=\"color:FFF";
        String regex = "color:.{0,2}";
        assertFalse(Text.endsWithRegex(text, regex));
    }

    @Test
    public void replacePhrasesInTextEverythingEmpty() {
        String text = "";
        String search = "";
        String replacement = "";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void replacePhrasesInTextSearchAtFront() {
        String text = "Apfel Banane";
        String search = "Apfel";
        String replacement = "Birne";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "Birne Banane";
        assertEquals(expected, actual);
    }

    @Test
    public void replacePhrasesInTextSearchAtEnd() {
        String text = "Banane Apfel";
        String search = "Apfel";
        String replacement = "Birne";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "Banane Birne";
        assertEquals(expected, actual);
    }

    @Test
    public void replacePhrasesInTextOnlySearch() {
        String text = "Apfel";
        String search = "Apfel";
        String replacement = "Birne";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "Birne";
        assertEquals(expected, actual);
    }

    @Test
    public void replacePhrasesInTextOnlyPart() {
        String text = "Apfelbaum";
        String search = "Apfel";
        String replacement = "Birne";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "Apfelbaum";
        assertEquals(expected, actual);
    }

    @Test
    public void replacePhrasesInTextMixed() {
        String text = "Apfel: Ein Apfel hängt im Apfelbaum - phantastische Äpfel! Und: Apfel";
        String search = "Apfel";
        String replacement = "Birne";
        String actual = Text.replacePhrasesInText(text, search, replacement);
        String expected = "Birne: Ein Birne hängt im Apfelbaum - phantastische Äpfel! Und: Birne";
        assertEquals(expected, actual);
    }

    @Test
    public void isSingleQuoted() {
        assertFalse(Text.isSingleQuoted(""));
        assertFalse(Text.isSingleQuoted("foo"));
        assertFalse(Text.isSingleQuoted("'foo"));
        assertFalse(Text.isSingleQuoted("foo'"));
        assertTrue(Text.isSingleQuoted("'foo'"));
        assertTrue(Text.isSingleQuoted("''"));
    }

    @Test
    public void deSingleQuoteQuotedString() {
        String input = "'foo'";
        String actual = Text.deSingleQuoteQuotedString(input);
        String expected = "foo";
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void deSingleQuoteQuotedStringOnNotQuotedString() {
        Text.deSingleQuoteQuotedString("foo");
    }

    @Test
    public void containsOnlyBigLettersOnlyBigLetters() {
        String input = "ABC";
        assertTrue(Text.containsOnlyBigLetters(input));
    }

    @Test
    public void containsOnlyBigLettersOnlyBigLettersWithAUmlaut() {
        String input = "ÄPFEL";
        assertTrue(Text.containsOnlyBigLetters(input));
    }

    @Test
    public void containsOnlyBigLettersBigAndASmallLettersWithAUmlaut() {
        String input = "Äpfel";
        assertFalse(Text.containsOnlyBigLetters(input));
    }

    @Test
    public void containsOnlyBigLettersEmpty() {
        String input = "";
        assertFalse(Text.containsOnlyBigLetters(input));
    }

    @Test
    public void containsOnlyBigLettersOnlySmallLetters() {
        String input = "abc";
        assertFalse(Text.containsOnlyBigLetters(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLetters() {
        String input = "ABC";
        assertTrue(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersWithAUmlaut() {
        String input = "ÄPFEL";
        assertTrue(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensBigAndASmallLettersWithAUmlaut() {
        String input = "Äpfel";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensEmpty() {
        String input = "";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlySmallLetters() {
        String input = "abc";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersAndSpace() {
        String input = "ABC DEF";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersAndHyphen() {
        String input = "ABC-DEF";
        assertTrue(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersAndTwoHyphens() {
        String input = "ABC-DEF-GHI";
        assertTrue(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersAndThreeHyphens() {
        String input = "ABC-DEF-GHI-JKL";
        assertTrue(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyHyphen() {
        String input = "-";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensBigLettersWithHyphenAtEnd() {
        String input = "ABC-";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensBigLettersWithHyphenAtStart() {
        String input = "-ABC";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void containsOnlyBigLettersAndHyphensOnlyBigLettersAndTwoConsequentHyphen() {
        String input = "ABC--DEF";
        assertFalse(Text.containsOnlyBigLettersAndHyphens(input));
    }

    @Test
    public void createLongJavaStringDefinition1() {
        String variableName = "sentence";
        String variableContent = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        int indentation = 4;
        int maxLength = 20;
        List<String> lines = Text.createLongJavaStringDefinition(variableName, variableContent,
                indentation, maxLength);
        assertEquals(7, lines.size());
        assertEquals("    String sentence = \"\"",
                lines.get(0));
        assertEquals("            + \"Mal \"",
                lines.get(1));
        assertEquals("            + \"sehen \"",
                lines.get(2));
        assertEquals("            + \"was \"",
                lines.get(3));
        assertEquals("            + \"so \"",
                lines.get(4));
        assertEquals("            + \"AUSSERGEWÖHNLICHLANGES \"",
                lines.get(5));
        assertEquals("            + \"passiert!\";",
                lines.get(6));
    }

    @Test
    public void createLongJavaStringDefinition2() {
        String variableName = "sentence";
        String variableContent = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        int indentation = 4;
        int maxLength = 30;
        List<String> lines = Text.createLongJavaStringDefinition(variableName, variableContent,
                indentation, maxLength);
        assertEquals(5, lines.size());
        assertEquals("    String sentence = \"\"",
                lines.get(0));
        assertEquals("            + \"Mal sehen \"",
                lines.get(1));
        assertEquals("            + \"was so \"",
                lines.get(2));
        assertEquals("            + \"AUSSERGEWÖHNLICHLANGES \"",
                lines.get(3));
        assertEquals("            + \"passiert!\";",
                lines.get(4));
    }

    @Test
    public void createLongJavaStringDefinition3() {
        String variableName = "sentence";
        String variableContent = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        int indentation = 4;
        int maxLength = 50;
        List<String> lines = Text.createLongJavaStringDefinition(variableName, variableContent,
                indentation, maxLength);
        assertEquals(3, lines.size());
        assertEquals("    String sentence = \"\"",
                lines.get(0));
        assertEquals("            + \"Mal sehen was so \"",
                lines.get(1));
        assertEquals("            + \"AUSSERGEWÖHNLICHLANGES passiert!\";",
                lines.get(2));
    }

    @Test
    public void createLongJavaStringDefinition4() {
        String variableName = "sentence";
        String variableContent = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        int indentation = 4;
        int maxLength = 100;
        List<String> lines = Text.createLongJavaStringDefinition(variableName, variableContent,
                indentation, maxLength);
        assertEquals(2, lines.size());
        assertEquals("    String sentence = \"\"",
                lines.get(0));
        assertEquals("            + \"Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!\";",
                lines.get(1));
    }

    @Test
    public void createLongJavaStringDefinition5() {
        String variableName = "sentence";
        String variableContent = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        List<String> lines = Text.createLongJavaStringDefinition(variableName, variableContent);
        assertEquals(2, lines.size());
        assertEquals("        String sentence = \"\"",
                lines.get(0));
        assertEquals("                + \"Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!\";",
                lines.get(1));
    }

    @Test
    public void isInsideBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 13;
        assertTrue(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceIndexBeforeBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 3;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceIndexAfterBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 18;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceIndexOfClosingBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 17;
        assertTrue(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceIndexOfOpeningBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 9;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceIndexBehindOpeningBrace() {
        String text = "In einer (Klammer) jawohl!";
        int index = 10;
        assertTrue(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceNoClosingBrace() {
        String text = "In einer (Klammer( jawohl!";
        int index = 13;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceNoOpenbingBrace() {
        String text = "In einer )Klammer) jawohl!";
        int index = 13;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void isInsideBraceWrongBraces() {
        String text = "In einer )Klammer( jawohl!";
        int index = 13;
        assertFalse(Text.isInsideBrace(text, index));
    }

    @Test
    public void shortenJavaClassName() {
        String className = Text.class.getName();
        String actual = Text.shortenJavaClassName(className);
        String expected = "Text";
        assertEquals(expected, actual);
    }

    @Test
    public void getShortenedJavaClassName() {
        String actual = Text.getShortenedJavaClassName(Text.class);
        String expected = "Text";
        assertEquals(expected, actual);
    }

    @Test
    public void splitByFirstOccurence() {
        String text = ""
                + "Amtsgericht Charlottenburg Ausdruck Handelsregister A HRA 2046 B" + LINE_BREAK
                + "Abruf vom 27.04.2023, 11:55";
        Pair<String> splitResult = Text.splitByFirstOccurence(text, "Ausdruck", "register");
        String front = splitResult.getFirst();
        String rear = splitResult.getSecond();
        assertEquals("Amtsgericht Charlottenburg ", front);
        assertEquals("Ausdruck Handelsregister A HRA 2046 B" + LINE_BREAK
                + "Abruf vom 27.04.2023, 11:55", rear);
    }

    @Test
    public void splitAfterFirstOccurence() {
        String text = ""
                + "Handelsregister A des Amtsgerichts Stendal Ausdruck"
                        + "         Nummer der Firma: HRA 10045" + LINE_BREAK
                + "Abruf vom 02.06.2023 13:49         Seite 1 von 2";
        Pair<String> splitResult = Text.splitAfterFirstOccurence(text, "gerichts");
        String front = splitResult.getFirst();
        String rear = splitResult.getSecond();
        assertEquals("Handelsregister A des Amtsgerichts", front);
        assertEquals(" Stendal Ausdruck"
                + "         Nummer der Firma: HRA 10045" + LINE_BREAK
        + "Abruf vom 02.06.2023 13:49         Seite 1 von 2", rear);
    }

    @Test
    public void splitByFirstOccurenceWithoutSearch() {
        String text = ""
                + "Amtsgericht Charlottenburg Ausdruck Handelsregister A HRA 2046 B" + LINE_BREAK
                + "Abruf vom 27.04.2023, 11:55";
        Pair<String> splitResult =
                Text.splitByFirstOccurenceWithoutSearch(text, "Ausdruck", "register");
        String front = splitResult.getFirst();
        String rear = splitResult.getSecond();
        assertEquals("Amtsgericht Charlottenburg ", front);
        assertEquals(" Handelsregister A HRA 2046 B" + LINE_BREAK
                + "Abruf vom 27.04.2023, 11:55", rear);
    }

    @Test
    public void isMultipleStringSimple() {
        String text = "aaa";
        String part = "a";
        assertTrue(Text.isMultipleString(text, part));
    }

    @Test
    public void isMultipleStringWrongNumber() {
        String text = "aaa";
        String part = "aa";
        assertFalse(Text.isMultipleString(text, part));
    }

    @Test
    public void isMultipleStringEmptyPart() {
        String text = "aaa";
        String part = "";
        assertFalse(Text.isMultipleString(text, part));
    }

    @Test
    public void isMultipleStringEmptyTextAndPart() {
        String text = "";
        String part = "";
        assertTrue(Text.isMultipleString(text, part));
    }

    @Test
    public void isMultipleStringEmptyText() {
        String text = "";
        String part = "a";
        assertFalse(Text.isMultipleString(text, part));
    }

    @Test
    public void isMultipleStringWrongBetwen() {
        String text = "aabbaa";
        String part = "aa";
        assertFalse(Text.isMultipleString(text, part));
    }

    @Test
    public void replaceFirstMatchAtFront() {
        String text = "foo bar baz foo bar baz";
        String search = "foo";
        String replacement = "FOO";
        String actual = Text.replaceFirstMatch(text, search, replacement);
        String expected = "FOO bar baz foo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceFirstMatchMiddle() {
        String text = "foo bar baz foo bar baz";
        String search = "bar";
        String replacement = "BAR";
        String actual = Text.replaceFirstMatch(text, search, replacement);
        String expected = "foo BAR baz foo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceFirstMatchEnd() {
        String text = "foo bar baz foo bar faz";
        String search = "faz";
        String replacement = "FAZ";
        String actual = Text.replaceFirstMatch(text, search, replacement);
        String expected = "foo bar baz foo bar FAZ";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceFirstMatchSearchNotContained() {
        String text = "foo bar baz foo bar baz";
        String search = "boo";
        String replacement = "BOO";
        String actual = Text.replaceFirstMatch(text, search, replacement);
        String expected = "foo bar baz foo bar baz";
        assertEquals(expected, actual);
    }

    @Test
    public void textToCharactersList() {
        String text = "Hallo";
        List<String> list = Text.textToCharactersList(text);
        assertEquals(5, list.size());
        assertEquals("H", list.get(0));
        assertEquals("a", list.get(1));
        assertEquals("l", list.get(2));
        assertEquals("l", list.get(3));
        assertEquals("o", list.get(4));
    }

    @Test
    public void textToCharactersListEmptyInput() {
        String text = "";
        List<String> list = Text.textToCharactersList(text);
        assertEquals(0, list.size());
    }

    @Test
    public void concatenateTwoEmptyStrings() {
        String first = "";
        String second = "";
        String actual = Text.concatenate(first, second);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateFilledWithEmptyString() {
        String first = "abc";
        String second = "";
        String actual = Text.concatenate(first, second);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateEmptyWithFilledString() {
        String first = "";
        String second = "abc";
        String actual = Text.concatenate(first, second);
        String expected = "abc";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateTwoStringsWithoutSpaces() {
        String first = "abc";
        String second = "def";
        String actual = Text.concatenate(first, second);
        String expected = "abc def";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateFirstStringWithSpaceAtEndAndSecondStringWithoutSpaces() {
        String first = "abc ";
        String second = "def";
        String actual = Text.concatenate(first, second);
        String expected = "abc def";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateFirstStringWithoutSpacesAndSecondStringWithSpaceAtStart() {
        String first = "abc";
        String second = " def";
        String actual = Text.concatenate(first, second);
        String expected = "abc def";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateFirstStringWithSpaceAtEndAndSecondStringWithSpaceAtStart() {
        String first = "abc ";
        String second = " def";
        String actual = Text.concatenate(first, second);
        String expected = "abc  def";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateListNormalCase() {
        List<String> parts = CollectionsHelper.buildListFrom("abc", "def", "ghi", "jk");
        String actual = Text.concatenate(parts);
        String expected = "abc def ghi jk";
        assertEquals(expected, actual);
    }

    @Test
    public void concatenateList() {
        List<String> parts = CollectionsHelper.buildListFrom("abc", " def ", " ghi", "jk");
        String actual = Text.concatenate(parts);
        String expected = "abc def  ghi jk";
        assertEquals(expected, actual);
    }

    @Test
    public void replaceSpecialUtf8Chars() {
        String text = "abc „foo“ – eins • zwei ’drei‘‚ vier";
        String actual = Text.replaceSpecialUtf8Chars(text);
        String expected = "abc \"foo\" - eins - zwei 'drei', vier";
        assertEquals(expected, actual);
    }

    @Test
    public void isInAlphabeticalOrderEmptyList() {
        assertTrue(Text.isInAlphabeticalOrder());
    }

    @Test
    public void isInAlphabeticalOrderListWithOneElement() {
        assertTrue(Text.isInAlphabeticalOrder("a"));
    }

    @Test
    public void isInAlphabeticalOrderListWithTwoElementsCorrect() {
        assertTrue(Text.isInAlphabeticalOrder("a", "b"));
    }

    @Test
    public void isInAlphabeticalOrderListWithTwoEqualElements() {
        assertTrue(Text.isInAlphabeticalOrder("a", "a"));
    }

    @Test
    public void isInAlphabeticalOrderListWithTwoElementsIncorrect() {
        assertFalse(Text.isInAlphabeticalOrder("b", "a"));
    }

    @Test
    public void isInAlphabeticalOrderListWithThreeElementsCorrect() {
        assertTrue(Text.isInAlphabeticalOrder("a", "b", "c"));
    }

    @Test
    public void isInAlphabeticalOrderListWithThreeElementsIncorrect() {
        assertFalse(Text.isInAlphabeticalOrder("a", "c", "b"));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsEmptyInputs() {
        String input1 = "";
        String input2 = "";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsNonBraceInputs() {
        String input1 = "aaa";
        String input2 = "aaa";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsRoundBraceInputs() {
        String input1 = "(sicher)";
        String input2 = "(sicher)";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsNonBraceInputVersusRoundBraceInput() {
        String input1 = "sicher";
        String input2 = "(sicher)";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsRoundBraceInputVersusNonBraceInput() {
        String input1 = "(sicher)";
        String input2 = "sicher";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsSquareBraceInputs() {
        String input1 = "[sicher]";
        String input2 = "[sicher]";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsNonBraceInputVersusSquareBraceInput() {
        String input1 = "sicher";
        String input2 = "[sicher]";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsSquareBraceInputVersusNonBraceInput() {
        String input1 = "[sicher]";
        String input2 = "sicher";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsSquareBraceInputVersusRoundBraceInput() {
        String input1 = "[sicher]";
        String input2 = "(sicher)";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsRoundBraceInputVersusSquareBraceInput() {
        String input1 = "(sicher)";
        String input2 = "[sicher]";
        assertTrue(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void equalsIgnoringRoundOrSquareBracketsNotEqualInputs() {
        String input1 = "sicher";
        String input2 = "'sicher'";
        assertFalse(Text.equalsIgnoringRoundOrSquareBrackets(input1, input2));
    }

    @Test
    public void compareStringsBetterNormal() {
        String text1 = "Zufall";
        String text2 = "aber aber";
        assertTrue(text1.compareTo(text2) < 0); // "Zufall" vor "aber aber"
        assertTrue(Text.compareStringsBetter(text1, text2) > 0); // Zufall nach "aber aber"
    }

    @Test
    public void compareStringsBetterBigAndLower() {
        String text1 = "Zufall";
        String text2 = "zufall";
        assertTrue(text1.compareTo(text2) < 0); // "Zufall" vor "zufall"
        assertTrue(Text.compareStringsBetter(text1, text2) < 0); // "Zufall" vor "zufall"
    }

    @Test
    public void compareStringsBetterLowerAndBig() {
        String text1 = "zufall";
        String text2 = "Zufall";
        assertTrue(text1.compareTo(text2) > 0); // "Zufall" vor "zufall"
        assertTrue(Text.compareStringsBetter(text1, text2) > 0); // "Zufall" vor "zufall"
    }

    @Test
    public void compareStringsBetterUmlaute() {
        String text1 = "Ordnung";
        String text2 = "öffentlich";
        assertTrue(text1.compareTo(text2) < 0); // "Ordnung" vor "öffentlich"
        assertTrue(Text.compareStringsBetter(text1, text2) > 0); // "Ordnung" nach "öffentlich"
    }

    @Test
    public void prepareForFilenameEmptyInput() {
        String input = "";
        String actual = Text.prepareForFilename(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void prepareForFilenameOneSpaceInput() {
        String input = " ";
        String actual = Text.prepareForFilename(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void prepareForFilenameMultipleSpaceInput() {
        String input = "     ";
        String actual = Text.prepareForFilename(input);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void prepareForFilenameNormal() {
        String input = "createReplaced";
        String actual = Text.prepareForFilename(input);
        String expected = "create_replaced";
        assertEquals(expected, actual);
    }

    @Test
    public void containsAsWholeWord01() {
        String text = "Ist es enthalten?";
        String search = "enthalten";
        assertTrue(Text.containsAsWholeWord(text, search));
    }

    @Test
    public void containsAsWholeWord02() {
        String text = "Ist es enthalten?";
        String search = "ent";
        assertFalse(Text.containsAsWholeWord(text, search));
    }

    @Test
    public void containsAsWholeWord03() {
        String text = "Ist es enthalten?";
        String search = "halt";
        assertFalse(Text.containsAsWholeWord(text, search));
    }

    @Test
    public void containsAsWholeWord04() {
        String text = "Ist es enthalten?";
        String search = "Ist";
        assertTrue(Text.containsAsWholeWord(text, search));
    }

    @Test
    public void containsAsWholeWordNotFirstFound() {
        String text = "Ist es enthalten oder vielleicht doch halt erst hier hinten?";
        String search = "halt";
        assertTrue(Text.containsAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWord01() {
        String text = "Ist es enthalten?";
        String search = "enthalten";
        assertEquals(7, Text.indexOfAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWord02() {
        String text = "Ist es enthalten?";
        String search = "ent";
        assertEquals(-1, Text.indexOfAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWord03() {
        String text = "Ist es enthalten?";
        String search = "halt";
        assertEquals(-1, Text.indexOfAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWord04() {
        String text = "Ist es enthalten?";
        String search = "Ist";
        assertEquals(0, Text.indexOfAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWordNotFirstFound() {
        String text = "Ist es enthalten oder vielleicht doch halt erst hier hinten?";
        String search = "halt";
        assertEquals(38, Text.indexOfAsWholeWord(text, search));
    }

    @Test
    public void indexOfAsWholeWordFromStartIndex() {
        String text = "Ist es enthalten?";
        String search = "enthalten";
        int startIndex = 8;
        assertEquals(-1, Text.indexOfAsWholeWord(text, search, startIndex));
    }

    @Test
    public void hideWholeWordsInText() {
        String text = "Hier drin soll Obst wie Apfel, Birne und Orange ersetzt werden.";
        List<String> wordsToHide = CollectionsHelper.buildListFrom("Apfel", "Birne", "Orange");
        String actual = Text.hideWholeWordsInText(text, wordsToHide);
        String expected = "Hier drin soll Obst wie #####, ##### und ###### ersetzt werden.";
        assertEquals(expected, actual);
    }

    @Test
    public void hideWholeWordsInTextFindMultiplesToo() {
        String text = "Auch mehrfach auftauchende Begriffe sollen versteckt werden: "
                + "Apfel, Birne und Apfel";
        List<String> wordsToHide = CollectionsHelper.buildListFrom("Apfel", "Birne", "Orange");
        String actual = Text.hideWholeWordsInText(text, wordsToHide);
        String expected = "Auch mehrfach auftauchende Begriffe sollen versteckt werden: "
                + "#####, ##### und #####";
        assertEquals(expected, actual);
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesSameQuotes() {
        String text = "bla";
        String startQuote = "\"";
        String endQute = "\"";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesMoreOpeningThanClosing() {
        String text = "„Baum“ und „Strauch";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesLessOpeningThanClosing() {
        String text = "„Baum“ und Strauch“";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesWrong1() {
        String text = "„Baum„ und “Strauch“";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesWrong2() {
        String text = "“Baum„";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesWrong3() {
        String text = "“Baum“ und „Strauch„";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesNoQuotesContained() {
        String text = "'Baum' und 'Strauch'";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertTrue(quotedTexts.isEmpty());
    }

    @Test
    public void findTextsBetweenDifferentStartAndEndQuotesTwoFoundThings() {
        String text = "„Baum“ und „Strauch“";
        String startQuote = "„";
        String endQute = "“";
        List<String> quotedTexts = Text.findTextsBetweenDifferentStartAndEndQuotes(text, startQuote,
                endQute);
        assertEquals(2, quotedTexts.size());
        assertEquals("Baum", quotedTexts.get(0));
        assertEquals("Strauch", quotedTexts.get(1));
    }

    @Test
    public void findTextsBetweenJapaneseQuotesTwoFoundThings() {
        String text = "「Baum」 und 「Strauch」";
        List<String> quotedTexts = Text.findTextsBetweenJapaneseQuotes(text);
        assertEquals(2, quotedTexts.size());
        assertEquals("Baum", quotedTexts.get(0));
        assertEquals("Strauch", quotedTexts.get(1));
    }

    @Test
    public void removeSpacesAtStartEmptyInput() {
        String text = "";
        String actual = Text.removeSpacesAtStart(text);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesAtStartNoSpacesAtStartInput() {
        String text = "Ein Baum Steht am Wgesrand!    ";
        String actual = Text.removeSpacesAtStart(text);
        String expected = "Ein Baum Steht am Wgesrand!    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesAtStartSpacesAtStartInput() {
        String text = "     Ein Baum Steht am Wgesrand!    ";
        String actual = Text.removeSpacesAtStart(text);
        String expected = "Ein Baum Steht am Wgesrand!    ";
        assertEquals(expected, actual);
    }

    @Test
    public void removeSpacesAtStartOnlySpacesInput() {
        String text = "     ";
        String actual = Text.removeSpacesAtStart(text);
        String expected = "";
        assertEquals(expected, actual);
    }

}
