package de.duehl.swing.ui;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Container;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;

/**
 * Diese Klasse berechnet von einem Container (also z.B. einem JPanel) den wirklich zum Bemalen zur
 * Verfügung stehenden Bereich und stellt verschiedenste Methoden zur Verfügung, diesen zu
 * erfragen.
 *
 * @version 1.01     2017-07-18
 * @author Christian Dühl
 */

public class PaintableAreaCalculator {

    /** Breite des bemalbaren Bereichs. */
    private final int paintableWidth;

    /** Höhe des bemalbaren Bereichs. */
    private final int paintableHeight;


    /** x-Koordinate der oberen, linken Ecke des bemalbaren Bereichs. */
    private final int upperLeftX;

    /** y-Koordinate der oberen, linken Ecke des bemalbaren Bereichs. */
    private final int upperLeftY;

    /** Die obere, linke Ecke. */
    private final Point upperLeftPoint;


    /** x-Koordinate der unteren, rechten Ecke des bemalbaren Bereichs. */
    private final int lowerRightX;

    /** y-Koordinate der unteren, rechten Ecke des bemalbaren Bereichs. */
    private final int lowerRightY;

    /** Die untere, rechte Ecke. */
    private final Point lowerRightPoint;


    /** Bemalbarer Bereich als Rechteck. */
    private final Rectangle rectangle;

    /**
     * Konstruktor.
     *
     * Die wirklich Zeichenfläche wird ermittelt nach:
     * http://userpage.fu-berlin.de/~ram/pub/pub_jf47ht81Ht/java_swing_zeichenflaeche_ermitteln
     *
     * @param container
     *            Zu untersuchendes Gui-Element.
     */
    public PaintableAreaCalculator(Container container) {
        /* Gesamtmaße der Komponente bestimmen: */
        int totalPanelWidth = container.getWidth();
        int totalPanelHeight = container.getHeight();

        /* Größe des bemalbaren Bereiches ermitteln: */
        Insets insets = container.getInsets();
        paintableWidth = totalPanelWidth - insets.left - insets.right - 1;
        paintableHeight = totalPanelHeight - insets.top - insets.bottom - 1;

        /* Obere linke Ecke des bemalbaren Bereiches bestimmen: */
        upperLeftX = insets.left;
        upperLeftY = insets.top;
        upperLeftPoint = new Point(upperLeftX, upperLeftY);

        /* Untere rechte Ecke des bemalbaren Bereiches bestimmen: */
        lowerRightX = totalPanelWidth - insets.right - 1;
        lowerRightY = totalPanelHeight - insets.bottom - 1;
        lowerRightPoint = new Point(lowerRightX, lowerRightY);

        /* Rechteck bestimmen: */
        rectangle = new Rectangle(upperLeftX, upperLeftY, paintableWidth, paintableHeight);
    }

    /** Getter für die Breite des bemalbaren Bereichs. */
    public int getPaintableWidth() {
        return paintableWidth;
    }

    /** Getter für die Höhe des bemalbaren Bereichs. */
    public int getPaintableHeight() {
        return paintableHeight;
    }

    /** Getter für die x-Koordinate der oberen, linken Ecke des bemalbaren Bereichs. */
    public int getUpperLeftX() {
        return upperLeftX;
    }

    /** Getter für die y-Koordinate der oberen, linken Ecke des bemalbaren Bereichs. */
    public int getUpperLeftY() {
        return upperLeftY;
    }

    /** Getter für die obere, linke Ecke. */
    public Point getUpperLeftPoint() {
        return upperLeftPoint;
    }

    /** Getter für die x-Koordinate der unteren, rechten Ecke des bemalbaren Bereichs. */
    public int getLowerRightX() {
        return lowerRightX;
    }

    /** Getter für die y-Koordinate der unteren, rechten Ecke des bemalbaren Bereichs. */
    public int getLowerRightY() {
        return lowerRightY;
    }

    /** Getter für die untere, rechte Ecke. */
    public Point getLowerRightPoint() {
        return lowerRightPoint;
    }

    /** Getter für den bemalbaren Bereich als Rechteck. */
    public Rectangle getRectangle() {
        return rectangle;
    }

}
