package de.duehl.swing.ui.buttons;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.SwingUtilities;

/**
 * Diese Klasse stellt Hilfsmethoden für die speziellen Buttons zur Verfügung.
 * Diese Klasse ist bewusst nicht public.
 *
 * @version 1.01     2022-10-11
 * @author Christian Dühl
 */

public class ButtonHelper {

    private ButtonHelper() {}

    /** Default-Maße der selbstgezeichneten Buttons. */
    public final static Dimension BUTTON_DIMENSION = new Dimension(17, 17);

    /**
     * Ein MouseListener, der den Rahmen anzeigt, wenn die Maus über dem Button
     * ist.
     */
    public final static MouseListener BUTTON_MOUSE_LISTENER =
            getMouseListener();

    /**
     * Ein MouseListener, der einen Rahmen anzeigt, wenn die Maus über dem
     * Button ist.
     */
    public final static MouseListener TOGGLE_BUTTON_MOUSE_LISTENER =
            getToggleMouseListener();

    /**
     * Gibt einen MouseListener für diese Art von Buttons zurück.
     *
     * @return MouseListener, der einen Rahmen anzeigt, wenn die Maus über dem
     *         Button ist.
     */
    private static MouseListener getMouseListener() {
        return new MouseAdapter() {
            @Override
            public void mouseEntered(MouseEvent e) {
                Component component = e.getComponent();
                if (component instanceof AbstractButton button) {
                    button.setBorderPainted(true);
                }
            }

            @Override
            public void mouseExited(MouseEvent e) {
                Component component = e.getComponent();
                if (component instanceof AbstractButton button) {
                    button.setBorderPainted(false);
                }
            }
        };
    }

    /**
     * Gibt einen MouseListener für die Textbuttons zurück.
     *
     * @return MouseListener, der einen Rahmen anzeigt, wenn die Maus über dem
     *         Button ist und außerdem die Textfarbe ändert.
     */
    private static MouseListener getToggleMouseListener() {
        return  new MouseAdapter() {
            @Override
            public void mouseEntered(MouseEvent e) {
                Component component = e.getComponent();
                if (component instanceof AbstractButton button) {
                    button.setBorderPainted(true);
                    toggleColor(button);
                }
            }

            @Override
            public void mouseExited(MouseEvent e) {
                Component component = e.getComponent();
                if (component instanceof AbstractButton button) {
                    button.setBorderPainted(false);
                    toggleColor(button);
                }
            }

            private void toggleColor(AbstractButton button) {
                //Color fg = button.getForeground();
                //Color bg = button.getBackground();
                //button.setForeground(bg);
                //button.setBackground(fg);

                Color fg = button.getForeground();
                int red = fg.getRed();
                int green = fg.getGreen();
                int blue = fg.getBlue();
                Color anti = new Color(red, blue, green);
                button.setForeground(anti);
            }
        };
    }

    /** Erzeugt die entgegengesetzte Farbe. */
    public static Color antiColor(Color color) {
        int red = 255 - color.getRed();
        int green = 255 - color.getGreen();
        int blue = 255 - color.getBlue();
        return new Color(red, green, blue);
    }

    /**
     * Hinterlegt einen MouseLister, der bei Links- und Rechtsklick unterschiedliche Aktionen
     * ausführt.
     */
    public static void addLeftAndRightClickMouseListenerToButton(JButton button,
            Runnable leftClickAction, Runnable rightClickAction) {
        button.addMouseListener(new MouseAdapter() {
            private boolean pressed;

            @Override
            public void mousePressed(MouseEvent e) {
                button.getModel().setArmed(true);
                button.getModel().setPressed(true);
                pressed = true;
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                // if(isRightButtonPressed) {underlyingButton.getModel().setPressed(true));
                button.getModel().setArmed(false);
                button.getModel().setPressed(false);

                if (pressed) {
                    if (SwingUtilities.isRightMouseButton(e)) {
                        rightClickAction.run();
                    }
                    else {
                        leftClickAction.run();
                    }
                    pressed = false;
                }
            }

            @Override
            public void mouseExited(MouseEvent e) {
                pressed = false;
            }

            @Override
            public void mouseEntered(MouseEvent e) {
                pressed = true;
            }
        });
    }

}
