package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt ein Element mit einer Überschrift oben und einem frei bearbeitbaren Text
 * darunter dar.
 *
 * @version 1.01     2025-11-10
 * @author Christian Dühl
 */

public class TextFieldWithTitle {

    /** Überschrift. */
    private final JLabel titleLabel;

    /** Textfeld und Schalter zur Änderung. */
    private final JTextField textField;

    /** Panel mit Überschrift, Pfad und Änderungsbutton. */
    private final Component panel;

    /**
     * Konstruktor.
     *
     * @param title
     *            Überschrift für das Element.
     */
    public TextFieldWithTitle(String title) {
        titleLabel = new JLabel(title);
        textField = createTextField();
        panel = createPanel();
    }

    protected JTextField createTextField() {
        return new JTextField();
    }

    /** Erzeugt den Panel mit Überschrift, Option und Änderungsbutton. */
    private Component createPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(titleLabel, getTitleBorderLayoutOrientation());
        panel.add(textField, getTextFieldBorderLayoutOrientation());

        return panel;
    }

    protected String getTitleBorderLayoutOrientation() {
        return BorderLayout.NORTH;
    }

    protected String getTextFieldBorderLayoutOrientation() {
        return BorderLayout.CENTER;
    }

    /** Getter für das Panel mit Überschrift, Pfad und Änderungsbutton. */
    public Component getPanel() {
        return panel;
    }

    /** Färbt alle Komponenten mit dem übergebenen Colorizer ein. */
    public void colorize(Colorizer colorizer) {
        colorizer.setColors(panel);
        colorizer.setColors(titleLabel);
        colorizer.setColors(textField);
    }

    /** Getter für den (getrimmten) Text des Textfeldes. */
    public String getText() {
        return textField.getText().trim();
    }

    /** Setzt den Text des Textfeldes. */
    public void setText(String text) {
        textField.setText(text);
    }

    /** Setter für die gewünschte Größe des Textfeldes. */
    public void setPreferredSize(Dimension preferredSize) {
        textField.setPreferredSize(preferredSize);
    }

    /** Fügt einen KeyListener zum Textfeld hinzu. */
    public void addKeyListener(KeyListener keyListener) {
        textField.addKeyListener(keyListener);
    }

    /** Fügt einen Aktion hinzu, die beim Drücken einer Taste im Textfeld ausgelöst wird. */
    public void addKeyReleasedAction(Runnable runnable) {
        addKeyListener(new KeyListener() {
            @Override
            public void keyTyped(KeyEvent e) {
            }
            @Override
            public void keyReleased(KeyEvent e) {
                runnable.run();
            }
            @Override
            public void keyPressed(KeyEvent e) {
            }
        });
    }

    /** Fügt einen ActionListener zum Textfeld hinzu. */
    public void addActionListener(ActionListener actionListener) {
        textField.addActionListener(actionListener);
    }

    /** Setzt das Element enabled oder nicht enabled. */
    public void setEnabled(boolean enabled) {
        titleLabel.setEnabled(enabled);
        textField.setEnabled(enabled);
    }

    /** Setzt das Element editable oder nicht editable. */
    public void setEditable(boolean editable) {
        titleLabel.setEnabled(true);
        textField.setEnabled(true);
        textField.setEditable(editable);
    }

    /** Setzt den Fokus in das Textfeld. */
    public void requestFocus() {
        textField.requestFocus();
    }

    /** Setzt den Fokus in das Textfeld etwas später und im EDT. */
    public void requestFocusLater() {
        SwingUtilities.invokeLater(() -> requestFocus());
    }

}
