package de.duehl.swing.ui.geometry;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.geom.Line2D;
import java.awt.geom.Point2D;

/**
 * Diese Klasse sucht dem Schnittpunkt zweier Linien.
 *
 * @version 1.01     2025-07-23
 * @author Christian Dühl
 */

public class Line2DIntersection {

    /** Die erste Linie. */
    private final Line2D line1;

    /** Die zweite Linie. */
    private final Line2D line2;

    /** Gibt an, ob ein Schnittpunkt gefunden wurde. */
    private boolean found;

    /** Der Schnittpunkt, falls einer gefunden wurde. */
    private Point2D intersection;

    /**
     * Konstruktor.
     *
     * @param line1
     *            Die erste Linie.
     * @param line2
     *            Die zweite Linie.
     */
    public Line2DIntersection(Line2D line1, Line2D line2) {
        this.line1 = line1;
        this.line2 = line2;
    }

    /** Sucht nach dem Schnittpunkt. */
    public void findIntersection() {
        double x1 = line1.getX1();
        double y1 = line1.getY1();
        double x2 = line1.getX2();
        double y2 = line1.getY2();
        double x3 = line2.getX1();
        double y3 = line2.getY1();
        double x4 = line2.getX2();
        double y4 = line2.getY2();

        double det = (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4);
        if (det == 0) {
            found = false;
            return; // Lines are parallel
        }

        double t = ((x1 - x3) * (y3 - y4) - (y1 - y3) * (x3 - x4)) / det;
        double u = -((x1 - x2) * (y1 - y3) - (y1 - y2) * (x1 - x3)) / det;

        if (t > 0 && t < 1 && u > 0 && u < 1) {
            double intersectionX = x1 + t * (x2 - x1);
            double intersectionY = y1 + t * (y2 - y1);
            found = true;
            intersection = new Point2D.Double(intersectionX, intersectionY);
        }

        found = false; // Lines don't intersect
    }

    /** Gibt an, ob ein Schnittpunkt gefunden wurde. */
    public boolean isFound() {
        return found;
    }

    /** Getter für den Schnittpunkt, falls einer gefunden wurde. */
    public Point2D getIntersection() {
        return intersection;
    }

}
