package de.duehl.swing.ui.handler.error;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Image;
import java.awt.Point;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import de.duehl.basics.logic.ErrorHandler;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.error.ErrorDialog;

/**
 * Diese Klasse stellt einen einfachen ErrorHandler dar.
 *
 * @version 1.01     2017-07-19
 * @author Christian Dühl
 */

public class SimpleErrorHandler implements ErrorHandler {

    /** Icon für das Programm. */
    private final Image programImage;

    /** Frame vor dem der Error angezeigt wird. */
    private final JFrame parentFrame;

    /** Konstruktor. */
    public SimpleErrorHandler() {
        this(null, null);
    }

    /**
     * Konstruktor.
     *
     * @param programImage
     *            Icon für das Programm.
     * @param parentFrame
     *            Frame vor dem der Error angezeigt wird.
     */
    public SimpleErrorHandler(Image programImage, JFrame parentFrame) {
        this.programImage = programImage;
        this.parentFrame = parentFrame;
    }

    /** Zeigt dem Benutzer eine Warnung an. */
    @Override
    public void warning(String message) {
        SwingUtilities.invokeLater(() -> warningInEDT(message));
    }

    /** Zeigt dem Benutzer eine Warnung an. */
    private void warningInEDT(String message) {
        /* Text umbrechen, damit er nicht zu lang wird: */
        String brokenMessage = Text.addLineBreaks(message, 80);
        JOptionPane.showMessageDialog(parentFrame, brokenMessage, "Warnung",
                JOptionPane.WARNING_MESSAGE);
    }

    /** Zeigt dem Benutzer einen Fehler an. */
    @Override
    public void error(String message) {
        error(message, null);
    }

    /** Zeigt dem Benutzer einen Fehler mit Exception an. */
    @Override
    public void error(String message, Exception exception) {
        ErrorDialog dialog = new ErrorDialog(message, exception, programImage, calculateLocation());
        SwingUtilities.invokeLater(() -> errorInEDT(dialog));
    }

    /** Zeigt dem Benutzer einen Fehler mit Exception an. */
    private void errorInEDT(ErrorDialog dialog) {
        dialog.setVisible(true);
    }

    /** Bestimmt die aktuellen Lokation des Rahmens des übergeordneten Programms. */
    private Point calculateLocation() {
        if (parentFrame == null) {
            return new Point(400, 150);
        }
        else {
            return parentFrame.getLocation();
        }
    }

}
