package de.duehl.swing.ui.menu.collection;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * Diese Klasse stellt die gemerkten Eigenschaften eines Menüeintrag (MyMenuItem oder
 * MyCheckBoxMenuItem) dar.
 *
 * @version 1.01     2020-07-08
 * @author Christian Dühl
 */

public class StoredMyMenuItem {

    private static final char NOT_STORED_MNEMONIC = (char) 0;
    private static final int NOT_STORED_KEY_CODE = Integer.MAX_VALUE;
    private static final int NOT_STORED_MODIFIERS = Integer.MAX_VALUE;
    private static final ActionListener NOT_STORED_ACTIONLISTENER = new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
        }
    };

    /** Der hinterlegte Text des Menüeintrags. */
    private String text;

    /** Das unterstriche Zeichen im Text des Menüeintrags. */
    private char mnemonic;

    /** Der numerische Wert einer Taste auf der Tastatur für das Tasten-Kürzel des Menüeintrags. */
    private int keyCode;

    /** Bitweise ODER-verknüpfte Modifikatoren für das Tasten-Kürzel des Menüeintrags. */
    private int modifiers;

    /** Aktion die ausgeführt wird, wenn der Menüeintrag angeklickt wird. */
    private ActionListener actionListener;

    /**
     * Konstruktor.
     *
     * @param text
     *            Dargestellter Text des Menüeintrags.
     */
    public StoredMyMenuItem(String text) {
        this.text = text;
        this.mnemonic = NOT_STORED_MNEMONIC;
        this.keyCode = NOT_STORED_KEY_CODE;
        this.modifiers = NOT_STORED_MODIFIERS;
        this.actionListener = NOT_STORED_ACTIONLISTENER;

        AutomaticMyMenuItemCollection.collect(this);
    }

    /**
     * Legt das Zeichen des Menüeintrags fest, das unterstrichen wird, auf den übergebenen
     * char-Wert.
     */
    public void mnemonic(char mnemonic) {
        this.mnemonic = mnemonic;
    }

    /**
     * Legt das Tasten-Kürzel des Menüeintrags fest.
     *
     * @param keyCode
     *            Der numerische Wert einer Taste auf der Tastatur für das Tasten-Kürzel des
     *            Menüeintrags.
     * @param modifiers
     *            Bitweise ODER-verknüpfte Modifikatoren für das Tasten-Kürzel des Menüeintrags.
     */
    public void accelerator(int keyCode, int modifiers) {
        this.keyCode = keyCode;
        this.modifiers = modifiers;

        AutomaticMyMenuItemCollection.checkAccelerator(this);
    }

    /** Legt Aktion fest, die ausgeführt wird, wenn der Menüeintrag angeklickt wird. */
    public void actionListener(ActionListener actionListener) {
        this.actionListener = actionListener;
    }

    /** Getter für den hinterlegten Text des Menüeintrags. */
    public String getText() {
        return text;
    }

    /** Gibt an, ob ein unterstrichenes Zeichen im Text des Menüeintrags vergeben wurde. */
    public boolean isMnemonicStored() {
        return mnemonic != NOT_STORED_MNEMONIC;
    }

    /** Getter für das unterstriche Zeichen im Text des Menüeintrags. */
    public char getMnemonic() {
        return mnemonic;
    }

    /** Gibt an, ob ein Tasten-Kürzel für den Menüeintrag vergeben wurde. */
    public boolean isKeyCodeAndModifiersStored() {
        return keyCode != NOT_STORED_KEY_CODE
                && modifiers != NOT_STORED_MODIFIERS;
    }

    /**
     * Getter für den numerischen Wert einer Taste auf der Tastatur für das Tasten-Kürzel des
     * Menüeintrags.
     */
    public int getKeyCode() {
        return keyCode;
    }

    /**
     * Getter für die bitweise ODER-verknüpfte Modifikatoren für das Tasten-Kürzel des
     * Menüeintrags.
     */
    public int getModifiers() {
        return modifiers;
    }

    /**
     * Gibt an, ob eine Aktion festgelegt wurde, die ausgeführt wird, wenn der Menüeintrag
     * angeklickt wird.
     */
    public boolean isActionListenerStored() {
        return !actionListener.equals(NOT_STORED_ACTIONLISTENER);
    }

    /** Getter für die Aktion die ausgeführt wird, wenn der Menüeintrag angeklickt wird. */
    public ActionListener getActionListener() {
        return actionListener;
    }

    @Override
    public String toString() {
        return "StoredMyMenuItem [text=" + text + ", mnemonic=" + mnemonic + ", keyCode=" + keyCode
                + ", modifiers=" + modifiers + "]";
    }

}
