package de.duehl.swing.ui.tabs.bars.growing;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import de.duehl.basics.debug.Assure;
import de.duehl.swing.ui.buttons.painted.PlusButton;

/**
 * Diese Klasse stellt eine JTabbedPane mit einem Button zum Hinzufügen von Reitern dar.
 * Diese wird von der Klasse GrowingTabBar anstelle von JTabbedPane (wie sie die Klasse TabBar
 * erzeugt) erzeugt.
 *
 * Quelle: https://stackoverflow.com/questions/1971141/java-add-tab-button-for-a-jtabbedpane
 * Dann angepasst.
 *
 * @version 1.01     2021-11-30
 * @author Christian Dühl
 */

public class GrowingTabbedPane extends JTabbedPane {

    private static final long serialVersionUID = 1L;

    /** Objekt das einen neuen Reiter anlegen kann. */
    private final TabCreator tabCreator;

    /**
     * Konstruktor.
     *
     * @param tabCreator
     *            Objekt das einen neuen Reiter anlegen kann.
     */
    public GrowingTabbedPane(TabCreator tabCreator) {
        this(tabCreator, TOP, WRAP_TAB_LAYOUT);
    }

    /**
     * Konstruktor.
     *
     * @param tabCreator
     *            Objekt das einen neuen Reiter anlegen kann.
     * @param tabPlacement
     *            Die Platzierung des Reiters relativ zum Inhalt.
     * @param tabLayoutPolicy
     *            Die Richtlinie zum Anordnen von Reitern, wenn nicht alle Reiter in eine Zeile
     *            passen.
     */
    public GrowingTabbedPane(TabCreator tabCreator, int tabPlacement, int tabLayoutPolicy) {
        super(tabPlacement, tabLayoutPolicy);
        Assure.notNull("Der Tab-Erzeuger", tabCreator);
        this.tabCreator = tabCreator;
        addPlusButton();
    }

    /** Fügt den kleinen Reiter mit dem Plus zum Hinzufügen weiterer Reiter hinzu. */
    private void addPlusButton() {
        addTabWithoutContentForPlusButton();

        JPanel panelWithAddButton = createOpaquePanelForAddButonPanel();

        setTabComponentAt(getTabCount() - 1, panelWithAddButton);
    }

    private void addTabWithoutContentForPlusButton() {
        super.addTab("test", null);
    }

    private JPanel createOpaquePanelForAddButonPanel() {
        JPanel panel = new JPanel(new BorderLayout());
        panel.setOpaque(false);

        panel.add(createAddTabButton(), BorderLayout.CENTER);

        return panel;
    }

    private JButton createAddTabButton() {
        JButton button = new PlusButton();
        button.setFocusPainted(false);
        button.setFocusable(false);
        button.addActionListener(e -> addTabButtonAction());

        return button;
    }

    private void addTabButtonAction() {
        String title = "new " + String.valueOf(getTabCount());
        createNewTab(title);
    }

    private void createNewTab(String title) {
        tabCreator.createTab(title);
    }

    /** Fügt einen neuen Tab vor das Plus ein. */
    @Override
    public void addTab(String title, Component component) {
        int tabsCount = getTabCount();
        int indexBeforeAddTabButtonTab = tabsCount - 1;

        insertTab(title, null, component, null, indexBeforeAddTabButtonTab);
        setSelectedIndex(indexBeforeAddTabButtonTab);
    }

}
