package de.duehl.swing.ui.start.filter.project.gateway;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.swing.ui.filter.method.combination.CombinationElementList;
import de.duehl.swing.ui.filter.project.gateway.FilterGateway;
import de.duehl.swing.ui.filter.project.gateway.FilterWorker;
import de.duehl.swing.ui.filter.project.gateway.data.DescriptionAndFilterType;
import de.duehl.swing.ui.start.filter.project.data.TestFilterData;

public class TestFilterGateway extends FilterGateway<TestFilterData, TestFilterType> {

    private static final List<DescriptionAndFilterType<TestFilterType>> METHODS =
        CollectionsHelper.buildListFrom(
            new DescriptionAndFilterType<>(
                    "Daten ohne Bearbeitungscode",
                    TestFilterType.NO_WORKING_CODE
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Bearbeitungscode 0",
                    TestFilterType.WORKING_CODE_ZERO
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Bearbeitungscode 1",
                    TestFilterType.WORKING_CODE_ONE
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Bearbeitungscode 2",
                    TestFilterType.WORKING_CODE_TWO
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Bearbeitungscode 3",
                    TestFilterType.WORKING_CODE_THREE
                    ),
            new DescriptionAndFilterType<>(
                    "Daten ohne WZ-Codes",
                    TestFilterType.NO_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit einem WZ-Code",
                    TestFilterType.ONE_CODE
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit zwei WZ-Codes",
                    TestFilterType.TWO_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit drei WZ-Codes",
                    TestFilterType.THREE_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit vier WZ-Codes",
                    TestFilterType.FOUR_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit fünf WZ-Codes",
                    TestFilterType.FIVE_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit kurzen Texten",
                    TestFilterType.SHORT_TEXT
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit einem unerschlossenen Satz",
                    TestFilterType.ONE_UNKNOWN
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit einem unerschlossenen, kurzen Satz",
                    TestFilterType.ONE_SHORT_UNKNOWN
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit wenigen Treffern",
                    TestFilterType.FEW_HITS
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit kritischen WZ-Codes",
                    TestFilterType.CRITICAL_CODES
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Kandidaten auf Hinweise für den Eintragstyp",
                    TestFilterType.ENTRY_TYPE_CANDIDATES
                    )
            );

    private static final List<DescriptionAndFilterType<TestFilterType>> PARAMISED_METHODS =
        CollectionsHelper.buildListFrom(
            new DescriptionAndFilterType<>(
                    "Daten mit bestimmtem Suchbegriff in Text oder Firma",
                    TestFilterType.WITH_TEXT
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit bestimmtem vergebenem WZ-Code",
                    TestFilterType.WITH_CODE
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Doknummern, die mit einer bestimmten Nummer beginnen",
                    TestFilterType.DOKNR_STARTS_WITH
                    ),
            new DescriptionAndFilterType<>(
                    "Daten mit Doknummern, die auf einer bestimmten Nummer enden",
                    TestFilterType.DOKNR_ENDS_WITH
                    )
            );

    private final FilterWorker<TestFilterData> filterWorker;

    public TestFilterGateway(FilterWorker<TestFilterData> filterWorker) {
        super(METHODS,  PARAMISED_METHODS);

        this.filterWorker = filterWorker;
    }

    /** Gibt an, ob im übergeordneten Projekt ein kombinierter Filter angezeigt wird. */
    @Override
    public boolean isFilterCombinedMethods() {
        /*
         * Im WZ-Tool: logic.getFilterTypeOfInputSource() == TestFilterType.COMBINED_METHODS
         */
        return true;
    }

    /** Setzt im Übergeordneten Projekt den realen Filter. */
    @Override
    public void setRealFilter(CombinationElementList<TestFilterData> elements) {
        /*
         * Im WZ-Tool
            /* Filtermethode erzeugen und in der source setzen: * /
            logic.setStatus("");
            logic.setCombinedMethod(elements);

         */
        filterWorker.workWithFilter(elements);
    }

}
