package de.duehl.vocabulary.japanese.common.persistence;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.text.TextViewer;

/**
 * Dieses Klasse hilft beim Erstellen von Persistenz-Strings aus Umsetzungen der eigenen
 * Beschreibung von grammtikalischen Verb- oder Adjektivformen auf die vom Benutzer eingegebenen
 * und andersherum, aus einem solchen  Persistenz-String wieder ein Verzeichnis der Umsetzungen zu
 * erstellen
 *
 * @version 1.01     2025-11-29
 * @author Christian Dühl
 */

public class GrammarTableHeaderPersistenceHelper {

    /** Der Trenner zwischen Paaren von Zuordnungen meiner Formulierung auf die des Benutzers. */
    public static final String GRAMMAR_TABLE_HEADER_OUTER_DIVIDER = "##;##";

    /** Der Trenner zwischen meiner Formulierung und der des Benutzers. */
    public static final String GRAMMAR_TABLE_HEADER_INNER_DIVIDER = "#;#";

    /**
     * Erstellt die Umsetzungen der eigenen Beschreibung von grammtikalischen Verb- oder
     * Adjektivformen auf die vom Benutzer eingegebenen.
     *
     * @param grammarTableHeadersMap
     *            Das Verzeichnis der meiner Formulierungen aus der Klasse VerbSearchWords oder
     *            AdjectiveSearchWords auf die Formulierung des Benutzers.
     * @return Ein String um dieses Verzeichnis persistent abspeichern zu können.
     */
    public static String createGrammarTableHeaderPesistenceString(
            Map<String, String> grammarTableHeadersMap) {
        StringBuilder builder = new StringBuilder();

        for (String myForm : grammarTableHeadersMap.keySet()) {
            String userForm = grammarTableHeadersMap.get(myForm);
            if (!builder.isEmpty()) {
                builder.append(GRAMMAR_TABLE_HEADER_OUTER_DIVIDER);
            }
            builder.append(myForm)
                    .append(GRAMMAR_TABLE_HEADER_INNER_DIVIDER)
                    .append(userForm);
        }

        return builder.toString();
    }

    /**
     * Erstellt aus einem Persistenz-String der Umsetzungen der eigenen Beschreibung von
     * grammtikalischen Verb- oder Adjektivformen auf die vom Benutzer eingegebenen wieder ein
     * echtes Verzeichnis.
     *
     * @param grammarTableHeadersPersistenceString
     *            Ein String in dem ein Verzeichnis persistent abgespeichert wurde.
     * @return Das erzeugte Verzeichnis.
     */
    public static Map<String, String> createGrammarTableHeaderMapFormPersistenceString(
            String grammarTableHeadersPersistenceString) {
        List<String> parts = Text.splitBy(grammarTableHeadersPersistenceString,
                GRAMMAR_TABLE_HEADER_OUTER_DIVIDER);

        Map<String, String> grammarTableHeadersMap = new HashMap<>();
        for (String part : parts) {
            List<String> innerParts = Text.splitBy(part, GRAMMAR_TABLE_HEADER_INNER_DIVIDER);
            if (innerParts.size() == 2) {
                String myForm = innerParts.get(0);
                String userForm = innerParts.get(1);
                grammarTableHeadersMap.put(myForm, userForm);
            }
            else {
                System.err.println("Ein Persistence-String der Überschrften von Grammatik-Tabellen "
                        + "ließ sich nicht am Trenner aufteilen:\n"
                        + "\t" + "grammarTableHeadersPersistenceString = "
                                + grammarTableHeadersPersistenceString + "\n"
                        + "\t" + "GRAMMAR_TABLE_HEADER_INNER_DIVIDER   = "
                                + GRAMMAR_TABLE_HEADER_INNER_DIVIDER + "\n"
                        + "\t" + "part                                 = " + part + "\n"
                        + "\t" + "innerParts                           = " + innerParts + "\n"
                        );
                grammarTableHeadersMap.clear();
                break;
            }
        }

        return grammarTableHeadersMap;
    }

    /**
     * Hier prüfe ich, ob sich die in den Paaren abgespeicherten String von mir in der Zwischenzeit
     * vielleicht geändert haben (weil ich diese in der entsprechenden Klasse verändert habe). In
     * diesem Fall gebe ich eine Warnung aus, in der man die verloren gegangenen Zuordnungen
     * wenigstens sehen kann
     *
     * Dazu zeige ich in im TextViewer an, damit man den Text rauskopieren kann.
     *
     * @param grammarTableHeadersMap
     *            Das Verzeichnis der grammatikalischen Texte zu ihren vom Benutzer vorgesehenen
     *            Formulierungen für die Köpfe der Grammatik-Tabellen.
     * @param grammarForm
     *            Die grammatikalische Form.
     * @param myForms
     *            Die Liste der von mir aktuell verwendeten Formulierungen.
     */
    public static void checkForChangedMyForms(Map<String, String> grammarTableHeadersMap,
            String grammarForm, List<String> myForms) {
        List<String> notKnownMyForms = new ArrayList<>();


        for (String myForm : CollectionsHelper.getSortedMapStringIndices(grammarTableHeadersMap)) {
            if (!myForms.contains(myForm)) {
                notKnownMyForms.add(myForm);
            }
        }

        if (!notKnownMyForms.isEmpty()) {
            StringBuilder builder = new StringBuilder();
            builder.append(""
                    + "Änderung bei " + grammarForm + ":\n"
                    + "\n"
                    + "Die folgenden von mir vorgegebenen Formen haben sich geändert.\n"
                    + "Daher müssen die von Ihnen eingetragenen Titel für die Spalten\n"
                    + "der erzeugten Grammatik-Tabellen wieder angepasst werden.\n"
                    + "\n"
                    );

            for (String myForm : notKnownMyForms) {
                String userForm = grammarTableHeadersMap.get(myForm);
                builder.append("    Alter Text von mir: " + myForm + "\n");
                builder.append("    Ihr Spalten-Titel : " + userForm + "\n\n");
            }

            builder.append(""
                    + "Aus diesem Text kann Ihr Spalten-Titel herauskopiert werden,\n"
                    + "um ihn der veränderten grammatikalischen Form von mir wieder\n"
                    + "zuzuweisen.\n"
                    + "Die Titel für grammatikalische Tabellen kann man mit F5 in den\n"
                    + "Optionen vergeben."
                    );

            String contents = builder.toString();

            TextViewer dialog = new TextViewer("Veränderte grammatikalische Begriffe bei " +
                    grammarForm);
            dialog.setText(contents);
            dialog.useMonoscpacedText();
            dialog.biggerFont(5);
            dialog.setVisible(true);
        }
    }

}
