package de.duehl.vocabulary.japanese.common.ui.dialogs;

import static de.duehl.swing.ui.components.selections.tools.SelectionsHelper.NOT_EDITABLE_FIELD_COLOR;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import de.duehl.basics.text.html.HtmlTool;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.layout.VerticalLayout;

/**
 * Diese Klasse zeigt eine Meldung an, die die Url zur Webseite des Vokabeltrainers in
 * herauskopierbarer Form enthält.
 *
 * @version 1.01     2024-12-07
 * @author Christian Dühl
 */

public class DialogWithCopyableUrl extends ModalDialogBase {

    private static final int DIALOG_WIDTH = 750;
    private static final int BIGGER_TEXT_FACTOR = 5;


    /** Die im oberen Bereich anzuzeigende Nachricht. */
    private final String message;

    /** Der Text vor der Url. */
    private final String beforeUrl;

    /** Der Text nach der Url. */
    private final String afterUrl;

    /** Das Textfeld mit der Url. */
    private final JTextField textField;

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel des Dialogs.
     * @param message
     *            Die im oberen Bereich anzuzeigende Nachricht.
     * @param beforeUrl
     *            Der Text vor der Url.
     * @param url
     *            Die Url.
     * @param afterUrl
     *            Der Text nach der Url.
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     */
    public DialogWithCopyableUrl(String title, String message, String beforeUrl, String url,
            String afterUrl, Point parentLocation, Image programImage) {
        super(parentLocation, programImage, title);
        this.message = message;
        this.beforeUrl = beforeUrl;
        this.afterUrl = afterUrl;

        addEscapeBehaviour();
        addClosingWindowListener(() -> closeDialog());

        textField = new JTextField(url);

        init();
        fillDialog();
    }

    private void init() {
        textField.setEditable(false);
        textField.setFocusable(true);
        textField.setBackground(NOT_EDITABLE_FIELD_COLOR);
        GuiTools.setMonospacedFont(textField);
        GuiTools.biggerFont(textField, BIGGER_TEXT_FACTOR);
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createMainPart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);
    }

    private Component createMainPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(createDummyPanel(), BorderLayout.NORTH);
        panel.add(createMessagePart(), BorderLayout.CENTER);
        panel.add(createUrlPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createDummyPanel() {
        JPanel panel = new JPanel();
        panel.setPreferredSize(new Dimension(DIALOG_WIDTH, 0));
        return panel;
    }

    private Component createMessagePart() {
        return createLabel(message + "\n\n");
    }

    private Component createUrlPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));

        panel.add(createLabelBeforeUrl());
        panel.add(textField);
        panel.add(createLabelAfterUrl());

        return panel;
    }

    private Component createLabelBeforeUrl() {
        return createLabel(beforeUrl);
    }

    private Component createLabelAfterUrl() {
        return createLabel(afterUrl);
    }

    private static Component createLabel(String message) {
        JLabel label = new JLabel(HtmlTool.htmlify(message));
        GuiTools.biggerFont(label, BIGGER_TEXT_FACTOR);
        return label;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createOKButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createOKButton() {
        JButton button = new JButton("Schließen");
        button.addActionListener(e -> closeDialog());
        return button;
    }

}
