package de.duehl.vocabulary.japanese.common.website.update.vocables.parser;

import de.duehl.basics.datetime.date.DateHelper;
import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.text.NumberString;

/**
 * Diese Klasse parst einen String mit einer Version der Vokabeln.
 *
 * @version 1.01     2024-10-05
 * @author Christian Dühl
 */

public class VocableVersionParser {

    /** Der String mit der Version. */
    private final String vocablesVersion;

    /** Gibt an, ob der String mit der Version erfolgreich geparst werden konnte. */
    private boolean success;

    /**
     * Der Grund, warum der String mit der Version nicht geparst werden konnte (nur gefüllt wenn
     * success false ist).
     */
    private String failReason;

    /** Der Datumteil der Version als String. */
    private String dateAsString;

    /** Der Datumteil der Version als Datum. */
    private ImmutualDate date;

    /** Die Nummer aus der Version. */
    private int number;

    /**
     * Konstruktor.
     *
     * @param vocablesVersion
     *            Der String mit der Version.
     */
    public VocableVersionParser(String vocablesVersion) {
        this.vocablesVersion = vocablesVersion;
    }

    /** Führt das Parsen durch. */
    public void parse() {
        init();

        int underscoreIndex = vocablesVersion.indexOf("_");
        if (underscoreIndex == -1) {
            fail("Da die Version '" + vocablesVersion + "' keinen "
                    + "Unterstrich enthält, wird sie nicht berücksichtigt.");
        }
        else {
            String front = vocablesVersion.substring(0, underscoreIndex);
            String rear = vocablesVersion.substring(underscoreIndex + 1);
            if (front.length() != 8) {
                fail("Da der vordere Teil (" + front + ") der Version '"
                        + vocablesVersion + "' nicht acht Zeichen lang ist, wird sie nicht "
                        + "berücksichtigt.");
            }
            else if (!NumberString.isDigitSequence(front)) {
                fail("Da der vordere Teil (" + front + ") der Version '"
                        + vocablesVersion + "' nicht nur aus Ziffern besteht, wird sie "
                        + "nicht berücksichtigt.");
            }
            else if (!NumberString.isDigitSequence(rear)) {
                fail("Da der hintere Teil (" + rear + ") der Version '"
                        + vocablesVersion + "' nicht nur aus Ziffern besteht, wird sie "
                        + "nicht berücksichtigt.");
            }
            else {
                success = true;
                dateAsString = front;
                number = NumberString.parseInt(rear);
                date = DateHelper.getDateFromYYYYMMDD(dateAsString);
            }
        }
    }

    private void init() {
        success = true;
        failReason = "";
        dateAsString = "";
        number = -1;
    }

    private void fail(String failReason) {
        success = false;
        this.failReason = failReason;
    }

    /** Gibt an, ob der String mit der Version erfolgreich geparst werden konnte. */
    public boolean isSuccess() {
        return success;
    }

    /**
     * Getter für den Grund, warum der String mit der Version nicht geparst werden konnte (nur
     * gefüllt wenn isSuccess() false ist).
     */
    public String getFailReason() {
        return failReason;
    }

    /** Getter für den Datumteil der Version als String. */
    public String getDateAsString() {
        return dateAsString;
    }

    /** Getter für den Datumteil der Version als Datum. */
    public ImmutualDate getDate() {
        return date;
    }

    /** Getter für die Nummer aus der Version. */
    public int getNumber() {
        return number;
    }

}
