package de.duehl.vocabulary.japanese.grammar;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;

/**
 * Diese Klasse sucht zu einer Vokabel, welche ein Adjektiv ist, die zugehörige Grundform (positive
 * Gegenwart).
 *
 * @version 1.01     2025-06-11
 * @author Christian Dühl
 */

public class AdjectivePositivePresenceFinder {

    public static final Vocable NO_POSITIVE_PRESENCE_FOUND = new Vocable();

    /**
     * Sucht zur übergebenen Vokabel die Grundform.
     *
     * Hier wird davon ausgegangen, dass diese Vokabel ein Adjektiv ist.
     *
     * Außerdem werden nur Vokabeln mit Kanji gefunden, denn ansonsten kann man sie nicht mit
     * der Grundform vergleichen.
     *
     * @param vocable
     *            Die Vokabel zu der die Grundform (positive Gegenwart) gesucht wird.
     * @param vocabularies
     *            Alle bekannten Vokabularien.
     * @return Die gefundene Grundform (positive Gegenwart) oder NO_POSITIVE_PRESENCE_FOUND, wenn
     *         diese nicht gefunden wurde.
     */
    public static Vocable determinePositivePresence(Vocable vocable, List<Vocabulary> vocabularies) {
        List<String> quotedTexts = Text.findTextsBetweenJapaneseQuotes(vocable.getComment());
        if (quotedTexts.isEmpty()) {
            return NO_POSITIVE_PRESENCE_FOUND;
        }

        String kanji = vocable.getKanji();
        if (kanji.isEmpty()) {
            return NO_POSITIVE_PRESENCE_FOUND;
        }

        /*
         * TODO:
         *
         * Hier macht man beim Vergleich über das erste Kanji einen Fehler, wenn verschiedene
         * Adjektive mit dem gleichen Kanji anfangen. Eigentlich müsste man vermutlich alle Kanji
         * am Anfang vergleichen?
         *
         * Da muss ich mir nochmal Beispiele anschauen, neulich war mir so eine Liste bei der
         * Verwendung des Programms aufgefallen.
         *
         * Obwohl eigentlich sind das doch nur die Texte in der Bemerkung ... Hmmm.
         */
        String firstKanji = kanji.substring(0, 1);
        List<String> potentialPositivePresenceInKanjiTexts = new ArrayList<>();
        for (String quotedText : quotedTexts) {
            if (quotedText.startsWith(firstKanji)) {
                potentialPositivePresenceInKanjiTexts.add(quotedText);
            }
        }

        boolean iAdjective = vocable.isIAdjective();
        boolean naAdjective = vocable.isNaAdjective();

        if (!iAdjective && !naAdjective) {
            return NO_POSITIVE_PRESENCE_FOUND;
        }

        List<Vocable> otherAdjectives = createListOfOtherAdjectives(vocable, vocabularies);
        for (Vocable otherAdjective : otherAdjectives) {
            if (otherAdjective.isAdjectivInPositivePresence()) {
                if (iAdjective && otherAdjective.isIAdjective()
                        || naAdjective && otherAdjective.isNaAdjective()) {
                    String anOtherKanji = otherAdjective.getKanji();
                    if (potentialPositivePresenceInKanjiTexts.contains(anOtherKanji)) {
                        return otherAdjective;
                    }
                }
            }
        }

        return NO_POSITIVE_PRESENCE_FOUND;
    }

    private static List<Vocable> createListOfOtherAdjectives(Vocable vocable,
            List<Vocabulary> vocabularies) {
        List<Vocable> otherAdjectives = new ArrayList<>();

        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable anOtherVocable : vocabulary.getVocables()) {
                if (!anOtherVocable.equals(vocable) // eigentlich überflüssig, da das nur
                                                    // aufgerufen wird, wenn vocable KEINE
                                                    // Grundform ist.
                    && anOtherVocable.isAdjective()) {
                    otherAdjectives.add(anOtherVocable);
                }
            }
        }

        return otherAdjectives;
    }

}
