package de.duehl.vocabulary.japanese.startup.logic.steps;

import java.util.List;

import de.duehl.basics.datetime.time.watch.StopWatch;
import de.duehl.basics.text.NumberString;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.Vocabulary;
import de.duehl.vocabulary.japanese.io.VocabularyDirectoryReader;
import de.duehl.vocabulary.japanese.startup.ui.data.SplashScreenable;

/**
 * Diese Klasse steht für den Schritt zum Einlesen der Vokabularien beim Startup des
 * Vokabeltrainers.
 *
 * @version 1.01     2025-11-24
 * @author Christian Dühl
 */

public class Step01ReadVocabularies extends StartupStep {

    /** Die Liste mit den bekannten Vokabularien. */
    private List<Vocabulary> vocabularies;

    /**
     * Die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    private String vocabularyLoadMessage;

    /** Die Anzahl aller Vokabeln. */
    private int numberOfVocables;

    /** Die Anzahl der Phrasen. */
    private int numberOfPhrases;

    /**
     * Konstruktor.
     *
     * @param step
     *            Der Schritt der durchgeführt wird.
     * @param options
     *            Die Programmoptionen.
     * @param splashScreen
     *            Die grafische Oberfläche beim Start in der die Meldungen angezeigt werden.
     * @param watch
     *            Misst die Laufzeit des gesamten Startups.
     */
    public Step01ReadVocabularies(String step, Options options, SplashScreenable splashScreen,
            StopWatch watch) {
        super(step, options, splashScreen, watch);
    }

    /** Führt den eigentlichen Inhalt des Schritts aus. */
    @Override
    protected void runInternalStep() {
        readVocabularies();
        createAndAppendLoadUpMessage();
        storeVocabularyDescriptionIntoVocable();
    }

    private void readVocabularies() {
        appendMessage("Lese Vokabularien ein ...");
        String directory = options.getVocabulariesPath();
        VocabularyDirectoryReader reader = new VocabularyDirectoryReader(directory);
        reader.read();
        vocabularies = reader.getVocabularies();
    }

    private void createAndAppendLoadUpMessage() {
        numberOfVocables = 0;
        numberOfPhrases = 0;
        for (Vocabulary vocabulary : vocabularies) {
            for (Vocable vocable : vocabulary.getVocables()) {
                ++numberOfVocables;
                if (vocable.getPartsOfSpeech().contains("Phrase")) {
                    ++numberOfPhrases;
                }
            }
        }
        vocabularyLoadMessage = "Es wurden " + NumberString.taupu(vocabularies.size())
                + " Vokabulare mit zusammen " + NumberString.taupu(numberOfVocables)
                + " Vokabeln (davon " + NumberString.taupu(numberOfPhrases)
                + " Phrasen) eingelesen.";
        appendMessage(vocabularyLoadMessage);
    }

    private void storeVocabularyDescriptionIntoVocable() {
        for (Vocabulary vocabulary : vocabularies) {
            String description = vocabulary.getDescription();
            for (Vocable vocable : vocabulary.getVocables()) {
                vocable.setVocabularyDescription(description);
            }
        }
    }

    /** Getter für die Liste mit den bekannten Vokabularien. */
    public List<Vocabulary> getVocabularies() {
        return vocabularies;
    }

    /**
     * Getter für die Nachricht mit der Anzahl Vokabeln und Vokabularien, die nach dem Laden in der
     * Statuszeile der Oberfläche angezeigt werden soll.
     */
    public String getVocabularyLoadMessage() {
        return vocabularyLoadMessage;
    }

    /** Getter für die Anzahl aller Vokabeln. */
    public int getNumberOfVocables() {
        return numberOfVocables;
    }

    /** Getter für die Anzahl der Phrasen. */
    public int getNumberOfPhrases() {
        return numberOfPhrases;
    }

}
