package de.duehl.vocabulary.japanese.ui.dialog.grammar.options;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.additional.LocationGetter;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.grammar.headers.GrammarTableHeaderEditor;

import static de.duehl.vocabulary.japanese.ui.dialog.grammar.GrammarTableCreatorDialog.*;

import java.awt.BorderLayout;
import java.awt.Component;

/**
 * Diese Klasse erzeugt einen Panel zur Einstellung der Optionen zur Erstellung von
 * grammatikalischen Tabellen von Verben oder Adjektiven.
 *
 * Folgende Optionen der grammatikalischen Tabellen werden nicht hier im Panel bearbeitet:
 *     grammarTableLastUsedOwnHeadersDirectory
 *
 * @version 1.01     2025-11-29
 * @author Christian Dühl
 */

public class GrammarTableOptionsPanel {

    /** Die Optionen des Programms. */
    private final Options options;

    /**
     * Das Objekt, das die aktuelle Position der Oberfläche des nutzenden Dialogs zurück geben
     * kann.
     */
    private final LocationGetter locationGetter;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private final FumikoUiObjects uiObjects;

    /**
     * Der Panel auf dem die Optionen zur Erstellung von grammatikalischen Tabellen von Verben oder
     * Adjektiven angezeigt werden.
     */
    private JPanel panel;

    /* Die Gui-Elemente für die boolean-Optionen: */
    private final JCheckBox showKanjiCheckBox;
    private final JCheckBox showKanaCheckBox;
    private final JCheckBox showRomajiCheckBox;
    private final JCheckBox searchForBaseFormsCheckBox;
    private final JCheckBox leaveNotFoundFormsBlankCheckBox;
    private final JCheckBox useUserHeadersCheckBox;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Optionen des Programms.
     * @param locationGetter
     *            Das Objekt, das die aktuelle Position der Oberfläche des nutzenden Dialogs zurück
     *            geben kann.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     */
    public GrammarTableOptionsPanel(Options options, LocationGetter locationGetter,
            FumikoUiObjects uiObjects) {
        this.options = options;
        this.locationGetter = locationGetter;
        this.uiObjects = uiObjects;

        panel = new JPanel();

        showKanjiCheckBox = new JCheckBox("Kanji anzeigen");
        showKanaCheckBox = new JCheckBox("Kana anzeigen");
        showRomajiCheckBox = new JCheckBox("Romaji anzeigen");
        searchForBaseFormsCheckBox = new JCheckBox("Nach Wörterbuchformen / Grundformen suchen");
        leaveNotFoundFormsBlankCheckBox = new JCheckBox(
                "Nicht gefundene Formen bleiben in der Tabelle leer");
        useUserHeadersCheckBox = new JCheckBox(
                "Die vom Benutzer eingegebenen Überschriften verwenden.");

        init();
        fillPanel();
    }

    private void init() {
        initPanel();
        initOptionsCheckBoxesFromOptions();
    }

    private void initPanel() {
        panel.setLayout(new BorderLayout());
    }

    private void initOptionsCheckBoxesFromOptions() {
        showKanjiCheckBox.setSelected(options.isGrammarTableShowKanji());
        showKanaCheckBox.setSelected(options.isGrammarTableShowKana());
        showRomajiCheckBox.setSelected(options.isGrammarTableShowRomaji());
        searchForBaseFormsCheckBox.setSelected(options.isGrammarTableSearchForBaseForms());
        leaveNotFoundFormsBlankCheckBox.setSelected(options.isGrammarTableLeaveNotFoundFormsBlank());
        useUserHeadersCheckBox.setSelected(options.isGrammarTableUseUserHeaders());
    }

    private void fillPanel() {
        panel.add(createOptionsSelectionPart(), BorderLayout.CENTER);
        panel.add(createChangeHeadersPart(), BorderLayout.SOUTH);
    }

    private Component createOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(VERTICAL_SPACE_IN_PARTS, VerticalLayout.BOTH));
        GuiTools.createTitle("Optionen zur Tabellenerstellung", panel);

        panel.add(showKanjiCheckBox);
        panel.add(showKanaCheckBox);
        panel.add(showRomajiCheckBox);
        panel.add(searchForBaseFormsCheckBox);
        panel.add(leaveNotFoundFormsBlankCheckBox);
        panel.add(useUserHeadersCheckBox);

        return panel;
    }

    private Component createChangeHeadersPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createEditGrammarTableHeadersButton(), BorderLayout.CENTER);

        return panel;
    }

    private Component createEditGrammarTableHeadersButton() {
        JButton button = new JButton("Überschriften der Grammatik-Tabellen bearbeiten");
        button.addActionListener(e -> editGrammarTableHeaders());
        return button;
    }

    private void editGrammarTableHeaders() {
        GrammarTableHeaderEditor editor = new GrammarTableHeaderEditor(options,
                locationGetter.getLocation(), uiObjects);
        editor.setVisible(true);
    }

    /**
     * Getter für den Panel auf dem die Optionen zur Erstellung von grammatikalischen Tabellen von
     * Verben oder Adjektiven angezeigt werden.
     */
    public JPanel getPanel() {
        return panel;
    }

    /**
     * Wird beim Beenden des Dialogs aufgerufen, um die Werte zurück in die Optionen zu schreiben.
     */
    public void storeIntoOptions() {
        boolean grammarTableShowKanji = showKanjiCheckBox.isSelected();
        boolean grammarTableShowKana = showKanaCheckBox.isSelected();
        boolean grammarTableShowRomaji = showRomajiCheckBox.isSelected();
        boolean grammarTableSearchForBaseForms = searchForBaseFormsCheckBox.isSelected();
        boolean grammarTableLeaveNotFoundFormsBlank = leaveNotFoundFormsBlankCheckBox.isSelected();
        boolean grammarTableUseUserHeaders = useUserHeadersCheckBox.isSelected();

        options.setGrammarTableShowKanji(grammarTableShowKanji);
        options.setGrammarTableShowKana(grammarTableShowKana);
        options.setGrammarTableShowRomaji(grammarTableShowRomaji);
        options.setGrammarTableSearchForBaseForms(grammarTableSearchForBaseForms);
        options.setGrammarTableLeaveNotFoundFormsBlank(grammarTableLeaveNotFoundFormsBlank);
        options.setGrammarTableUseUserHeaders(grammarTableUseUserHeaders);
    }

}
